#include <stdio.h>
#include <errno.h>
#include <dirent.h>

#include <iostream>
#include <string>
#include <vector>
#include <map>

#include "validate_types.hh"
#include "vsh/common/vsh_common.hh"
#include "vsh/common/vsh_config.hh"
#include "vsh/common/vsh_str_proc.hh"

using namespace std;

int match_node(string &in_cmd, string &node, string &out_cmd, VecColl &cmd, bool debug);
int validate_command(VecColl &in_cmd, string &usr_str, VecColl &out_cmd, bool debug);
int validate_type(string &in_cmd, string &node, bool debug);
int write_conf(VecColl &cmd, bool debug);
     
/**
 *
 *
 **/
void
usage()
{
  printf("set -c [vyatta command]\n");
}


/**
 *
 *
 **/
int 
main(int argc, char* const argv[])
{
  bool debug = false;
  string cmd;

  int ch;
  while ((ch = getopt(argc, argv, "c:d")) != -1) {
    switch (ch) {
    case 'd':
      debug = true;
      break;
    case 'c':
      cmd = optarg;
      break;
    default:
      usage();
      return -1;
    }
  }

  if (cmd.empty()) {
    usage();
    return -1;
  }

  //strip off first arg, then mash up directory 
  if (debug) {
    cout << "starting vshell set: " << cmd << endl;
  }

  //strip the command, delimited by white space
  StrProc str_proc(cmd, " ");
  VecColl in_cmd = str_proc.get();
  if (!in_cmd.empty()) {
    in_cmd.erase(in_cmd.begin());
  }

  //on each directory match check type
  string usr_str;
  VecColl out_cmd;
  if (validate_command(in_cmd, usr_str, out_cmd, debug) == 0) {
    write_conf(out_cmd, debug);
  }

  //output string even if no error
  if (usr_str.empty() == false) {
    cout << usr_str << endl;
  }
  exit(0);
}


/**
 *
 *
 **/
int 
validate_command(VecColl &in_cmd, string &usr_str, VecColl &out_cmd, bool debug)
{
  if (debug) {
    cout << "validate_command: " << endl;
  }

  string dir_cmd;
  VecIter iter = in_cmd.begin();
  while (iter != in_cmd.end()) {
    //on each element validate against directory structure
    if (match_node(dir_cmd, *iter, dir_cmd, out_cmd, debug) != 0) {
      usr_str = "failed";
      return -1; //failed
    }
    ++iter;
  }

  //ok decide what to do under the following conditions:
  /*
    1) if this matches, then proceed to next token
    2) if this doesn't match check directory for type/value rules
    3) if type/values rules still don't match then exit with error
    4) otherwise exit with success
  */

  usr_str = "success";
  //matched entire command
  return 0;
}

/**
 * Used for matching a node.
 *
 **/
int 
match_node(string &in_cmd, string &node, string &out_cmd, VecColl &conf_cmd, bool debug)
{
  if (debug) {
    cout << "match_node() in_cmd: " << in_cmd << ", node: " << node << endl;
  }
  //first compare all child nodes for a node match
  VecColl coll = vcmn_get_all_nodes(in_cmd, kTEMPLATE);
  VecIter iter = coll.begin();
  while (iter != coll.end()) {
    if (strncmp(iter->c_str(), node.c_str(), node.length()) == 0) {
      out_cmd = in_cmd + "/" + *iter;
      if (debug) {
	cout << "match_node(), matched on dir: " << *iter << endl;
      }
      vcmn_add_node(out_cmd, conf_cmd);
      return 0;
    } 
    ++iter;
  }

  //does token match type for current directory
  if (validate_type(in_cmd, node, debug) == 0) {
    //don't change out_cmd
    if (debug) {
      cout << "match_node(), matched on type: " << node << endl;
    }

    //if multi--create directory, if leaf create value file
    if (vcmn_is_multi(in_cmd)) {
      out_cmd = in_cmd + "/" + node;
      vcmn_add_node(out_cmd, conf_cmd);
    }
    else { //add element
      vcmn_add_element(out_cmd, node, conf_cmd);
    }


    return 0;
  }

  if (debug) {
    cout << "match_node(), no match" << endl;
  }
  return -1;
}

/**
 * returns matched type 
 *
 **/
int 
validate_type(string &in_cmd, string &node, bool debug)
{
  if (debug) {
    cout << "validate_type: " << in_cmd << ", " << node << endl;
  }

  string type, match;
  //access the element
  VecColl coll = vcmn_get_element(in_cmd, kTYPE);

  VecIter iter = coll.begin();
  while (iter != coll.end()) {
    StrProc str_proc(*iter, " ");
    if (iter->find("type:") != string::npos) {
      type = str_proc.get(1);
    }
    else if (iter->find("match:") != string::npos) {
      match = str_proc.get(1);
    }
    else {
    }
    ++iter;
  }

  if (type.empty()) {
    return -1;
  }

  if (debug) {
    cout << "validate_type: " << type << ", " << match << endl;
  }

  int err = 0;
  if (type == "string") {
    err = validate_string(node, match, debug);
  }
  else if (type == "bool") {
    err = validate_bool(node, match, debug);
  }
  else if (type == "ipv4") {
    err = validate_ipv4(node, match, debug);
  }
  else if (type == "ipv4net") {
    err = validate_ipv4net(node, match, debug);
  }
  else if (type == "mac") {
    err = validate_mac(node, match, debug);
  }
  else {
    //unknown type
    cout << "unknown data type" << endl;
    err = -1;
  }
  return err;
}








/**
 *
 *
 **/
int 
write_conf(VecColl &conf_coll, bool debug)
{
  if (debug) {
    cout << "write_conf" << endl;
  }
  //write conf to local here
  return vcmn_commit_to_local(conf_coll);
}
