#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <dirent.h>

#include <sys/types.h>
#include <unistd.h>

#include <iostream>
#include <string>
#include <vector>

#include "vsh_execute.hh"
#include "vsh_common.hh"
#include "vsh_config.hh"

using namespace std;

string _template_config_root("./var/vyatta/template");
string _global_config_root("./var/vyatta/config");
string _local_config_root("./var/vyatta/local");

/**
 * The template path
 *
 **/
string
vcmn_get_template_root()
{
  return _template_config_root;
}

/**
 * The global path
 *
 **/
string
vcmn_get_global_root()
{
  return _global_config_root;
}

/**
 * The local root will be stored as a shell variable. Local
 * root will be the pid of the shell process.
 *
 **/
string
vcmn_get_local_root()
{
  char buf[20];
  int pid = getppid();
  sprintf(buf, "%d", pid);
  return _local_config_root + "/" + string(buf);
}

/**
 * Walks all nodes residing in directory location
 * and returns all nodes in a collection
 **/
VecColl
vcmn_get_all_nodes(string &loc, RootNode root)
{
  DIR *dir;
  struct dirent *ent;
  VecColl coll;
  string root_loc;

  if (root == kTEMPLATE) {
    root_loc = vcmn_get_template_root() + loc;
  }
  else if (root == kGLOBAL) {
    root_loc = vcmn_get_global_root() + loc;
  }
  else {
    root_loc = vcmn_get_local_root() + loc;
  }

  if ((dir = opendir(root_loc.c_str()))) {
    while ((ent = readdir(dir))) {
      //does token match directory
      coll.push_back(ent->d_name);
    }
    closedir(dir);
  }
  return coll;
}

/**
 * Will recursively iterate through the directory
 * structure and grab.
 *
 * todo:
 *  1) handle transactions
 *  2) handle expansion of variables
 *  3) break out action processing into seperate file
 **/
void
vcmn_get_commit_actions(string &loc, VecColl &coll)
{
  //iterate through the local configuration structure
  //recursively call myself and 
  DIR *dir;
  struct dirent *ent;
  string root_loc = vcmn_get_local_root() + loc;

  if ((dir = opendir(root_loc.c_str()))) {
    while ((ent = readdir(dir))) {
      //record entries and return. will append.

      if ((string(ent->d_name) != ".") && (string(ent->d_name) != "..") && (ent->d_type == DT_DIR)) {
	VecColl tmp_coll = vcmn_get_element(loc, kACTION);

	//here is where additional processing of actions will take place...

	copy(tmp_coll.begin(), tmp_coll.end(), back_inserter(coll));

	//recursive call
	string tmp_loc = loc + "/" + string(ent->d_name);
	vcmn_get_commit_actions(tmp_loc, coll);
      }
    }
    closedir(dir);
  }
  return;
}

/**
 * Accesses a single element in the a configuration
 * directory and returns contents.
 **/
VecColl
vcmn_get_element(string &loc, ElementType e)
{
  VecColl coll;
  string file = vcmn_get_template_root() + loc;

  switch (e) {
  case kTYPE:
    file += "/type";
    break;
  case kHELP:
    file += "/help";
    break;
  case kACTION:
    file += "/action";
    break;
  default:
    return coll;
  }

  FILE *fd = fopen(file.c_str(), "r");
  if (fd) {
    char line[256];
    while (fgets(line, 255, fd) != NULL) {
      coll.push_back(line);
    }
    fclose(fd);
  }
  return coll;
}

/**
 * Instructions for setting up a directory node.
 *
 **/
void
vcmn_add_node(string &path_to_node, VecColl &conf_coll)
{
  //simple for now--will be system call once I figure out the best
  //way to represent this.
  string tmp = "mkdir -p " + vcmn_get_local_root() + "/" + path_to_node;
  conf_coll.push_back(tmp);
}

/**
 * Instructions for setting up a directory element
 *
 **/
void
vcmn_add_element(string &path_to_element, string &element, VecColl &conf_coll)
{
  //simple for now--will be system call once I figure out the best
  //way to represent this.
  string tmp = "mkdir -p " + vcmn_get_local_root() + "/" + path_to_element;
  conf_coll.push_back(tmp);

  /*
   * Note that if this is a leaf then this should create a value file
   */


  tmp = "touch " + vcmn_get_local_root() + "/" + path_to_element + "/" + element;
  //  tmp = "echo '" + element + "' > " + vcmn_get_local_root() + "/" + path_to_element + "/value";
  
  conf_coll.push_back(tmp);
}

/**
 * Takes configuration and applies changes from local
 * working configuration directory and applies these
 * changes to the global configuration. 
 *
 * Of course, this is done as a transaction where 
 * an error will back out all changes.
 *
 **/
int
vcmn_commit_to_global()
{
  //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  // NEED TO BE ABLE TO REVERT CHANGES ON ERROR HERE!
  //
  // In other words this needs to be an atomic operation
  //
  //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!      

  //write out root lock to prevent any other session from 
  //writing to global
  int err = 0;

  
  string filelock = vcmn_get_global_root() + ".lck";
  int lck = open(filelock.c_str(), O_WRONLY | O_CREAT | O_EXCL, 0644);
  if (lck < 0 && errno == EEXIST) {
    cout << "vcmn_commit_to_global(): Error: lock already in place" << endl;
    return -1;
  }
  else if (lck < 0) {
    cout << "vcmn_commit_to_global(): Error: cannot write lock" << endl;
    return -1;
  }

  /*
    Copies over the configuration over to the global. Leafs are stored in 
    value files so that they will be replaced in the cp
   */
  string tmp = "cp -fr " + vcmn_get_local_root() + "/. " + vcmn_get_global_root();
  if (vcmn_execute(tmp)) {
    cerr << "Error in local commit transaction: " << endl;
    err = -1;
    goto end;
  }

  tmp = "rm -fr " + vcmn_get_local_root();
  if (vcmn_execute(tmp)) {
    cerr << "Error in local commit transaction: " << endl;
    err = -1;
    goto end;
  }

 end:
  //last stage is to remove file lock on configuration
  unlink(filelock.c_str());
  return err;
}


/**
 * Execute instructions on local tree
 *
 **/
int
vcmn_commit_to_local(VecColl &conf_coll)
{
 //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  // NEED TO BE ABLE TO REVERT CHANGES ON ERROR HERE!
  //
  // In other words this needs to be an atomic operation
  //
  //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!      


  //copy out old config
  string tmp = "rm -fr " + vcmn_get_local_root() + "tmp";
  if (vcmn_execute(tmp)) {
    cerr << "Error in local commit transaction: " << endl;
    return -1;
  }

  //first mkdir the root (even if it exists)
  tmp = "mkdir -p " + vcmn_get_local_root() + "tmp";
  if (vcmn_execute(tmp)) {
    cerr << "Error in local commit transaction: " << endl;
    return -1;
  }

  tmp = "mkdir -p " + vcmn_get_local_root();
  if (vcmn_execute(tmp)) {
    cerr << "Error in local commit transaction: " << endl;
    return -1;
  }

  

  tmp = "cp -fr " + vcmn_get_local_root() + " " + vcmn_get_local_root() + "tmp";
  if (vcmn_execute(tmp)) {
    cerr << "Error in local commit transaction: " << endl;
    return -1;
  }

  //apply new changes
  VecIter iter = conf_coll.begin();
  while (iter != conf_coll.end()) {
    if (vcmn_execute(*iter)) {
      cerr << "Error in local commit transaction, attempting to rollback" << endl;
      
      tmp = "rm -fr " + vcmn_get_local_root();
      if (vcmn_execute(*iter)) {
	cerr << "Failure in rolling back local config " << endl;
	return -1;
      }
      tmp = "cp -fr " + vcmn_get_local_root() + "tmp " + vcmn_get_local_root();
      if (vcmn_execute(tmp)) {
	cerr << "Error in rolling back local config" << endl;
	return -1;
      }
      return -1;
    }
    ++iter;
  }

  //now clean up old configure as this is a success
  tmp = "rm -fr " + vcmn_get_local_root() + "tmp";
  if (vcmn_execute(tmp)) {
    cerr << "Error in cleaning after transaction" << endl;
    return -1;
  }

  return 0;
}

bool
vcmn_is_multi(string &cmd)
{
  return(cmd[cmd.length()-1] == ':');
}
