/**
 * Module: xrl_rl_system_node.hh
 *
 * Author: Michael Larson
 * Date: 2005
 */

#ifndef __RL_SYSTEM_XRL_RL_SYSTEM_NODE_HH__
#define __RL_SYSTEM_XRL_RL_SYSTEM_NODE_HH__


//
// RLSystem XRL-aware node definition.
//

#include "libxorp/status_codes.h"
#include "libxipc/xrl_std_router.hh"
#include "xrl/interfaces/finder_event_notifier_xif.hh"

#include "xrl/targets/rl_system_base.hh"

#include "rl_bridge.hh"
#include "rl_syslog_node.hh"
#include "rl_system_node.hh"

//
// The top-level class that wraps-up everything together under one roof
//
class XrlRLSystemNode :     public XrlStdRouter, /* used for dispatch of xrl messages */
			    public XrlRlSystemTargetBase { /* used for defined interface */
public:
    XrlRLSystemNode(EventLoop&	eventloop,
		    const string&	class_name,
		    const string&	finder_hostname,
		    uint16_t	        finder_port,
		    const string&	finder_target);
    ~XrlRLSystemNode();

    /**
     * Startup the node operation.
     *
     * @return true on success, false on failure.
     */
    bool	startup();

    /**
     * Shutdown the node operation.
     *
     * @return true on success, false on failure.
     */
    XrlCmdError	rl_system_0_1_shutdown_system();

    /**
     * Shutdown the node operation.
     *
     * @return true on success, false on failure.
     */
    XrlCmdError	rl_system_0_1_shutdown_service();

    /**
     * Shutdown the node operation.
     *
     * @return true on success, false on failure.
     */
    XrlCmdError	rl_system_0_1_shutdown_interface();

    /**
     * Shutdown the node operation.
     *
     * @return true on success, false on failure.
     */
    XrlCmdError	rl_system_0_1_shutdown_snmp();

    /**
     * Get a reference to the XrlRouter instance.
     *
     * @return a reference to the XrlRouter (@ref XrlRouter) instance.
     */
    XrlRouter&	xrl_router() { return *this; }

    /**
     * Test if the node processing is done.
     * 
     * @return true if the node processing is done, otherwise false.
     */
    bool	is_done() const { return (_status == PROC_DONE); }

protected:
    //
    // XRL target methods
    //

    /**
     *  Get name of Xrl Target
     */
    XrlCmdError common_0_1_get_target_name(
	// Output values, 
	string&	name);

    /**
     *  Get version string from Xrl Target
     */
    XrlCmdError common_0_1_get_version(
	// Output values, 
	string&	version);

    /**
     *  Get status of Xrl Target
     */
    XrlCmdError common_0_1_get_status(
	// Output values, 
	uint32_t&	status, 
	string&	reason);

    /**
     *  Request clean shutdown of Xrl Target
     */
    XrlCmdError common_0_1_shutdown();

    /**
     *  Announce target birth to observer.
     *
     *  @param target_class the target class name.
     *
     *  @param target_instance the target instance name.
     */
    XrlCmdError finder_event_observer_0_1_xrl_target_birth(
	// Input values,
	const string&	target_class,
	const string&	target_instance);

    /**
     *  Announce target death to observer.
     *
     *  @param target_class the target class name.
     *
     *  @param target_instance the target instance name.
     */
    XrlCmdError finder_event_observer_0_1_xrl_target_death(
	// Input values,
	const string&	target_class,
	const string&	target_instance);

    /**
     *  Enable/disable/start/stop RLSystem
     *
     *  @param enable if true, then enable RLSystem, otherwise disable it.
     */
    XrlCmdError rl_system_0_1_enable_rl_system(
	// Input values,
	const bool&	enable);

    XrlCmdError rl_system_0_1_start_rl_system();

    XrlCmdError rl_system_0_1_stop_rl_system();

  /**
   * Transaction only for ntp server
   *
   *
   */
  XrlCmdError rl_system_0_1_start_commit();
  XrlCmdError rl_system_0_1_end_commit();


    /**
     *  Configure an interface
     *
     *  @param address of interface
     */
    XrlCmdError rl_system_0_1_add_interface(
	// Input values,
	const IPv4&	address);

    /**
     *  Enable/disable the RLSystem trace log for all operations.
     *
     *  @param enable if true, then enable the trace log, otherwise disable it.
     */
    XrlCmdError rl_system_0_1_enable_log_trace_all(
	// Input values,
	const bool&	enable);


    /**
     * Set the host name of the system.
     *
     * @param string containing name that the host system is to be set to.
     *
     */
    XrlCmdError rl_system_0_1_set_host_name(
	// Input values,
	 const string& name,
	 const string &domain_name);
	
    XrlCmdError rl_system_0_1_delete_host_name(
	// Input values,
	 const string& name,
	 const string &domain_name);
	
    /**
     * Set the domain name of the system.
     *
     * @param string containing name that the domain name is to be set to.
     *
     */
    XrlCmdError rl_system_0_1_set_domain_name(
	// Input values,
	 const string& name);
	
    XrlCmdError rl_system_0_1_delete_domain_name(
	// Input values,
	 const string& name);
	
    /**
     * Set the domain search name of the system.
     *
     * @param string containing name that the domain search name is to be set to.
     *
     */
    XrlCmdError rl_system_0_1_set_domain_search(
	// Input values,
	 const string& name);
	
    XrlCmdError rl_system_0_1_delete_domain_search(
	// Input values,
	 const string& name);
	
    /**
     * Add name server.
     *
     * @param ipv4 containing the address for a name server.
     *
     */
    XrlCmdError rl_system_0_1_add_name_server(
	// Input values,
	 const IPv4& address);

    XrlCmdError rl_system_0_1_delete_name_server(
	// Input values,
	 const IPv4& address);

    /**
     * Sets the npt server address
     *
     * @param ipv4 address of the ntp server
     *
     */
    XrlCmdError rl_system_0_1_set_ntp(
				      // Input values,
				      const string& host);

    XrlCmdError rl_system_0_1_delete_ntp(
					 // Input values,
					 const string& host);

    /**
     * Configure the timezone.
     *
     * @param string containing the name for the timezone.
     *
     */
    XrlCmdError rl_system_0_1_set_timezone(
	// Input values,
	 const string& name);

    XrlCmdError rl_system_0_1_delete_timezone(
	// Input values,
	 const string& name);

    /**
     * Configure the host name for the static host mapping.
     *
     * @param address for the static host.
     *
     */
  XrlCmdError rl_system_0_1_set_static_host_map_inet(
						// Input values,
						const string &host_name,
						const IPv4& address);
  
  XrlCmdError rl_system_0_1_delete_static_host_map_inet(
						   // Input values,
						   const string &host_name,
						   const IPv4& address);

  XrlCmdError rl_system_0_1_set_static_host_map_inet_alias(
						// Input values,
						const string &host_name,
						const IPv4& address,
						const string& name);
  
  XrlCmdError rl_system_0_1_delete_static_host_map_inet_alias(
						   // Input values,
						   const string &host_name,
						   const IPv4& address,
						   const string& name);

  XrlCmdError rl_system_0_1_set_gateway_address(
						   // Input values,
						const IPv4& address);

  XrlCmdError rl_system_0_1_delete_gateway_address(
						   // Input values,
						   const IPv4& address);


  XrlCmdError rl_system_0_1_delete_syslog();
  
  XrlCmdError rl_system_0_1_set_syslog_console(
					       //Input values,
					       const string& facility,						
					       const string& level);
  
  XrlCmdError rl_system_0_1_delete_syslog_console(
						  //Input values,
						  const string& facility,					    
						  const string& level);
  
  XrlCmdError rl_system_0_1_set_syslog_system(
					       //Input values,
					       const string& facility,						
					       const string& level);
  
  XrlCmdError rl_system_0_1_delete_syslog_system(
						  //Input values,
						  const string& facility,					    
						  const string& level);
  
    XrlCmdError rl_system_0_1_set_syslog_system_archive(
 	//Input values,
	const uint32_t& files,
	const uint32_t& size);

    XrlCmdError rl_system_0_1_delete_syslog_system_archive(
 	//Input values,
	const uint32_t& files,
	const uint32_t& size);
	
  XrlCmdError rl_system_0_1_set_syslog_file_facility(
						     //Input values,
						     const string& file,
						     const string& facility,
						     const string& level);

    XrlCmdError rl_system_0_1_delete_syslog_file_facility(
 	//Input values,
	const string& file,
	const string& facility,
	const string& level);
	
    XrlCmdError rl_system_0_1_set_syslog_file_archive(
 	//Input values,
	const string& file,
	const uint32_t& files,
	const uint32_t& size);

    XrlCmdError rl_system_0_1_delete_syslog_file_archive(
 	//Input values,
	const string& file,
	const uint32_t& files,
	const uint32_t& size);
	
  XrlCmdError rl_system_0_1_set_syslog_host_facility(
 	//Input values,
						     const string& host,			   
						     const string& facility, 
						     const string& level);
  
  XrlCmdError rl_system_0_1_delete_syslog_host_facility(
							//Input values,
							const string& host,			   
							const string& facility,
							const string& level);
  
  XrlCmdError rl_system_0_1_set_syslog_host_log_prefix(
						       //Input values,
						       const string& host,			   
						       const string& prefix);
  
  XrlCmdError rl_system_0_1_delete_syslog_host_log_prefix(
							  //Input values,
							  const string& host,			   
							  const string& prefix);
  
  XrlCmdError rl_system_0_1_set_syslog_user(
					    //Input values,
					    const string& user, 
					    const string& facility,
					    const string& level);
  
  XrlCmdError rl_system_0_1_delete_syslog_user(
					       //Input values,
					       const string& user,
					       const string& facility,
					       const string& level);
  
  XrlCmdError rl_system_0_1_delete_service_dhcp_local();
  
  XrlCmdError rl_system_0_1_set_service_dhcp_local_name(
							//Input values,
							const string &name,
							const IPv4Net &network,
							const uint32_t &lease,
							const string &interface);
  XrlCmdError rl_system_0_1_delete_service_dhcp_local_name(
							   //Input values,
							   const string &name,
							   const IPv4Net &network,
							   const uint32_t &lease,
							   const string &interface);
  
  XrlCmdError rl_system_0_1_set_service_dhcp_local_name_default(
								//Input values,
								const string &name,
								const IPv4 &default_router);
  
  XrlCmdError rl_system_0_1_delete_service_dhcp_local_name_default(
								   //Input values,
								   const string &name,
								   const IPv4 &default_router);
  
  XrlCmdError rl_system_0_1_set_service_dhcp_local_name_dns(
							    //Input values,
							    const string &name,
							    const IPv4 &dns_server);
  XrlCmdError rl_system_0_1_delete_service_dhcp_local_name_dns(
							       //Input values,
							       const string &name,
							       const IPv4 &dns_server);
  
  XrlCmdError rl_system_0_1_set_service_dhcp_local_name_wins(
							       //Input values,
							       const string &name,
							       const IPv4 &wins_server);
    XrlCmdError rl_system_0_1_delete_service_dhcp_local_name_wins(
								  //Input values,
								  const string &name,
								  const IPv4 &wins_server);

    XrlCmdError rl_system_0_1_set_system_login_user(
						    //Input values,
						    const string &user,
						    const string &full_name,
						    const string &encrypted_pw,
						    const string &plaintext_pw);

    XrlCmdError rl_system_0_1_delete_system_login_user(
						       //Input values,
						       const string &user,
						       const string &full_name,
						       const string &encrypted_pw,
						       const string &plaintext_pw);

    XrlCmdError rl_system_0_1_delete_system_login();

    XrlCmdError rl_system_0_1_set_system_login_authentication_order(
								    //Input values,
								    const string &authentication_order);

    XrlCmdError rl_system_0_1_set_system_login_radius_server(
							     //Input values,
							     const IPv4 &address,
							     const uint32_t &port,
							     const string &secret,
							     const uint32_t &timeout);

    XrlCmdError rl_system_0_1_delete_system_login_radius_server(
								//Input values,
								const IPv4 &address,
								const uint32_t &port,
								const string &secret,
								const uint32_t &timeout);
    
    XrlCmdError rl_system_0_1_set_system_login_tacplus_server(
							      //Input values,
							      const IPv4 &address,
							      const string &secret);
    
    XrlCmdError rl_system_0_1_delete_system_login_tacplus_server(
								 //Input values,
								 const IPv4 &address,
								 const string &secret);

    XrlCmdError rl_system_0_1_set_service_http(
					       //Input values,
					       const uint32_t &port);
    XrlCmdError rl_system_0_1_delete_service_http(
					       //Input values,
					       const uint32_t &port);

    XrlCmdError rl_system_0_1_set_service_ssh(
					       //Input values,
					       const uint32_t &port,
					       const string &protocol_version);
    XrlCmdError rl_system_0_1_delete_service_ssh(
					       //Input values,
					       const uint32_t &port,
					       const string &protocol_version);

    XrlCmdError rl_system_0_1_set_service_telnet(
						 //Input values,
						 const uint32_t &port);
    XrlCmdError rl_system_0_1_delete_service_telnet(
						    //Input values,
						    const uint32_t &port);

    XrlCmdError rl_system_0_1_set_protocols_snmp_community_clients_address(
									   //Input values,
									   const string &community,
									   const string &authorization,
									   const IPv4 &address);
    XrlCmdError rl_system_0_1_delete_protocols_snmp_community_clients_address(
									   //Input values,
									   const string &community,
									   const string &authorization,
									   const IPv4 &address);


    XrlCmdError rl_system_0_1_set_protocols_snmp_trap_target(
							     //Input values,
							     const IPv4 &trap_target);
    XrlCmdError rl_system_0_1_delete_protocols_snmp_trap_target(
							     //Input values,
							     const IPv4 &trap_target);

    XrlCmdError rl_system_0_1_set_protocols_snmp_contact(
							     //Input values,
							     const string &contact);
    XrlCmdError rl_system_0_1_delete_protocols_snmp_contact(
							     //Input values,
							    const string &contact);

    XrlCmdError rl_system_0_1_set_protocols_snmp_description(
							     //Input values,
							     const string &description);
    XrlCmdError rl_system_0_1_delete_protocols_snmp_description(
							     //Input values,
								const string &description);

    XrlCmdError rl_system_0_1_set_protocols_snmp_location(
							     //Input values,
							  const string &location);
    XrlCmdError rl_system_0_1_delete_protocols_snmp_location(
							     //Input values,
							     const string &location);

  XrlCmdError rl_system_0_1_set_service_nat_pool_address(
							 //Input values,
							 const string &pool,
							 const IPv4 &address);

  XrlCmdError rl_system_0_1_delete_service_nat_pool_address(
							 //Input values,
							 const string &pool,
							 const IPv4 &address);

  XrlCmdError rl_system_0_1_set_service_nat_pool_address_range(
						     //Input values,
						     const string &pool,
						     const IPv4 &low,
						     const IPv4 &high);

  XrlCmdError rl_system_0_1_delete_service_nat_pool_address_range(
							//Input values,
							const string &pool,
							const IPv4 &address,
							const IPv4 &address);

  XrlCmdError rl_system_0_1_set_service_nat_pool_address_range_high(
						      //Input values,
						      const string &pool,
						      const IPv4 &address);

  XrlCmdError rl_system_0_1_delete_service_nat_pool_address_range_high(
							 //Input values,
							 const string &pool,
							 const IPv4 &address);

  XrlCmdError rl_system_0_1_set_service_nat_rule_match_direction(
						   //Input values,
						   const uint32_t &rule,
						   const string &match_direction);

  XrlCmdError rl_system_0_1_delete_service_nat_rule_match_direction(
						      //Input values,
						      const uint32_t &rule,
						      const string &match_direction);

  XrlCmdError rl_system_0_1_set_service_nat_rule_from_applications(
								   //Input values,
								   const uint32_t &rule,
								   const string &applications);

  XrlCmdError rl_system_0_1_delete_service_nat_rule_from_applications(
								      //Input values,
								      const uint32_t &rule,
								      const string &applications);
  
  XrlCmdError rl_system_0_1_set_service_nat_rule_from_destination_address(
									  //Input values,
									  const uint32_t &rule,
									  const IPv4 &destination_address);
  
  XrlCmdError rl_system_0_1_delete_service_nat_rule_from_destination_address(
									     //Input values,
									     const uint32_t &rule,
									     const IPv4 &destination_address);

  XrlCmdError rl_system_0_1_set_service_nat_rule_from_source_address(
								     //Input values,
								     const uint32_t &rule,
								     const IPv4 &source_address);
  
  XrlCmdError rl_system_0_1_delete_service_nat_rule_from_source_address(
									//Input values,
									const uint32_t &rule,
									const IPv4 &source_address);

  XrlCmdError rl_system_0_1_set_service_nat_rule_then_destination_pool(
								       //Input values,
								       const uint32_t &rule,
								       const string &destination_pool);
  
  XrlCmdError rl_system_0_1_delete_service_nat_rule_then_destination_pool(
									  //Input values,
									  const uint32_t &rule,
									  const string &destination_pool);
  
  
  XrlCmdError rl_system_0_1_set_service_nat_rule_then_source_pool(
								  //Input values,
								  const uint32_t &rule,
								  const string &source_pool);

  XrlCmdError rl_system_0_1_delete_service_nat_rule_then_source_pool(
								     //Input values,
								     const uint32_t &rule,
								     const string &source_pool);

  
  XrlCmdError rl_system_0_1_set_service_nat_rule_then_translation_type(
								       //Input values,
								       const uint32_t &rule,
								       const string &translation_type);
  
  XrlCmdError rl_system_0_1_delete_service_nat_rule_then_translation_type(
									  //Input values,
									  const uint32_t &rule,
									  const string &translation_type);

  XrlCmdError rl_system_0_1_start_system_bridging_bridge_group(
						 //Input values,
						 const string &bridge_group);

  XrlCmdError rl_system_0_1_commit_system_bridging_bridge_group(
						 //Input values,
						 const string &bridge_group);

  XrlCmdError rl_system_0_1_set_system_bridging_bridge_group_stp(
							   //Input values,
							   const string &bridge_group,
							   const bool &stp);
  
  XrlCmdError rl_system_0_1_set_system_bridging_bridge_group_forwarding_delay(
									//Input values,
									const string &bridge_group,
									const uint32_t &forwarding_delay);
  
  XrlCmdError rl_system_0_1_set_system_bridging_bridge_group_hello_time(
								  //Input values,
								  const string &bridge_group,
								  const uint32_t &hello_time);
  
  XrlCmdError rl_system_0_1_set_system_bridging_bridge_group_max_hello_age(
								     //Input values,
								     const string &bridge_group,
								     const uint32_t &max_hello_age);
  
  XrlCmdError rl_system_0_1_set_interfaces_bridge_priority(
							   //Input values,
							   const string &interface,
							   const string &group,
							   const uint32_t &priority);

  XrlCmdError rl_system_0_1_set_interfaces_bridge_cost(
						       //Input values,
						       const string &interface,
						       const string &group,
						       const uint32_t &cost);


  XrlCmdError rl_system_0_1_set_package_repository_component (
						       //Input values,
						       const string &repository,
						       const string &host_name,
						       const string &component,
						       const string &method,
						       const string &username,
						       const string &encrypted_password,
						       const string &plaintext_password);

  XrlCmdError rl_system_0_1_delete_package_repository_component (
						       //Input values,
						       const string &repository,
						       const string &host_name,
						       const string &component,
						       const string &method,
						       const string &username,
						       const string &encrypted_password,
						       const string &plaintext_password);

private:
  /**
     * Called when Finder connection is established.
     *
     * Note that this method overwrites an XrlRouter virtual method.
     */
  virtual void finder_connect_event();
  
  /**
     * Called when Finder disconnect occurs.
     *
     * Note that this method overwrites an XrlRouter virtual method.
     */
  virtual void finder_disconnect_event();
  
  const string& my_xrl_target_name() {
    return XrlRlSystemTargetBase::name();
  }
  
  EventLoop&		_eventloop;
  const string	_class_name;
  const string	_instance_name;
  const string	_finder_target;
    
  XrlFinderEventNotifierV0p1Client	_xrl_finder_client;
  
  static const TimeVal RETRY_TIMEVAL;
  
  bool		_is_finder_alive;
  
  RLSystemNode      _rl_system_node;
  RLSyslogNode      _rl_syslog_node;
  RLBridge             _rl_bridge;
  uint32_t          _status;
};

#endif // __RL_SYSTEM_XRL_RL_SYSTEM_NODE_HH__
