/**
 * Module: xrl_rl_system_node.hh
 *
 * Author: Michael Larson
 * Date: 2005
 */

#include "rl_system_module.h"

#include "libxorp/xorp.h"
#include "libxorp/xlog.h"
#include "libxorp/debug.h"
#include "libxorp/ipvx.hh"
#include "libxorp/status_codes.h"

#include "librl_common/rl_validate.hh"
#include "rl_syslog_node.hh"

#include "xrl_rl_system_node.hh"

XrlRLSystemNode::XrlRLSystemNode(EventLoop&	eventloop,
				 const string&	class_name,
				 const string&	finder_hostname,
				 uint16_t	finder_port,
				 const string&	finder_target)
    : XrlStdRouter(eventloop, class_name.c_str(), finder_hostname.c_str(),
		   finder_port),
      XrlRlSystemTargetBase(&xrl_router()),
      _eventloop(eventloop),
      _class_name(xrl_router().class_name()),
      _instance_name(xrl_router().instance_name()),
      _finder_target(finder_target),
      _xrl_finder_client(&xrl_router()),
      _is_finder_alive(false),
      _status(PROC_STARTUP)
{
}

XrlRLSystemNode::~XrlRLSystemNode()
{
    rl_system_0_1_shutdown_system();
}

bool
XrlRLSystemNode::startup()
{
  _status = PROC_READY;
  return true;
}

XrlCmdError
XrlRLSystemNode::rl_system_0_1_shutdown_system()
{
  //handles shutdown of system and deletion of nodes?
  _status = PROC_SHUTDOWN;
  _rl_system_node.reset();
  return XrlCmdError::OKAY();
}

//
// Finder-related events
//
/**
 * Called when Finder connection is established.
 *
 * Note that this method overwrites an XrlRouter virtual method.
 */
void
XrlRLSystemNode::finder_connect_event()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    _is_finder_alive = true;
}

/**
 * Called when Finder disconnect occurs.
 *
 * Note that this method overwrites an XrlRouter virtual method.
 */
void
XrlRLSystemNode::finder_disconnect_event()
{
    XLOG_ERROR("Finder disconnect event. Exiting immediately...");
    //printf("%s\n",__PRETTY_FUNCTION__);

    _is_finder_alive = false;
}

//
// XRL target methods
//

/**
 *  Get name of Xrl Target
 */
XrlCmdError
XrlRLSystemNode::common_0_1_get_target_name(
    // Output values, 
    string&	name)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    name = my_xrl_target_name();

    return XrlCmdError::OKAY();
}

/**
 *  Get version string from Xrl Target
 */
XrlCmdError
XrlRLSystemNode::common_0_1_get_version(
    // Output values, 
    string&	version)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    version = XORP_MODULE_VERSION;

    return XrlCmdError::OKAY();
}

/**
 *  Get status of Xrl Target
 */
XrlCmdError
XrlRLSystemNode::common_0_1_get_status(
    // Output values, 
    uint32_t&	status, 
    string&	reason)
{
  reason = " ";
  status = _status;

  if (PROC_SHUTDOWN == _status) {
    _status = PROC_DONE;
  }

  return XrlCmdError::OKAY();
}

/**
 *  Request clean shutdown of Xrl Target
 */
XrlCmdError
XrlRLSystemNode::common_0_1_shutdown()
{
    string error_msg;
    //printf("%s\n",__PRETTY_FUNCTION__);
    /*
    if (shutdown() != true) {
	error_msg = c_format("Failed to shutdown RLSystem");
	return XrlCmdError::COMMAND_FAILED(error_msg);
    }
    */
    return XrlCmdError::OKAY();
}

/**
 *  Announce target birth to observer.
 *
 *  @param target_class the target class name.
 *
 *  @param target_instance the target instance name.
 */
XrlCmdError
XrlRLSystemNode::finder_event_observer_0_1_xrl_target_birth(
    // Input values,
    const string&   target_class,
    const string&   target_instance)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
    UNUSED(target_class);
    UNUSED(target_instance);
}

/**
 *  Announce target death to observer.
 *
 *  @param target_class the target class name.
 *
 *  @param target_instance the target instance name.
 */
XrlCmdError
XrlRLSystemNode::finder_event_observer_0_1_xrl_target_death(
    // Input values,
    const string&   target_class,
    const string&   target_instance)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
    UNUSED(target_class);
    UNUSED(target_instance);
}

/**
 *  Enable/disable/start/stop RLSystem.
 *
 *  @param enable if true, then enable RLSystem, otherwise disable it.
 */
XrlCmdError
XrlRLSystemNode::rl_system_0_1_enable_rl_system(
    // Input values,
    const bool&	enable)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    UNUSED(enable);

    // XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

XrlCmdError
XrlRLSystemNode::rl_system_0_1_start_rl_system()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    // XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

XrlCmdError
XrlRLSystemNode::rl_system_0_1_stop_rl_system()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_start_commit()
{
  return _rl_system_node.start_commit();
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_end_commit()
{
  return _rl_system_node.end_commit();
}


/**
 *  Add an interface
 *
 *  @param add an interface
 */
XrlCmdError
XrlRLSystemNode::rl_system_0_1_add_interface(
    // Input values,
    const IPv4&	address)
{
    UNUSED(address);
    //printf("%s\n",__PRETTY_FUNCTION__);
    string error_msg;

    //ADD INTERFACE EXEC HERE!
    //printf("XrlRLSystemNode::rl_system_0_1_add_interface()--called!\n");

    return XrlCmdError::OKAY();
}

/**
 *  Enable/disable the RLSystem trace log for all operations.
 *
 *  @param enable if true, then enable the trace log, otherwise disable it.
 */
XrlCmdError
XrlRLSystemNode::rl_system_0_1_enable_log_trace_all(
    // Input values,
    const bool&	enable)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
    UNUSED(enable);
}

/**
 *  Enable/disable the RLSystem trace log for all operations.
 *
 *  @param enable if true, then enable the trace log, otherwise disable it.
 */
XrlCmdError
XrlRLSystemNode::rl_system_0_1_set_host_name(
    // Input values,
    const string& name,
    const string &domain_name)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    return _rl_system_node.set_host_name(name, domain_name);
}

XrlCmdError
XrlRLSystemNode::rl_system_0_1_delete_host_name(
    // Input values,
    const string& name,
    const string &domain_name)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    return _rl_system_node.delete_host_name(name, domain_name);
}

/**
 * Set the domain name of the system.
 *
 * @param string containing name that the domain name is to be set to.
 *
 */
XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_domain_name(
					       // Input values,
					       const string& name)
{
    return _rl_system_node.set_domain_name(name);
}
XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_domain_name(
						  // Input values,
						  const string& name)
{
    return _rl_system_node.delete_domain_name(name);
}

/**
 * Set the domain search name of the system.
 *
 * @param string containing name that the domain search name is to be set to.
 *
 */
XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_domain_search(
						 // Input values,
						 const string& name)
{
    _rl_system_node.set_domain_search(name);
    return XrlCmdError::OKAY();
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_domain_search(
						    // Input values,
						    const string& name)
{
    _rl_system_node.set_domain_search(name);
    return XrlCmdError::OKAY();
}

/**
 * Add name server.
 *
 * @param ipv4 containing the address for a name server.
 *
 */
XrlCmdError 
XrlRLSystemNode::rl_system_0_1_add_name_server(
					  // Input values,
					  const IPv4& address)
{
    _rl_system_node.add_name_server(address);
    return XrlCmdError::OKAY();
}

/**
 * Add name server.
 *
 * @param ipv4 containing the address for a name server.
 *
 */
XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_name_server(
					  // Input values,
					  const IPv4& address)
{
    _rl_system_node.delete_name_server(address);
    return XrlCmdError::OKAY();
}

/**
 * Configure the ntp address.
 *
 * @param IPv4 address of ntp server.
 *
 */
XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_ntp(
				       // Input values,
				       const string& host)
{
    return _rl_system_node.set_ntp(host);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_ntp(
				       // Input values,
					  const string& host)
{
    return _rl_system_node.delete_ntp(host);
}

/**
 * Configure the timezone.
 *
 * @param string containing the name for the timezone.
 *
 */
XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_timezone(
				       // Input values,
				       const string& name)
{
    _rl_system_node.set_timezone(name);
    return XrlCmdError::OKAY();
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_timezone(
				       // Input values,
				       const string& name)
{
    _rl_system_node.delete_timezone(name);
    return XrlCmdError::OKAY();
}

/**
 * Configure the host name for the static host mapping.
 *
 * @param address for the static host.
 *
 */
XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_static_host_map_inet(
						   // Input values,
						   const string &host_name,
						   const IPv4& address)
{
    return _rl_system_node.set_static_host_map_host_name_inet(host_name, address);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_static_host_map_inet(
						   // Input values,
						   const string &host_name,
						   const IPv4& address)
{
    return _rl_system_node.delete_static_host_map_host_name_inet(host_name, address);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_static_host_map_inet_alias(
						   // Input values,
						   const string &host_name,
						   const IPv4& address,
						   const string& name)
{
    return _rl_system_node.set_static_host_map_host_name_inet_alias(host_name, address, name);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_static_host_map_inet_alias(
						   // Input values,
						   const string &host_name,
						   const IPv4& address,
						   const string& name)
{
    return _rl_system_node.delete_static_host_map_host_name_inet_alias(host_name, address, name);
}

/*
 * Setting gateway address
 *
 *
 */
XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_gateway_address(
						      // Input values,
						      const IPv4& address)
{
    return _rl_system_node.set_gateway(address);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_gateway_address(
						      // Input values,
						      const IPv4& address)
{
    return _rl_system_node.delete_gateway(address);
}

/**
 * Syslog configuration handler functions below.
 *
 */

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_syslog()
{
  return _rl_syslog_node.delete_syslog();
}


XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_syslog_console(
				 //Input values,
						  const string& facility,
						  const string& level)
{
    return _rl_syslog_node.set_syslog_console(facility, level);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_syslog_console(
				    //Input values,
						     const string& facility,
						     const string& level)
{
    return _rl_syslog_node.delete_syslog_console(facility, level);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_syslog_system(
				 //Input values,
						  const string& facility,
						  const string& level)
{
    return _rl_syslog_node.set_syslog_system(facility, level);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_syslog_system(
				    //Input values,
						     const string& facility,
						     const string& level)
{
    return _rl_syslog_node.delete_syslog_system(facility, level);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_syslog_system_archive(
					  //Input values,
					  const uint32_t& files,
					  const uint32_t& size)
{
    return _rl_syslog_node.set_syslog_system_archive(files, size);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_syslog_system_archive(
					     //Input values,
					     const uint32_t& files,
					     const uint32_t& size)
{
    return _rl_syslog_node.delete_syslog_system_archive(files, size);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_syslog_file_facility(
					  //Input values,
					  const string& file, 
					  const string& facility,
					  const string& level)
{
    return _rl_syslog_node.set_syslog_file(file, facility, level);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_syslog_file_facility(
					     //Input values,
					     const string& file,
					     const string& facility,
					     const string& level)
{
    return _rl_syslog_node.delete_syslog_file(file, facility, level);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_syslog_file_archive(
					  //Input values,
					  const string& file, 
					  const uint32_t& files,
					  const uint32_t& size)
{
    return _rl_syslog_node.set_syslog_file(file, files, size);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_syslog_file_archive(
					     //Input values,
					     const string& file,
					     const uint32_t& files,
					     const uint32_t& size)
{
    return _rl_syslog_node.delete_syslog_file(file, files, size);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_syslog_host_facility(
					  //Input values,
					  const string& host,			   
					  const string& facility, 
					  const string& level)
{
    return _rl_syslog_node.set_syslog_host(host, facility, level);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_syslog_host_facility(
					     //Input values,
					     const string& host,			   
					     const string& facility,
					     const string& level)
{
    return _rl_syslog_node.delete_syslog_host(host, facility, level);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_syslog_host_log_prefix(
					  //Input values,
					  const string& host,			   
					  const string& prefix)
{
    return _rl_syslog_node.set_syslog_host(host, prefix);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_syslog_host_log_prefix(
					     //Input values,
					     const string& host,			   
					     const string& prefix)
{
    return _rl_syslog_node.delete_syslog_host(host, prefix);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_syslog_user(
					  //Input values,
					  const string& user, 
					  const string& facility,
					  const string& level)
{
    return _rl_syslog_node.set_syslog_user(user, facility, level);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_syslog_user(
					     //Input values,
					     const string& user,
					     const string& facility,
					     const string& level)
{
    return _rl_syslog_node.delete_syslog_user(user, facility, level);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_system_login_user(
						//Input values,
						const string &user,
						const string &full_name,
						const string &encrypted_pw,
						const string &plaintext_pw)
{
    return _rl_system_node.set_system_login_user(user, full_name, encrypted_pw, plaintext_pw);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_system_login_user(
						   //Input values,
						   const string &user,
						   const string &full_name,
						   const string &encrypted_pw,
						   const string &plaintext_pw)
{
    return _rl_system_node.delete_system_login_user(user, full_name, encrypted_pw, plaintext_pw);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_system_login()
{
    return _rl_system_node.delete_system_login_all();
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_system_login_authentication_order(
								//Input values,
								const string &authentication_order)
{
    return _rl_system_node.set_login_auth_order(authentication_order);
}


XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_system_login_radius_server(
							      //Input values,
							      const IPv4 &address,
							      const uint32_t &port,
							      const string &secret,
							      const uint32_t &timeout)
{
    return _rl_system_node.set_login_radius_server(address, port, secret, timeout);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_system_login_radius_server(
								 //Input values,
								 const IPv4 &address,
								 const uint32_t &port,
								 const string &secret,
								 const uint32_t &timeout)
{
    return _rl_system_node.delete_login_radius_server(address, port, secret, timeout);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_system_login_tacplus_server(
							       //Input values,
							       const IPv4 &address,
							       const string &secret)
{
    return _rl_system_node.set_login_tacplus_server(address, secret);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_system_login_tacplus_server(
								  //Input values,
								  const IPv4 &address,
								  const string &secret)
{
    return _rl_system_node.delete_login_tacplus_server(address, secret);
}


XrlCmdError 
XrlRLSystemNode::rl_system_0_1_start_system_bridging_bridge_group(
					       //Input values,
					       const string &bridge_group)
{
  return _rl_bridge.start(bridge_group);
}

XrlCmdError XrlRLSystemNode::rl_system_0_1_commit_system_bridging_bridge_group(
					       //Input values,
						 const string &bridge_group)
{
  return _rl_bridge.commit(bridge_group);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_system_bridging_bridge_group_stp(
					 //Input values,
					 const string &group,
					 const bool &stp)
{
  return _rl_bridge.set_stp(group, stp);
}
  
XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_system_bridging_bridge_group_forwarding_delay(
						      //Input values,
						      const string &group,
						      const uint32_t &forwarding_delay)
{
  return _rl_bridge.set_forwarding_delay(group, forwarding_delay);
}
  
XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_system_bridging_bridge_group_hello_time(
						//Input values,
						const string &group,
						const uint32_t &hello_time)
{
  return _rl_bridge.set_hello_time(group, hello_time);
}
  
XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_system_bridging_bridge_group_max_hello_age(
						   //Input values,
						   const string &group,
						   const uint32_t &max_hello_age)
{
  return _rl_bridge.set_max_hello_age(group, max_hello_age);
}
  
XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_interfaces_bridge_priority(
							      //Input values,
							      const string &interface,
							      const string &group,
							      const uint32_t &priority)
{
  return _rl_bridge.set_priority(interface, group, priority);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_interfaces_bridge_cost(
							  //Input values,
							  const string &interface,
							  const string &group,
							  const uint32_t &cost)
{
  return _rl_bridge.set_cost(interface, group, cost);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_set_package_repository_component (
						       //Input values,
						       const string &repository,
						       const string &host_name,
						       const string &component,
						       const string &method,
						       const string &username,
						       const string &encrypted_password,
						       const string &plaintext_password)
{
  return _rl_system_node.set_package_repository_component(repository, host_name, component, method, username, encrypted_password, plaintext_password);
}

XrlCmdError 
XrlRLSystemNode::rl_system_0_1_delete_package_repository_component (
						       //Input values,
						       const string &repository,
						       const string &host_name,
						       const string &component,
						       const string &method,
						       const string &username,
						       const string &encrypted_password,
						       const string &plaintext_password)
{
  return _rl_system_node.delete_package_repository_component(repository, host_name, component, method, username, encrypted_password, plaintext_password);
}



