/*
 * Module: rl_system_node.cc
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Michael Larson
 * Date: 2005
 * Description:
 *
 * **** End License ****
 *
 */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>

#include "rl_system_module.h"
#include "libxorp/xlog.h"

#include "librl_common/rl_str_proc.hh"
#include "librl_common/rl_fileaccess.hh"
#include "librl_common/rl_validate.hh"
#include "librl_common/rl_command.hh"
#include "rl_system_node.hh"
#include "librl_common/rl_system_node_util.hh"

using namespace std;

/**
 *
 *
 **/
RLSystemNode::RLSystemNode() :
  _resolv_filename("/etc/resolv.conf"),
  _hostname_filename("/etc/hostname.conf"),
  _localtime_filename("/etc/localtime"),
  _localtime_file_dir("/usr/share/zoneinfo/"),
  _hosts_filename("/etc/hosts"),
  _radius_filename("/etc/raddb/server"),
  _crontab_filename("/var/spool/cron/crontabs/root"), //guess as to where this should be
  _pam_filename("/etc/pam.conf"), //guess as to where this should be
  _hosts_tag("#xorp-rl entry"),
  _rdate_log_dir("/var/log/rdate"),
  _shadow_pw_file("/etc/shadow"),
  _ntp_conf_file("/etc/ntp/ntp.conf"),
  _apt_sources_list("/etc/apt/sources.list"),
  _ntp_config_changed(false)
{
  reset();
}

/**
 *
 *
 **/
RLSystemNode::~RLSystemNode()
{
}

/**
 * Resets to initial state--needs to be called on shutdown and startup
 *
 **/
XrlCmdError
RLSystemNode::reset()
{
  // should not need to do anything in this module (migrated to new CLI).
  return XrlCmdError::OKAY();
}

XrlCmdError
RLSystemNode::start_commit()
{
  _ntp_config_changed = false;
  return XrlCmdError::OKAY();
}

XrlCmdError
RLSystemNode::end_commit()
{
  if (_ntp_config_changed == true) {
    {
      rl_fileaccess fa(_ntp_conf_file);
      FILE *fd_rd, *fd_wr;
      bool success = fa.get(fd_rd, fd_wr);
      if (success == false) {
	XLOG_ERROR("RLSystemNode: failed to open file: %s", _ntp_conf_file.c_str());
	return XrlCmdError::COMMAND_FAILED();
      }
    
      //write to file only if this entry is not found
      NTPIter iter = _ntp_coll.begin();
      while (iter != _ntp_coll.end()) {
	//put line in here!
	string line("server ");
	line += *iter + "\n";
	
	fputs(line.c_str(), fd_wr);
	++iter;
      }
    }
  
    string cmd;
    if (_ntp_coll.empty() == true) {
      cmd = SBINDIR "/ntpd.init stop";
      if (rl_command::execute(cmd) == false) {
	XLOG_ERROR("RLSystemNode: error on execution: %s", cmd.c_str());
      }
    }
    else {
      cmd = SBINDIR "/ntpd.init restart";
      if (rl_command::execute(cmd) == false) {
	XLOG_ERROR("RLSystemNode: error on execution: %s", cmd.c_str());
      }
    }
  }
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::set_host_name(const std::string &name, const std::string &domain_name)
{
  std::string cmd("hostname '" + name + "'");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLSystemNode: error on execution: %s", cmd.c_str());
  }
  
  rl_fileaccess fa(_hosts_filename);
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _hosts_filename.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //walk through file looking for match
  char line[256];
  while (fgets(line, 255, fd_rd) != NULL) {
    string match_line(line);
    if (match_line.find("localhost") == string::npos) {
      fputs(line, fd_wr);
    }
  }  
  
  string tmp;
  tmp = "\n";
  fputs(tmp.c_str(), fd_wr);
  
  if (domain_name.empty() == false) {
    tmp = "127.0.0.1\t localhost " + name + domain_name + "\t " + _hosts_tag + "\n";
    fputs(tmp.c_str(), fd_wr);
  }
  
  tmp = "127.0.0.1\t localhost " + name + "\t " + _hosts_tag + "\n";
  fputs(tmp.c_str(), fd_wr);
  
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::delete_host_name(const std::string &name, const std::string &domain_name)
{
  UNUSED(name);
  UNUSED(domain_name);
  
  //note that the hostname conf file is used by busybox and needs to be
  //explicitly cleared on deletion (i.e. the command doesn't support this)
  std::string cmd("echo > " + _hostname_filename);
  if (rl_command::execute(cmd) == false) {
      XLOG_ERROR("RLSystemNode: error on execution: %s", cmd.c_str());
  }

  cmd = "hostname ''";
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLSystemNode: error on execution: %s", cmd.c_str());
  }
    
  rl_fileaccess fa(_hosts_filename);
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _hosts_filename.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //walk through file looking for match
  char line[256];
  while (fgets(line, 255, fd_rd) != NULL) {
    string match_line(line);
    if (match_line.find("localhost") == string::npos &&
	match_line.find(_hosts_tag) == string::npos) {
      fputs(line, fd_wr);
    }
  }  
  
  
  //fix for bug 1300, if ntp is configured restart it
  if (_ntp_coll.empty() == false) {
    cmd = SBINDIR "/ntpd.init restart";
    if (rl_command::execute(cmd) == false) {
      XLOG_ERROR("RLSystemNode: error on execution: %s", cmd.c_str());
    }
  }
  
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::set_domain_name(const std::string &name)
{
  if (name.empty()) {
    return XrlCmdError::OKAY();
  }

  if (rl_validate::domain_name(name) == false) {
    XLOG_ERROR("RLSystemNode: bad argument received: %s", name.c_str());
    return XrlCmdError::BAD_ARGS();
  }

  //get access to the file
  rl_fileaccess fa(_resolv_filename);

  //write to file only if this entry is not found
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _resolv_filename.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //walk through file looking for match
  char line[256];
  while (fgets(line, 255, fd_rd) != NULL) {
    string match_line(line);
    if (match_line.find("domain\t") == string::npos) {
      fputs(line, fd_wr);
    }
  }
  string str("domain\t " + name + "\n");
  fputs(str.c_str(), fd_wr);
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::delete_domain_name(const std::string &name)
{
  if (rl_validate::domain_name(name) == false) {
    XLOG_ERROR("RLSystemNode: bad argument received: %s", name.c_str());
    return XrlCmdError::BAD_ARGS();
  }

  //get access to the file
  rl_fileaccess fa(_resolv_filename);

  //write to file only if this entry is not found
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _resolv_filename.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //walk through file looking for match
  char line[256];
  while (fgets(line, 255, fd_rd) != NULL) {
    if (string(line) != string("domain\t " + name + "\n")) {
      fputs(line, fd_wr);
    }
  }
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::set_domain_search(const std::string &name)
{
  //get access to the file
  rl_fileaccess fa(_resolv_filename);

  //write to file only if this entry is not found
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _resolv_filename.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //walk through file looking for match
  bool found = false;
  char line[256];
  while (fgets(line, 255, fd_rd) != NULL) {
    fputs(line, fd_wr);
    if (string(line) == string("searchorder\t " + name + "\n")) {
      found = true;
    }
  }
  if (found == false) {
    string str("searchorder\t " + name + "\n");
    fputs(str.c_str(), fd_wr);
  }

  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::delete_domain_search(const std::string &name)
{
  //get access to the file
  rl_fileaccess fa(_resolv_filename);

  //write to file only if this entry is not found
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _resolv_filename.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //walk through file looking for match
  char line[256];
  while (fgets(line, 255, fd_rd) != NULL) {
    if (string(line) != string("searchorder " + name)) {
      fputs(line, fd_wr);
    }
  }
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::add_name_server(const IPv4 &address)
{
  {
    //get access to the file
    rl_fileaccess fa(_resolv_filename);
    
    //write to file only if this entry is not found
    FILE *fd_rd, *fd_wr;
    bool success = fa.get(fd_rd, fd_wr);
    if (success == false) {
      XLOG_ERROR("RLSystemNode: failed to open file: %s", _resolv_filename.c_str());
      return XrlCmdError::COMMAND_FAILED();
    }
    
    //walk through file looking for match
    char line[256];
    bool found = false;
    while (fgets(line, 255, fd_rd) != NULL) {
      fputs(line, fd_wr);
      string match_line(line);
      if (match_line.find(address.str()) != string::npos) {
	found = true;
      }
    }
    if (found == false) {
      string tmp("nameserver\t " + address.str() + "\n");
      fputs(tmp.c_str(), fd_wr);
    }
  }

  //fix for bug 1300, if ntp is configured restart it
  if (_ntp_coll.empty() == false) {
    string cmd = SBINDIR "/ntpd.init restart";
    if (rl_command::execute(cmd) == false) {
      XLOG_ERROR("RLSystemNode: error on execution: %s", cmd.c_str());
    }
  }
  
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::delete_name_server(const IPv4 &address)
{
  {
    //get access to the file
    rl_fileaccess fa(_resolv_filename);
    
    //write to file only if this entry is not found
    FILE *fd_rd, *fd_wr;
    bool success = fa.get(fd_rd, fd_wr);
    if (success == false) {
      XLOG_ERROR("RLSystemNode: failed to open file: %s", _hosts_filename.c_str());
      return XrlCmdError::COMMAND_FAILED();
    }
    
    //walk through file looking for match
    char line[256];
    while (fgets(line, 255, fd_rd) != NULL) {
      string match_line(line);
      if (match_line.find(address.str()) == string::npos) {
	fputs(line, fd_wr);
      }
    }
  }

  //fix for bug 1300, if ntp is configured restart it
  if (_ntp_coll.empty() == false) {
    string cmd = SBINDIR "/ntpd.init restart";
    if (rl_command::execute(cmd) == false) {
      XLOG_ERROR("RLSystemNode: error on execution: %s", cmd.c_str());
    }
  }
  
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::set_ntp(const string& host)
{
  _ntp_coll.insert(host);
  _ntp_config_changed = true;
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::delete_ntp(const string& host)
{
  _ntp_coll.erase(host);
  _ntp_config_changed = true;
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::set_timezone(const std::string &name)
{
  if (rl_validate::timezone(name) == false) {
    XLOG_ERROR("RLSystemNode: bad argument received: %s", name.c_str());
    return XrlCmdError::BAD_ARGS();
  }

  string base_name;
  if (strncasecmp("los", name.c_str(), 3) == 0) {
    base_name = _localtime_file_dir + "US/Pacific";
  } 
  else if (strncasecmp("den", name.c_str(), 3) == 0) {
    base_name = _localtime_file_dir + "US/Mountain";
  }
  else if (strncasecmp("hon", name.c_str(), 3) == 0) {
    base_name = _localtime_file_dir + "US/Hawaii";
  }
  else if (strncasecmp("new", name.c_str(), 3) == 0) {
    base_name = _localtime_file_dir + "US/Eastern";
  }
  else if (strncasecmp("chi", name.c_str(), 3) == 0) {
    base_name = _localtime_file_dir + "US/Central";
  }
  else if (strncasecmp("anc", name.c_str(), 3) == 0) {
    base_name = _localtime_file_dir + "US/Alaska";
  }
  else if (strncasecmp("pho", name.c_str(), 3) == 0) {
    base_name = _localtime_file_dir + "US/Arizona";
  }
  else {
    base_name = _localtime_file_dir + "Etc/" + name;
  }


  //  std::string cmd("echo '" + name + "' > " + _localtime_filename);
  std::string cmd("ln -fs " + base_name + " " + _localtime_filename);
  if (rl_command::execute(cmd) == false) {
      XLOG_ERROR("RLSystemNode: error on execution: %s", cmd.c_str());
  }
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::delete_timezone(const std::string &name)
{
  if (rl_validate::timezone(name) == false) {
    XLOG_ERROR("RLSystemNode: bad argument received: %s", name.c_str());
    return XrlCmdError::BAD_ARGS();
  }
  std::string cmd("ln -fs " + _localtime_file_dir + "GMT " + _localtime_filename);
  if (rl_command::execute(cmd) == false) {
      XLOG_ERROR("RLSystemNode: error on execution: %s", cmd.c_str());
  }
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::set_static_host_map_host_name_inet(const string &host_name, const IPv4 &address)
{
  rl_fileaccess fa(_hosts_filename);
  //write to file only if this entry is not found
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _hosts_filename.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //walk through file looking for match
  char line[256];
  string localhost("127.0.0.1");
  while (fgets(line, 255, fd_rd) != NULL) {
    string match_line(line);
    if (match_line.find(host_name) != string::npos &&
	match_line.find(_hosts_tag) != string::npos) {
      //just skip this entry and write out later
    }
    else if (match_line.find(localhost) != string::npos && 
	     match_line.find(_hosts_tag) != string::npos) {
      //just skip this entry and write out later
    }
    else {
      fputs(line, fd_wr);
    }
  }

  //alias stuff
  string tmp(address.str() + "\t " + host_name + " \t" + _hosts_tag + "\n");
  fputs(tmp.c_str(), fd_wr);

  //localhost stuff
  tmp = localhost + " \t localhost ";
  char buf[256];
  gethostname(buf, 255);
  tmp += string(buf);
  
  if (getdomainname(buf, 255) == 0) {
    string domainname_str(buf);
    if (rl_validate::domain_name(domainname_str) == true) {
      tmp += string(".") + domainname_str;
    }
  }
  tmp += "\t" + _hosts_tag + "\n";
  fputs(tmp.c_str(), fd_wr);


  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::delete_static_host_map_host_name_inet(const string &host_name, const IPv4 &address)
{
  UNUSED(address);

  rl_fileaccess fa(_hosts_filename);
  //write to file only if this entry is not found
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _hosts_filename.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //walk through file looking for match
  char line[256];
  while (fgets(line, 255, fd_rd) != NULL) {
    string match_line(line);
    if (match_line.find(" " + host_name + " ") != string::npos &&
	match_line.find(_hosts_tag) != string::npos &&
	match_line.find("localhost") == string::npos) {
      //skip this entry--fast to use the &&
    }
    else {
      fputs(line, fd_wr);
    }
  }
  /* work is done, now time to commit changes. */
  return XrlCmdError::OKAY();
  }

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::set_static_host_map_host_name_inet_alias(const string &host_name, const IPv4 &address, const string &alias)
{
  rl_fileaccess fa(_hosts_filename);
  //write to file only if this entry is not found
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _hosts_filename.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //walk through file looking for match
  char line[256];
  string localhost("127.0.0.1");
  while (fgets(line, 255, fd_rd) != NULL) {
    string match_line(line);
    if (match_line.find(host_name) != string::npos &&
	match_line.find(_hosts_tag) != string::npos) {
      //just skip this entry and write out later
    }
    else if (match_line.find(localhost) != string::npos && 
	     match_line.find(_hosts_tag) != string::npos) {
      //just skip this entry and write out later
    }
    else {
      fputs(line, fd_wr);
    }
  }

  //alias stuff
  string tmp(address.str() + "\t " + host_name + " " + alias + " \t" + _hosts_tag + "\n");
  fputs(tmp.c_str(), fd_wr);

  //localhost stuff
  tmp = localhost + " \t localhost ";
  char buf[256];
  gethostname(buf, 255);
  tmp += string(buf);
  
  if (getdomainname(buf, 255) == 0) {
    string domainname_str(buf);
    if (rl_validate::domain_name(domainname_str) == true) {
      tmp += string(".") + domainname_str;
    }
  }
  tmp += "\t" + _hosts_tag + "\n";
  fputs(tmp.c_str(), fd_wr);


  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::delete_static_host_map_host_name_inet_alias(const string &host_name, const IPv4 &address, const string &alias)
{
  UNUSED(address);
  UNUSED(alias);

  rl_fileaccess fa(_hosts_filename);
  //write to file only if this entry is not found
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _hosts_filename.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //walk through file looking for match
  char line[256];
  while (fgets(line, 255, fd_rd) != NULL) {
    string match_line(line);
    if (match_line.find(" " + host_name + " ") != string::npos &&
	match_line.find(_hosts_tag) != string::npos &&
	match_line.find("localhost") == string::npos) {
      //skip this entry--fast to use the &&
    }
    else {
      fputs(line, fd_wr);
    }
  }
  /* work is done, now time to commit changes. */
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::set_gateway(const IPv4 &address)
{
  std::string cmd("route add default gw " + address.str());
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLSystemNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
  
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::delete_gateway(const IPv4 &address)
{
  UNUSED(address);

  std::string cmd("route delete default");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLSystemNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::set_login_auth_order(const string &authentication_order)
{
    rl_fileaccess fa(_pam_filename);
    //write to file only if this entry is not found
    FILE *fd_rd, *fd_wr;
    bool success = fa.get(fd_rd, fd_wr);
    if (success == false) {
	XLOG_ERROR("RLSystemNode: failed to open file: %s", _pam_filename.c_str());
	return XrlCmdError::COMMAND_FAILED();
    }
    
    //walk through file looking for match
    char line[256];
    //needs to match tacplus and server address
    while (fgets(line, 255, fd_rd) != NULL) {
	string match_line(line);
/* TACACS temporarily removed; instead use radius for identification string
	string find_string("tacacs");
*/
	string find_string("radius");
	if (match_line.find(find_string) == string::npos) {
	    fputs(line, fd_wr);
	}
    }
    fputs(authentication_order.c_str(), fd_wr);
    return XrlCmdError::OKAY();
}


/**
 *
 *
 **/
XrlCmdError 
RLSystemNode::set_system_login_user(
				      const string &user,
				      const string &full_name,
				      const string &encrypted_pw,
				      const string &plaintext_pw)
{
  //strip off special characters for this string
  string tmp = plaintext_pw;
  if (plaintext_pw.find("||") != string::npos) {
    tmp = plaintext_pw.substr(2, plaintext_pw.length()-2);
  }  

  return RLSystemNodeUtil::set_system_login_user(
						 user,
						 full_name,
						 encrypted_pw,
						 tmp,
						 _shadow_pw_file);
}

/**
 *
 *
 **/
XrlCmdError 
RLSystemNode::delete_system_login_user(
					 const string &user,
					 const string &full_name,
					 const string &encrypted_pw,
					 const string &plaintext_pw)
{
  UNUSED(full_name);
  UNUSED(encrypted_pw);
  UNUSED(plaintext_pw);

  if (user == "root") {
    XLOG_ERROR("RLSystemNode: Attempt to delete user root");
    return XrlCmdError::COMMAND_FAILED("User root cannot be deleted\n");
  }

  if (user.empty() == true) {
    XLOG_ERROR("RLSystemNode: bad argument was received: user name is empty");
    return XrlCmdError::BAD_ARGS();
  }
  
  std::string cmd("moduser -d " + user);
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLSystemNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

XrlCmdError
RLSystemNode::delete_system_login_all()
{
  //disable deletion of login node...
  if (true) {
    XLOG_ERROR("RLSystemNode: Attempt to delete user root");
    return XrlCmdError::COMMAND_FAILED("User root cannot be deleted\n");
  }

  string cmd = "cp /etc/tmp/passwd /etc/passwd";
  if (rl_command::execute(cmd) == false) {
      XLOG_ERROR("RLSystemNode: error on execution: %s", cmd.c_str());
  }

  cmd = "cp /etc/tmp/shadow /etc/shadow";
  if (rl_command::execute(cmd) == false) {
      XLOG_ERROR("RLSystemNode: error on execution: %s", cmd.c_str());
  }

  return XrlCmdError::OKAY();
}
  
/**
 *
 *
 **/
XrlCmdError 
RLSystemNode::set_login_radius_server(
					       const IPv4 &address,
					       const uint32_t &port,
					       const string &secret,
					       const uint32_t &timeout)
{
  char timeout_buf[20], port_buf[20];
  int num = sprintf(timeout_buf, "%d", timeout);
  if (num < 1) {
    XLOG_ERROR("RLSystemNode: Error in converting timeout: %d", timeout);
    return XrlCmdError::COMMAND_FAILED();
  }
  num = sprintf(port_buf, "%d", port);
  if (num < 1) {
    XLOG_ERROR("RLSystemNode: Error in converting port: %d", port);
    return XrlCmdError::COMMAND_FAILED();    
  }

  rl_fileaccess fa(_radius_filename);
  //write to file only if this entry is not found
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _radius_filename.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //walk through file looking for match
  char line[256];
  while (fgets(line, 255, fd_rd) != NULL) {
    string match_line(line);
    if (match_line.find(address.str()) == string::npos) {
      fputs(line, fd_wr);
    }
  }

  string tmp(address.str() + ":" + port_buf + "\t" + secret + "\t" + timeout_buf + "\n");
  fputs(tmp.c_str(), fd_wr);

  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError 
RLSystemNode::delete_login_radius_server(
						  const IPv4 &address,
						  const uint32_t &port,
						  const string &secret,
						  const uint32_t &timeout)
{
  UNUSED(port);
  UNUSED(secret);
  UNUSED(timeout);

  rl_fileaccess fa(_radius_filename);
  //write to file only if this entry is not found
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _radius_filename.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //walk through file looking for match
  char line[256];
  while (fgets(line, 255, fd_rd) != NULL) {
    string match_line(line);
    if (match_line.find(address.str()) == string::npos) {
      fputs(line, fd_wr);
    }
  }

  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError 
RLSystemNode::set_login_tacplus_server(const IPv4 &address, const string &secret)
{
  bool found = false;
  rl_fileaccess fa(_pam_filename);
  //write to file only if this entry is not found
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _pam_filename.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //walk through file looking for match
  char line[256];
  //needs to match tacplus and server address
  while (fgets(line, 255, fd_rd) != NULL) {
    string match_line(line);
    if (match_line.find(address.str()) == string::npos || match_line.find("pam_tacplus") == string::npos) {
      fputs(line, fd_wr);
    }
    if (match_line.find("pam_tacplus") != string::npos) {
      found = true;
    }
  }
  
  //now squirt in the tacplus configuration here!
  string cmd = "auth \trequired \t/lib/security/pam_tacplus.so debug server=" + address.str() + "secret=" + secret + " encrypt\n";
  fputs(cmd.c_str(), fd_wr);

  cmd = "account \trequired \t/lib/security/pam_tacplus.so debug secret=" + address.str() + " encrypt server=ppp protocol=lcp\n";
  fputs(cmd.c_str(), fd_wr);

  cmd = "session \trequired \t/lib/security/pam_tacplus.so debug server=" + address.str() + " secret=" + secret + " encrypt service=ppp protocol=lcp\n";
  fputs(cmd.c_str(), fd_wr);

  if (found == false) {
    cmd = "password \trequired/lib/security/pam_cracklib.so\n";
    fputs(cmd.c_str(), fd_wr);
    cmd = "password \trequired/lib/security/pam_pwdb.so shadow use_authtok\n";
    fputs(cmd.c_str(), fd_wr);
  }
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError 
RLSystemNode::delete_login_tacplus_server(const IPv4 &address, const string &secret)
{
  UNUSED(secret);

  rl_fileaccess fa(_pam_filename);
  //write to file only if this entry is not found
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _pam_filename.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //walk through file looking for match
  char line[256];
  bool found_other_config = false;
  //needs to match tacplus and server address
  while (fgets(line, 255, fd_rd) != NULL) {
    bool write = true;
    string match_line(line);
    if (match_line.find(address.str()) != string::npos && match_line.find("pam_tacplus") != string::npos) {
      write = false;
    }
    else if (match_line.find("pam_cracklib.so") != string::npos) {
      write = false;
    }
    else if (match_line.find("pam_pwdb.so") != string::npos) {
      write = false;
    }

    if (match_line.find(address.str()) == string::npos && match_line.find("pam_tacplus") != string::npos) {
      found_other_config = true;
    }

    if (write == true) {
      fputs(line, fd_wr);
    }
  }
  
  //re-apply these changes
  if (found_other_config == true) {
    string cmd = "password \trequired/lib/security/pam_cracklib.so\n";
    fputs(cmd.c_str(), fd_wr);
    cmd = "password \trequired/lib/security/pam_pwdb.so shadow use_authtok\n";
    fputs(cmd.c_str(), fd_wr);
  }
  return XrlCmdError::OKAY();
}
  
/**
 *
 *
 **/
XrlCmdError
RLSystemNode::set_package_repository_component(
				      const string &repository,
				      const string &host_name,
				      const string &component,
				      const string &method,
				      const string &username,
				      const string &encrypted_password,
				      const string &plaintext_password)
{
  // Lock Config File
  rl_fileaccess fa(_apt_sources_list);

  FILE *fd_rd, *fd_wr;
  if(false == fa.get(fd_rd, fd_wr)) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _apt_sources_list.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }

  // Walk through file looking for matches, ignore non-matches
  char line[999];
  while (fgets(line, -1+sizeof line, fd_rd) != NULL) {
    string match_line(line);
    if (string::npos == match_line.find(repository + " " + component)) {
      fputs(line, fd_wr); // echo this line out, it's not a match
    } else {
       // echo out new line as just configured
       string tmp("deb " + host_name + "/ " + repository + " " + component + "\n");
       fputs(tmp.c_str(), fd_wr);
    }
  }

  
  UNUSED (method);
  UNUSED (username);
  UNUSED (encrypted_password);
  UNUSED (plaintext_password);
  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLSystemNode::delete_package_repository_component(
				      const string &repository,
				      const string &host_name,
				      const string &component,
				      const string &method,
				      const string &username,
				      const string &encrypted_password,
				      const string &plaintext_password)
{
  // Lock Config File
  rl_fileaccess fa(_apt_sources_list);

  // Write to file only if this entry is not found
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("RLSystemNode: failed to open file: %s", _apt_sources_list.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }

  // Walk through file looking for match
  char line[256];
  while (fgets(line, 255, fd_rd) != NULL) {
    string match_line(line);
    if (match_line.find(repository + " " + component) == string::npos  ) {
      fputs(line, fd_wr);
    }
    else {
      // do nothing, skip the matched line
    }
  }

  UNUSED (host_name);
  UNUSED (method);
  UNUSED (username);
  UNUSED (encrypted_password);
  UNUSED (plaintext_password);
  return XrlCmdError::OKAY();
}
