/*
 * Module: rl_syslog_node.cc
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Michael Larson
 * Date: 2005
 * Description:
 *
 * **** End License ****
 *
 */

#include <stdio.h>
#include <string>
#include <algorithm>
#include <iostream>
#include <cctype>
#include "rl_system_module.h"
#include "libxorp/xlog.h"
#include "librl_common/rl_fileaccess.hh"
#include "librl_common/rl_validate.hh"
#include "librl_common/rl_command.hh"
#include "rl_syslog_node.hh"


using namespace std;

RLSyslogNode::RLSyslogNode() :
  _syslog_filename("/etc/syslog.conf"),
  _syslog_logrotate_dir("/etc/logrotate.d/"),
  _rl_syslog_conf("rl_syslog.conf"),
  _rl_syslog_file_dir("/var/log/user/")
{
  reset();
}

RLSyslogNode::~RLSyslogNode()
{
  reset();
}

/**
 * Resets all syslog configurations to factory state
 *
 */
XrlCmdError
RLSyslogNode::reset()
{
  printf("RLSyslogNode::reset(), not yet implemented!");

  //copy over all default files
  /* syslog conf */

  //remove any files
  /* rl syslog conf */
  /* logrotate directory */


  //remove this on startup for now as this is now a permanent node...
  /*
  //remove all filelocks, if any
  string tmp(_syslog_logrotate_dir + "*");
  rl_fileaccess::init(tmp);
  rl_fileaccess::init(_rl_syslog_conf);
  rl_fileaccess::init(_syslog_filename);

  //stop any processes

  //hup any processes
  // syslog restart 
  std::string cmd("/etc/init.d/syslogd restart");
  if (rl_command::execute(cmd) == false) {
    return XrlCmdError::COMMAND_FAILED();
  }
*/
  return XrlCmdError::OKAY();
}

XrlCmdError
RLSyslogNode::delete_syslog()
{
  //remove all filelocks, if any
  string tmp(_syslog_logrotate_dir + "*");
  rl_fileaccess::init(tmp);
  rl_fileaccess::init(_rl_syslog_conf);
  rl_fileaccess::init(_syslog_filename);
  
  //now treat the global syslog
  string empty;
  delete_syslog_system(empty, empty);

  //stop any processes
  
  //hup any processes
  // syslog restart 
  std::string cmd(SBINDIR "/sysklogd.init restart");
  if (rl_command::execute(cmd) == false) {
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 * Looks for /dev/console entry in syslog.conf file.
 *
 */
XrlCmdError
RLSyslogNode::set_syslog_console(const string &facility, const string &level)
{
  /* TODO: validate facility to only alphanumber characters*/
  std::ostringstream tmp;
  tmp << uppercase << level;
  string match_one(facility + ".");
  string match_two("/dev/console");
  string update_line(match_one + tmp.str() + "\t" + match_two + "\n"); 
  return update_syslog(match_one, match_two, update_line);
}

XrlCmdError
RLSyslogNode::delete_syslog_console(const string &facility, const string &level)
{
  UNUSED(level);

  string match_one(facility + ".");
  string match_two("/dev/console");
  string update_line("");
  return update_syslog(match_one, match_two, update_line);
}

/**
 *
 *
 */
XrlCmdError
RLSyslogNode::set_syslog_file(const string &file,
				const string &facility,
				const string &level)
{
  if (rl_validate::domain_name(file) == false) {
    return XrlCmdError::COMMAND_FAILED("file name contains illegal characters");
  }

  std::ostringstream tmp;
  tmp << uppercase << level;
  string match_one = string(facility + ".");
  string match_two = string(_rl_syslog_file_dir + file + " ");
  string update_line(match_one + tmp.str() + "\t" + match_two + " \n"); 
  return update_syslog(match_one, match_two, update_line);
}

/**
 *
 *
 */
XrlCmdError
RLSyslogNode::set_syslog_file(const string &file,
				const uint32_t &files, 
				const uint32_t &size)
{
  /*
   * Now we need to write to the logrotate conf file
   */
  return update_logrotate(file, files, size, true);
}

XrlCmdError
RLSyslogNode::set_syslog_system(const string &facility, const string &level)
{
  string file("/var/log/messages");
  std::ostringstream tmp;
  tmp << uppercase << level;
  string match_one = string(facility + ".");
  string match_two = string(file);
  string update_line(match_one + tmp.str() + "\t" + match_two + " \n"); 
  string empty;
  return update_syslog(empty, match_two, update_line);
}

XrlCmdError
RLSyslogNode::delete_syslog_system(const string &facility, const string &level)
{
  UNUSED(level);
  UNUSED(facility);
  string file("/var/log/messages");
  string match_one;
  string match_two = string(file);
  string update_line("*.warning\t/var/log/messages \n"); 
  return update_syslog(match_one, match_two, update_line);
}

XrlCmdError
RLSyslogNode::set_syslog_system_archive(const uint32_t &files,
					const uint32_t &size)
{
  /*
   * Now we need to write to the logrotate conf file
   */
  string file;
  return update_logrotate(file, files, size, true);
}

XrlCmdError
RLSyslogNode::delete_syslog_system_archive(const uint32_t &files,
			     const uint32_t &size)
{
  /*
   * Now we need to write to the logrotate conf file
   */
  string file;
  return update_logrotate(file, files, size, false);
}


XrlCmdError
RLSyslogNode::delete_syslog_file(const string &file,
				   const string &facility,
				   const string &level)
{
  UNUSED(level);

  string match_one = string(facility + ".");
  string match_two = string(_rl_syslog_file_dir + file);
  string update_line("");
  return update_syslog(match_one, match_two, update_line);
}


XrlCmdError
RLSyslogNode::delete_syslog_file(const string &file,
				   const uint32_t &files,
				   const uint32_t &size)
{
  /*
   * Now we need to write to the logrotate conf file
   */
  return update_logrotate(file, files, size, false);
}


XrlCmdError
RLSyslogNode::set_syslog_host(const string& host,			   
				const string& facility,
				const string& level)
{
  std::ostringstream tmp;
  tmp << uppercase << level;
  string match_one = string(facility + ".");
  string match_two("@" + host + " ");
  string update_line(match_one + tmp.str() + "\t" + match_two + " \n"); 
  return update_syslog(match_one, match_two, update_line);
}
  
XrlCmdError
RLSyslogNode::delete_syslog_host(const string& host,			   
				   const string& facility,
				   const string& level)
{
  UNUSED(level);

  string match_one = string(facility + ".");
  string match_two("@" + host + " ");
  string update_line("");
  return update_syslog(match_one, match_two, update_line);
}


XrlCmdError
RLSyslogNode::set_syslog_host(const string& host,			   
		  const string& prefix)
{
  FILE *fd_rd, *fd_wr;
  //let's write this out to a file for reading...
  rl_fileaccess fa(_rl_syslog_conf);
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    return XrlCmdError::COMMAND_FAILED();
  }

  char line[256];
  string key = host + ":";
  while (fgets(line, 255, fd_rd) != NULL) {
    string match_line(line);
    if (match_line.find(string(key)) == string::npos) {
      fputs(line, fd_wr);
    }
    else {
      string keyvalue = key + prefix;
      fputs(keyvalue.c_str(), fd_wr);
    }
  }
  return XrlCmdError::OKAY();
}
  
XrlCmdError
RLSyslogNode::delete_syslog_host(const string& host,			   
		   const string& prefix)
{
  UNUSED(prefix);

  FILE *fd_rd, *fd_wr;
  //let's write this out to a file for reading...
  rl_fileaccess fa(_rl_syslog_conf);
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    return XrlCmdError::COMMAND_FAILED();
  }

  char line[256];
  string key = host + ":";
  while (fgets(line, 255, fd_rd) != NULL) {
    if (key.find(string(line)) == string::npos) {
      fputs(line, fd_wr);
    }
  }
  return XrlCmdError::OKAY();
}


XrlCmdError
RLSyslogNode::set_syslog_user(const string& user,			   
		  const string& facility,
		  const string& level)
{
  std::ostringstream tmp;
  tmp << uppercase << level;
  string match_one = string(facility + "." + tmp.str());
  string match_two = string(" ") + user + string(" ");
  string update_line(facility + "." + tmp.str() + " " + user + "\n"); 
  return update_syslog(match_one, match_two, update_line);
}

XrlCmdError
RLSyslogNode::delete_syslog_user(const string& user,			   
		     const string& facility,
		     const string& level)
{
  std::ostringstream tmp;
  tmp << uppercase << level;
  string match_one = string(facility + "." + tmp.str());
  string match_two = string(" ") + user + string(" ");
  string update_line("");
  return update_syslog(match_one, match_two, update_line);
}

/**
 *
 *
 */
XrlCmdError
RLSyslogNode::update_syslog(const string &match_one, const string &match_two, const string &update_line)
{
    {
	FILE *fd_rd, *fd_wr;
	rl_fileaccess fa(_syslog_filename);
	bool success = fa.get(fd_rd, fd_wr);
	if (success == false) {
	    return XrlCmdError::COMMAND_FAILED();
	}
	
	//walk through file looking for match
	char line[256];
	while (fgets(line, 255, fd_rd) != NULL) {
	    string match_line(line);
#ifdef DEBUG
	    printf("%s: match_one: %s, match_two: %s, match_line: %s\n", __PRETTY_FUNCTION__, match_one.c_str(), match_two.c_str(), match_line.c_str());
	    if (match_line.find(match_one) != string::npos)
		printf("match_one found!\n");
	    if (match_line.find(match_two) != string::npos)
		printf("match_two found!\n");
#endif
	    if (!(match_line.find(match_one) != string::npos && match_line.find(match_two) != string::npos)) {
		fputs(line, fd_wr);
	    }
	}
	
	/* Now write our changes out. */
	if (update_line.empty() == false) {
	    fputs(update_line.c_str(), fd_wr);
	}
    }
    /* work is done, now time to commit changes. */
    /* Now hup the syslog daemon to recognize the changes. */

    std::string cmd(SBINDIR "/sysklogd.init restart");
    if (rl_command::execute(cmd) == false) {
	return XrlCmdError::COMMAND_FAILED();
    }

    return XrlCmdError::OKAY();
}

/**
 *
 *
 */
XrlCmdError
RLSyslogNode::update_logrotate(const string &file, const uint32_t &files, const uint32_t &size, bool set)
{
  string dir_plus_file;
  if (file.empty()) {
    dir_plus_file = "/etc/logrotate.d/messages";
  }
  else {
    dir_plus_file = _syslog_logrotate_dir + file;
  }

  {
    FILE *fd_rd, *fd_wr;
    
    char files_buf[80], size_buf[80];
    int num = sprintf(files_buf, "%d", files);
    if (num < 1) {
      XLOG_ERROR("RLSyslogNode::update_logrotate() error writing file number: %d", files);
      return XrlCmdError::COMMAND_FAILED();
    }
    
    num = sprintf(size_buf, "%d", size);
    if (num < 1) {
      XLOG_ERROR("RLSyslogNode::update_logrotate() error writing size number: %d", size);
      return XrlCmdError::COMMAND_FAILED();
    }
    
    rl_fileaccess fa(dir_plus_file);
    bool success = fa.get(fd_rd, fd_wr);
    if (success == false) {
      XLOG_ERROR("RLSyslogNode::update_logrotate() error opening file: %s", dir_plus_file.c_str());
      return XrlCmdError::COMMAND_FAILED();
    }
    
    bool found = false;
    char line[255];
    while (fgets(line, 255, fd_rd) != NULL) {
      string match_line(line);
      if (match_line.find(dir_plus_file) == string::npos) {
	//now skip over entry
	while (fgets(line, 255, fd_rd) != NULL) {
	  string delimiter("}");
	  if (delimiter.find(line) != string::npos) {
	    break;
	  }
	}
	found = true;
      } 
      else {
	fputs(line, fd_wr);
      }
    }
    
    if (set == true) {
      //write out new entry here
      string new_entry;
      char buf[80];
      if (file.empty()) {
	new_entry = "/var/log/messages {\n";
      }
      else {
	new_entry = _rl_syslog_file_dir + file + " {\n";
      }

      fputs(new_entry.c_str(), fd_wr);

      new_entry = "  missingok\n";
      fputs(new_entry.c_str(), fd_wr);

      new_entry = "  notifempty\n";
      fputs(new_entry.c_str(), fd_wr);

      sprintf(buf, "%d", files);
      new_entry = string("  rotate ") + buf + "\n";
      fputs(new_entry.c_str(), fd_wr);

      sprintf(buf, "%d", size);
      new_entry = "  size=" + string(buf) + "k\n";
      fputs(new_entry.c_str(), fd_wr);

      new_entry = "  postrotate\n";
      fputs(new_entry.c_str(), fd_wr);

      new_entry = "  kill -HUP `cat /var/run/syslogd.pid`\n";
      fputs(new_entry.c_str(), fd_wr);

      new_entry = "  endscript\n";
      fputs(new_entry.c_str(), fd_wr);

      new_entry = "}\n";
      fputs(new_entry.c_str(), fd_wr);
    }
  }

  /* work is done, now time to commit changes. */
  /* Now hup the syslog daemon to recognize the changes. */

  std::string cmd(SBINDIR "/sysklogd.init restart");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLSyslogNode::update_logrotate() error opening file: %s", dir_plus_file.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }

  return XrlCmdError::OKAY();
}
