/*
 * Module: rl_bridge.cc
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Michael Larson
 * Date: 2005
 * Description:
 *
 * **** End License ****
 *
 */

#include <stdio.h>
#include <string>

#include "config.h"

#include "rl_system_module.h"

#include "libxorp/xlog.h"
#include "librl_common/rl_validate.hh"
#include "librl_common/rl_command.hh"
#include "rl_bridge.hh"

using namespace std;

/**
 *
 *
 **/
RLBridge::RLBridge()
{
    RL_TRACE;
    reset();
}

/**
 *
 *
 **/
RLBridge::~RLBridge()
{
    RL_TRACE;
    reset();
}

/**
 * Resets to initial state--needs to be called on shutdown and startup
 *
 **/
XrlCmdError
RLBridge::reset()
{
    RL_TRACE;
    printf("RLBridge::reset(), Not yet implemented\n");
  
  //copy over all default files

  //remove any files

  //remove all filelocks, if any

  //stop any processes
  /* should be none here... */

  //hup any processes
  /* How do we hup the login service after replacing passwd and shadow files? */
  /* Hup crontab process */
  /* do we do anything with localtime? */
  /* radius server? */
  /* tacplus server? */

  return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLBridge::start(const string &group)
{
    UNUSED(group);

    RL_TRACE;
    return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLBridge::commit(const string &group)
{
    RL_TRACE;
    bool found = false;
    
    BridgeDataIter data_iter = bridge_data_coll_.find(group);
    if (data_iter == bridge_data_coll_.end()) {
	XLOG_ERROR("RLBridge error in accessing bridge group data");
	return XrlCmdError::COMMAND_FAILED();
    }
    
    //if no interfaces are defined remove bridge group and exit
    BridgeInterfaceIter iter = bridge_interface_coll_.begin();
    while (iter != bridge_interface_coll_.end()) {
	if (iter->second.bridge_ == group) {
	    found = true;
	    break;
	}
	++iter;
    }
    
    //remove bridge entry here
    if (data_iter->second.added_ == false ) {
	string cmd = "brctl addbr " + group;
	if (rl_command::execute(cmd) == false) {
	    return XrlCmdError::COMMAND_FAILED();
	}
	data_iter->second.added_ = true;
    }
    else {
	if (found == false) {
	    string cmd = "brctl delbr " + group;
	    if (rl_command::execute(cmd) == false) {
		return XrlCmdError::COMMAND_FAILED();
	    }
	    data_iter->second.added_ = false;
	    XLOG_INFO("RLBridge deleted bridge group: %s", group.c_str());
	    return XrlCmdError::OKAY();
	}
    }

    
    iter = bridge_interface_coll_.begin();
    while (iter != bridge_interface_coll_.end()) {
	if (iter->second.bridge_ == group) {
	    //for each interface set interface to 0.0.0.0
	    
	    //apply stp
	    string cmd = "brctl setstp " + group + " ";
	    if (data_iter->second.stp_ == true) {
		cmd += "on";
	    }
	    else {
		cmd += "off";
	    }
	    if (rl_command::execute(cmd) == false) {
		return XrlCmdError::COMMAND_FAILED();
	    }
	    
	    char buf[40];
	    cmd = "brctl setfd " + group + " ";
	    sprintf(buf, "%d", data_iter->second.forwarding_delay_);
	    cmd += string(buf);
	    if (rl_command::execute(cmd) == false) {
		return XrlCmdError::COMMAND_FAILED();
	    }
	    
	    cmd = "brctl sethello " + group + " ";
	    sprintf(buf, "%d", data_iter->second.hello_time_);
	    cmd += string(buf);
	    if (rl_command::execute(cmd) == false) {
		return XrlCmdError::COMMAND_FAILED();
	    }
	    
	    cmd = "brctl maxage " + group + " ";
	    sprintf(buf, "%d", data_iter->second.max_hello_age_);
	    cmd += string(buf);
	    if (rl_command::execute(cmd) == false) {
		return XrlCmdError::COMMAND_FAILED();
	    }
	}
	++iter;
    }
    return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLBridge::set_stp(const string &group, const bool &stp)
{
    RL_TRACE;
    BridgeDataIter iter = bridge_data_coll_.find(group);
    if (iter == bridge_data_coll_.end()) {
	bridge_data_coll_.insert(pair<string, BridgeData>(group, BridgeData()));
	iter = bridge_data_coll_.find(group);
    }
    iter->second.stp_ = stp;;
    return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLBridge::set_forwarding_delay(const string &group, const uint32_t &forwarding_delay)
{
    RL_TRACE;
    BridgeDataIter iter = bridge_data_coll_.find(group);
    if (iter == bridge_data_coll_.end()) {
	bridge_data_coll_.insert(pair<string, BridgeData>(group, BridgeData()));
	iter = bridge_data_coll_.find(group);
    }
    iter->second.forwarding_delay_ = forwarding_delay;
    return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLBridge::set_hello_time(const string &group, const uint32_t &hello_time)
{
    RL_TRACE;
    BridgeDataIter iter = bridge_data_coll_.find(group);
    if (iter == bridge_data_coll_.end()) {
	bridge_data_coll_.insert(pair<string, BridgeData>(group, BridgeData()));
	iter = bridge_data_coll_.find(group);
    }
    iter->second.hello_time_ = hello_time;
    return XrlCmdError::OKAY();
}

/**
 *
 *
 **/
XrlCmdError
RLBridge::set_max_hello_age(const string &group, const uint32_t &max_hello_age)
{
    RL_TRACE;
    BridgeDataIter iter = bridge_data_coll_.find(group);
    if (iter == bridge_data_coll_.end()) {
	bridge_data_coll_.insert(pair<string, BridgeData>(group, BridgeData()));
	iter = bridge_data_coll_.find(group);
    }
    iter->second.max_hello_age_ = max_hello_age;
    return XrlCmdError::OKAY();
}


/**
 *
 *
 */
XrlCmdError 
RLBridge::set_priority(const string &interface, const string &group, const uint32_t &priority)
{
    RL_TRACE;
    bool new_group = false;

    BridgeInterfaceIter iter = bridge_interface_coll_.find(interface);
    if (iter == bridge_interface_coll_.end()) {
	new_group = true;
	bridge_interface_coll_.insert(pair<string, BridgeInterfaceData>(interface, BridgeInterfaceData()));
	iter = bridge_interface_coll_.find(interface);
	iter->second.bridge_ = group;
    }
    iter->second.priority_ = priority;
    
    if (new_group == true) {
	ZeroInterfaceAddress zero_addr(interface);
	if (zero_addr.success() == true) {
	    string cmd = "brctl addbr " + group;
	    if (rl_command::execute(cmd) == false) {
		return XrlCmdError::COMMAND_FAILED();
	    }
	}
    }

    //set the bridge here
    char buf[40];
    sprintf(buf, "%d", priority);
    string cmd = "brctl setportprio " + group + " " + interface + " " + string(buf);
    if (rl_command::execute(cmd) == false) {
	return XrlCmdError::COMMAND_FAILED();
    }
    return XrlCmdError::OKAY();
}

/**
 *
 *
 */
XrlCmdError 
RLBridge::set_cost(const string &interface, const string &group, const uint32_t &cost)
{
    RL_TRACE;
    bool new_group = false;

    BridgeInterfaceIter iter = bridge_interface_coll_.find(interface);
    if (iter == bridge_interface_coll_.end()) {
	bridge_interface_coll_.insert(pair<string, BridgeInterfaceData>(interface, BridgeInterfaceData()));
	iter = bridge_interface_coll_.find(interface);
	new_group = true;
	iter->second.bridge_ = group;
    }
    iter->second.cost_ = cost;
    
    if (new_group == true) {
	ZeroInterfaceAddress zero_addr(interface);
	if (zero_addr.success() == true) {
	    string cmd = "brctl addbr " + group;
	    if (rl_command::execute(cmd) == false) {
		return XrlCmdError::COMMAND_FAILED();
	    }
	}
    }

    //set the bridge here
    char buf[40];
    sprintf(buf, "%d", cost);
    string cmd = "brctl setpathcost " + group + " " + interface + " " + string(buf);
    if (rl_command::execute(cmd) == false) {
	return XrlCmdError::COMMAND_FAILED();
    }
    return XrlCmdError::OKAY();
}

/**
 *
 *
 */
ZeroInterfaceAddress::ZeroInterfaceAddress(const string &interface) : interface_(interface) 
{
    RL_TRACE;
    address_ = set_interface("0.0.0.0");
}

/**
 *
 *
 */
ZeroInterfaceAddress::~ZeroInterfaceAddress()
{
    RL_TRACE;
    if (address_.empty() == false) {
	set_interface(address_);
    }
}

/**
 *
 *
 */
string 
ZeroInterfaceAddress::set_interface(const string address)
{
    RL_TRACE;
    char line[256];
    
    //ifconfig eth0 | grep 'inet addr'
    string cmd = "ifconfig " + interface_ + " | grep 'inet addr'";
    FILE *f = popen(cmd.c_str(), "r");
    if (f) {
	fgets(line, 255, f);
	string output(line);
	int start = output.find(":");
	string output2 = output.substr(start, output.length() - start);
	int end = output2.find(" ");
	
	if (pclose(f) == -1) {
	    XLOG_ERROR("RLBridge failure in set_interface");
	    return string("");
	}

	cmd = PKGLIBEXECDIR "/xrl_shell_funcs_vrrp.sh set_interface " + interface_ + " " + interface_ + " " + address;
	if (rl_command::execute(cmd) == true) {
	    printf("ZeroInterfaceAddress::set_interface(): output received: %s\n", output2.substr(1,end).c_str());
	    return output2.substr(1, end);
	}

	
	/*
	//now set the interface
	cmd = "ifconfig -i " + interface_ + " " + address;
	if (rl_command::execute(cmd) == true) {
	    printf("ZeroInterfaceAddress::set_interface(): output received: %s\n", output2.substr(1,end).c_str());
	    return output2.substr(1, end);
	}
	*/
    }
    XLOG_ERROR("RLBridge failure in set_interface");
    return string("");
}
