/**
 * Module: xrl_rl_service_node.hh
 *
 * Author: Michael Larson
 * Date: 2005
 */

#include "rl_service_module.h"

#include "libxorp/xorp.h"
#include "libxorp/xlog.h"
#include "libxorp/debug.h"
#include "libxorp/ipvx.hh"
#include "libxorp/status_codes.h"

#include "librl_common/rl_validate.hh"

#include "rl_service_node.hh"
#include "xrl_rl_service_node.hh"

XrlRLServiceNode::XrlRLServiceNode(EventLoop&	eventloop,
				 const string&	class_name,
				 const string&	finder_hostname,
				 uint16_t	finder_port,
				 const string&	finder_target)
    : XrlStdRouter(eventloop, class_name.c_str(), finder_hostname.c_str(),
		   finder_port),
      XrlRlServiceTargetBase(&xrl_router()),
      _eventloop(eventloop),
      _class_name(xrl_router().class_name()),
      _instance_name(xrl_router().instance_name()),
      _finder_target(finder_target),
      _xrl_finder_client(&xrl_router()),
      _is_finder_alive(false),
      _status(PROC_STARTUP)
{
}

XrlRLServiceNode::~XrlRLServiceNode()
{
}

bool
XrlRLServiceNode::startup()
{
  _status = PROC_READY;
  return true;
}

XrlCmdError
XrlRLServiceNode::rl_service_0_1_shutdown_service()
{
  //handles shutdown of service and deletion of nodes?
  _status = PROC_SHUTDOWN;
  _rl_service_node.reset();
  return XrlCmdError::OKAY();
}

//
// Finder-related events
//
/**
 * Called when Finder connection is established.
 *
 * Note that this method overwrites an XrlRouter virtual method.
 */
void
XrlRLServiceNode::finder_connect_event()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    _is_finder_alive = true;
}

/**
 * Called when Finder disconnect occurs.
 *
 * Note that this method overwrites an XrlRouter virtual method.
 */
void
XrlRLServiceNode::finder_disconnect_event()
{
    XLOG_ERROR("Finder disconnect event. Exiting immediately...");
    //printf("%s\n",__PRETTY_FUNCTION__);

    _is_finder_alive = false;
}

//
// XRL target methods
//

/**
 *  Get name of Xrl Target
 */
XrlCmdError
XrlRLServiceNode::common_0_1_get_target_name(
    // Output values, 
    string&	name)
{
    name = my_xrl_target_name();
    return XrlCmdError::OKAY();
}

/**
 *  Get version string from Xrl Target
 */
XrlCmdError
XrlRLServiceNode::common_0_1_get_version(
    // Output values, 
    string&	version)
{
    version = XORP_MODULE_VERSION;
    return XrlCmdError::OKAY();
}

/**
 *  Get status of Xrl Target
 */
XrlCmdError
XrlRLServiceNode::common_0_1_get_status(
    // Output values, 
    uint32_t&	status, 
    string&	reason)
{
  /*
    returns current status of module:

    case PROC_STARTUP:
    case PROC_READY:
    case PROC_SHUTDOWN:
    case PROC_DONE:
  */

  reason = " ";
  status = _status;

  if (PROC_SHUTDOWN == _status) {
    _status = PROC_DONE;
  }

  return XrlCmdError::OKAY();
}

/**
 *  Request clean shutdown of Xrl Target
 */
XrlCmdError
XrlRLServiceNode::common_0_1_shutdown()
{
    return XrlCmdError::OKAY();
}

/**
 *  Announce target birth to observer.
 *
 *  @param target_class the target class name.
 *
 *  @param target_instance the target instance name.
 */
XrlCmdError
XrlRLServiceNode::finder_event_observer_0_1_xrl_target_birth(
    // Input values,
    const string&   target_class,
    const string&   target_instance)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
    UNUSED(target_class);
    UNUSED(target_instance);
}

/**
 *  Announce target death to observer.
 *
 *  @param target_class the target class name.
 *
 *  @param target_instance the target instance name.
 */
XrlCmdError
XrlRLServiceNode::finder_event_observer_0_1_xrl_target_death(
    // Input values,
    const string&   target_class,
    const string&   target_instance)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
    UNUSED(target_class);
    UNUSED(target_instance);
}

/**
 *  Enable/disable/start/stop RLService.
 *
 *  @param enable if true, then enable RLService, otherwise disable it.
 */
XrlCmdError
XrlRLServiceNode::rl_service_0_1_enable_rl_service(
    // Input values,
    const bool&	enable)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    UNUSED(enable);

    // XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

XrlCmdError
XrlRLServiceNode::rl_service_0_1_start_rl_service()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    // XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

XrlCmdError
XrlRLServiceNode::rl_service_0_1_stop_rl_service()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

/**
 *  Add an interface
 *
 *  @param add an interface
 */
XrlCmdError
XrlRLServiceNode::rl_service_0_1_add_interface(
    // Input values,
    const IPv4&	address)
{
    UNUSED(address);

    //printf("%s\n",__PRETTY_FUNCTION__);
    string error_msg;

    //ADD INTERFACE EXEC HERE!
    //printf("XrlRLServiceNode::rl_service_0_1_add_interface()--called!\n");

    return XrlCmdError::OKAY();
}

/**
 *  Enable/disable the RLService trace log for all operations.
 *
 *  @param enable if true, then enable the trace log, otherwise disable it.
 */
XrlCmdError
XrlRLServiceNode::rl_service_0_1_enable_log_trace_all(
    // Input values,
    const bool&	enable)
{
    UNUSED(enable);

    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_server()
{
  return _rl_service_node.delete_dhcp();
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_start_service_dhcp_server_name(
							       //Input values,
							       const string &name)
{
  return _rl_service_node.start_dhcp(name);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_commit_service_dhcp_server_name(
							    //Input values,
							    const string &name)
{
  return _rl_service_node.commit_dhcp(name);
}


XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_server_name(
							    //Input values,
							    const string &name)
{
  return _rl_service_node.delete_dhcp(name);
}


XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_server_name_start_stop(
							   //Input values,
							   const string &name,
							   const IPv4 &start,
							   const IPv4 &stop)
{
  return _rl_service_node.set_service_dhcp_server_name_start_stop(name, start, stop);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_server_name_start_stop(
							      //Input values,
							      const string &name,
							      const IPv4 &start,
							      const IPv4 &stop)
{
  return _rl_service_node.delete_service_dhcp_server_name_start_stop(name, start, stop);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_server_name_exclude(
								    //Input values,
								    const string &name,
								    const IPv4 &exclude)
{
  return _rl_service_node.set_service_dhcp_server_name_exclude(name, exclude);
}
  
XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_server_name_exclude(
								    //Input values,
								    const string &name,
								    const IPv4 &exclude)
{
  return _rl_service_node.delete_service_dhcp_server_name_exclude(name, exclude);
}
  
XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_server_name_static_mac(
								    //Input values,
								    const string &name,
								    const IPv4 &static_addr,
								    const Mac &mac)
{
  return _rl_service_node.set_service_dhcp_server_name_static_mac(name, static_addr, mac);
}
  
XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_server_name_static_mac(
								    //Input values,
								    const string &name,
								    const IPv4 &static_addr,
								    const Mac &mac)
{
  return _rl_service_node.delete_service_dhcp_server_name_static_mac(name, static_addr, mac);
}
  

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_server_name_default(
							   //Input values,
							   const string &name,
							   const IPv4 &default_router)
{
    return _rl_service_node.set_service_dhcp_server_name_default(name, default_router);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_server_name_default(
									//Input values,
									const string &name,
									const IPv4 &default_router)
{
    return _rl_service_node.delete_service_dhcp_server_name_default(name, default_router);
}


XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_server_name_network_mask(
							   //Input values,
							   const string &name,
							   const uint32_t &network_mask)
{
    return _rl_service_node.set_dhcp_mask(name, network_mask);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_server_name_network_mask(
									//Input values,
									const string &name,
									const uint32_t &network_mask)
{
  return _rl_service_node.delete_dhcp_mask(name, network_mask);
}


XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_server_name_dns(
							       //Input values,
							       const string &name,
							       const IPv4 &dns_server)
{
  return _rl_service_node.set_service_dhcp_server_name_dns(name, dns_server);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_server_name_dns(
								  //Input values,
								  const string &name,
								  const IPv4 &dns_server)
{
  return _rl_service_node.delete_service_dhcp_server_name_dns(name, dns_server);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_server_name_wins(
							       //Input values,
							       const string &name,
							       const IPv4 &wins_server)
{
  return _rl_service_node.set_service_dhcp_server_name_wins(name, wins_server);

}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_server_name_wins(
								  //Input values,
								  const string &name,
								  const IPv4 &wins_server)
{
    return _rl_service_node.delete_service_dhcp_server_name_wins(name, wins_server);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_server_name_lease(
							   //Input values,
							   const string &name,
							   const uint32_t &lease)
{
  return _rl_service_node.set_dhcp_lease(name, lease);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_server_name_lease(
							      //Input values,
							      const string &name,
							      const uint32_t &lease)
{
  return _rl_service_node.delete_dhcp_lease(name, lease);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_server_name_interface(
							   //Input values,
							   const string &name,
							   const string &interface)
{
    return _rl_service_node.set_dhcp_interface(name, interface);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_server_name_interface(
							      //Input values,
							      const string &name,
							      const string &interface)
{
    return _rl_service_node.delete_dhcp_interface(name, interface);
}



XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_server_name_domain_name(
							   //Input values,
							   const string &name,
							   const string &domain_name)
{
    return _rl_service_node.set_dhcp_domain_name(name, domain_name);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_server_name_domain_name(
							      //Input values,
							      const string &name,
							      const string &domain_name)
{
    return _rl_service_node.delete_dhcp_domain_name(name, domain_name);
}


XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_server_name_authoritative(
								      //Input values,
								      const string &name,
								      const string &enable)
{
  return _rl_service_node.set_dhcp_auth(name, enable);
}

XrlCmdError
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_relay_interface(
								     const string &interface)
{
  return _rl_service_node.delete_dhcp_relay(interface);
}

XrlCmdError
XrlRLServiceNode::rl_service_0_1_start_service_dhcp_relay(
							  const string &interface)
{
  return _rl_service_node.start_dhcp_relay(interface);
}

XrlCmdError
XrlRLServiceNode::rl_service_0_1_commit_service_dhcp_relay(
							   const string &interface)
{
  return _rl_service_node.commit_dhcp_relay(interface);
}


XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_relay_interface_server(
							  //Input values,
							  const string &interface,
							  const IPv4 &server)
{
  return _rl_service_node.set_dhcp_relay_server(interface, server);
}
  
XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_relay_interface_server(
							     const string &interface,
							     const IPv4 &server)
{
  return _rl_service_node.delete_dhcp_relay_server(interface, server);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_relay_interface_relay_options_port(
								      const string &interface,
								      const uint32_t &port)
{
  return _rl_service_node.set_dhcp_relay_port(interface, port);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_relay_interface_relay_options_port(
									 const string &interface,
									 const uint32_t &port)
{
  return _rl_service_node.delete_dhcp_relay_port(interface, port);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_relay_interface_relay_options_max_size(
									  const string &interface,
									  const uint32_t &max_size)
{
  return _rl_service_node.set_dhcp_relay_max_size(interface, max_size);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_relay_interface_relay_options_max_size(
									     const string &interface,
									     const uint32_t &max_size)
{
  return _rl_service_node.delete_dhcp_relay_max_size(interface, max_size);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_relay_interface_relay_options_hop_count(
									   const string &interface,
									   const uint32_t &hop_count)
{


  return _rl_service_node.set_dhcp_relay_hop_count(interface, hop_count);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_relay_interface_relay_options_hop_count(
									      const string &interface,
									      const uint32_t &hop_count)
{
  return _rl_service_node.delete_dhcp_relay_hop_count(interface, hop_count);
}


XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_dhcp_relay_interface_relay_options_relay_agents_packets(
										      const string &interface,
										      const string &relay_agents_packets)
{
  return _rl_service_node.set_dhcp_relay_agents_packets(interface, relay_agents_packets);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_dhcp_relay_interface_relay_options_relay_agents_packets(
											 const string &interface,
											 const string &relay_agents_packets)
{
  return _rl_service_node.delete_dhcp_relay_agents_packets(interface, relay_agents_packets);
}







XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_http(
					       //Input values,
					       const uint32_t &port)
{
    UNUSED(port);
    return XrlCmdError::COMMAND_FAILED();
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_http(
					       //Input values,
					       const uint32_t &port)
{
    UNUSED(port);
    return XrlCmdError::COMMAND_FAILED();
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_ssh(
					  //Input values,
					  const uint32_t &port,
					  const string &protocol_version)
{
    return _rl_service_node.set_ssh(port, protocol_version);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_ssh(
						  //Input values,
						  const uint32_t &port,
						  const string &protocol_version)
{
    return _rl_service_node.delete_ssh(port, protocol_version);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_telnet(
						  //Input values,
						  const uint32_t &port)
{
    return _rl_service_node.set_telnet(port);
}
XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_telnet(
						    //Input values,
						    const uint32_t &port)
{
    return _rl_service_node.delete_telnet(port);
}


XrlCmdError 
XrlRLServiceNode::rl_service_0_1_start_service_nat_rule(
						  //Input values,
						  const uint32_t &rule)
{
  return _rl_service_node.start_transaction(rule);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_commit_service_nat_rule(
						    //Input values,
						    const uint32_t &rule)
{
  return _rl_service_node.end_transaction(rule);
}

XrlCmdError
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule(
							 //Input values,
							 const uint32_t &rule)
{
  return _rl_service_node.delete_rule(rule);
}


  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_type(
						       //Input values,
						       const string &type)
{
  return _rl_service_node.set_nat_type(type);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_type(
						       //Input values,
						       const string &type)
{
  return _rl_service_node.delete_nat_type(type);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_translation_type(
						       //Input values,
						       const string &translation_type)
{
  return _rl_service_node.set_nat_translation(translation_type);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_translation_type(
						       //Input values,
						       const string &translation_type)
{
  return _rl_service_node.delete_nat_translation(translation_type);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_inbound_interface(
						       //Input values,
						       const string &interface)
{
  return _rl_service_node.set_nat_inbound_interface(interface);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_inbound_interface(
						       //Input values,
						       const string &interface)
{
  return _rl_service_node.delete_nat_inbound_interface(interface);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_outbound_interface(
						       //Input values,
						       const string &interface)
{
  return _rl_service_node.set_nat_outbound_interface(interface);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_outbound_interface(
						       //Input values,
						       const string &interface)
{
  return _rl_service_node.delete_nat_outbound_interface(interface);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_protocol(
						       //Input values,
						       const string &protocol)
{
  return _rl_service_node.set_nat_protocol(protocol);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_protocol(
						       //Input values,
						       const string &protocol)
{
  return _rl_service_node.delete_nat_protocol(protocol);
}


  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_source_address_address(
									 //Input values,
									 const IPv4 &address)
{
  return _rl_service_node.set_nat_source_address(address);
}
  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_source_address_address(
									    //Input values,
									    const IPv4 &address)
{
    return _rl_service_node.delete_nat_source_address(address);
}

 
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_source_address_network(
									 //Input values,
									 const IPv4Net &network)
{
  return _rl_service_node.set_nat_source_address(network);
}

  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_source_address_network(
									    //Input values,
									    const IPv4Net &network)
{
  return _rl_service_node.delete_nat_source_address(network);
}

  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_source_port_port(
								   //Input values,
								   const uint32_t &port)
{
  return _rl_service_node.set_nat_source_port(port);
}

  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_source_port_port(
								      //Input values,
								      const uint32_t &port)
{
  return _rl_service_node.delete_nat_source_port(port);
}

  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_source_port_name(
								   //Input values,
								   const string &name)
{
  return _rl_service_node.set_nat_source_port(name);
}

  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_source_port_name(
								      //Input values,
								      const string &name)
{
  return _rl_service_node.delete_nat_source_port(name);
}

  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_source_port_range(
								   //Input values,
								    const uint32_t &start,
								    const uint32_t &stop)
{
  return _rl_service_node.set_nat_source_port(start, stop);
}

  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_source_port_range(
								      //Input values,
								    const uint32_t &start,
								    const uint32_t &stop)
{
  return _rl_service_node.delete_nat_source_port(start, stop);
}

  
 
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_destination_address_address(
									      //Input values,
									      const IPv4 &address)
{
  return _rl_service_node.set_nat_destination_address(address);}

  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_destination_address_address(
										 //Input values,
										 const IPv4 &address)
{
  return _rl_service_node.delete_nat_destination_address(address);
}
  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_destination_address_network(
									      //Input values,
									      const IPv4Net &network)
{
  return _rl_service_node.set_nat_destination_address(network);
}

  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_destination_address_network(
										 //Input values,
										 const IPv4Net &network)
{
  return _rl_service_node.delete_nat_destination_address(network);
}
  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_destination_port_port(
								   //Input values,
								   const uint32_t &port)
{
  return _rl_service_node.set_nat_destination_port(port);
}
  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_destination_port_port(
								      //Input values,
								      const uint32_t &port)
{
  return _rl_service_node.delete_nat_destination_port(port);
}
  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_destination_port_name(
								   //Input values,
								   const string &name)
{
  return _rl_service_node.set_nat_destination_port(name);
}
  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_destination_port_name(
								      //Input values,
								      const string &name)
{
  return _rl_service_node.delete_nat_destination_port(name);
}
  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_destination_port_range(
								   //Input values,
								    const uint32_t &start,
								    const uint32_t &stop)
{
  return _rl_service_node.set_nat_destination_port(start, stop);
}
  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_destination_port_range(
								      //Input values,
								    const uint32_t &start,
								    const uint32_t &stop)
{
  return _rl_service_node.delete_nat_destination_port(start, stop);
}
  
  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_inside_address_address(
						       //Input values,
						       const IPv4 &address)
{
  return _rl_service_node.set_nat_inside_address(address);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_inside_address_address(
						       //Input values,
										const IPv4 &address)
{
  return _rl_service_node.delete_nat_inside_address(address);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_inside_address_network(
						       //Input values,
						       const IPv4Net &network)
{
  return _rl_service_node.set_nat_inside_address(network);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_inside_address_network(
						       //Input values,
						       const IPv4Net &network)
{
  return _rl_service_node.delete_nat_inside_address(network);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_outside_address_address(
						       //Input values,
						       const IPv4 &address)
{
  return _rl_service_node.set_nat_outside_address(address);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_outside_address_address(
						       //Input values,
						       const IPv4 &address)
{
  return _rl_service_node.delete_nat_outside_address(address);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_outside_address_network(
						       //Input values,
						       const IPv4Net &network)
{
  return _rl_service_node.set_nat_outside_address(network);
}

  XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_outside_address_network(
						       //Input values,
						       const IPv4Net &network)
{
  return _rl_service_node.delete_nat_outside_address(network);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_set_service_nat_rule_outside_address_range(
								    //Input values,
								    const IPv4 &start,
								    const IPv4 &stop)
{
  return _rl_service_node.set_nat_outside_address(start, stop);
}

XrlCmdError 
XrlRLServiceNode::rl_service_0_1_delete_service_nat_rule_outside_address_range(
								       //Input values,
								       const IPv4 &start,
								       const IPv4 &stop)
{
  return _rl_service_node.delete_nat_outside_address(start, stop);
}


XrlCmdError 
XrlRLServiceNode::rl_service_0_1_get_nat_rule_list(
					   //output values,
						   const uint32_t &filter, 
						   XrlAtomList &areas)
{
  return _rl_service_node.retrieve_rules(filter, areas);
}
