// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
// vim:set sts=4 ts=8:

// Copyright (c) 2001-2005 International Computer Science Institute
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software")
// to deal in the Software without restriction, subject to the conditions
// listed in the XORP LICENSE file. These conditions include: you must
// preserve this copyright notice, and you cannot mention the copyright
// holders in advertising related to the Software without their permission.
// The Software is provided WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED. This
// notice is a summary of the XORP LICENSE file; the license in that file is
// legally binding.

// #define DEBUG_LOGGING
// #define DEBUG_PRINT_FUNCTION_NAME

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string>
#include <set>
#include <list>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "rl_service/rl_service_module.h"

#include "libxorp/debug.h"
#include "libxorp/xlog.h"

#include "libxorp/status_codes.h"
#include "libxorp/eventloop_factory.hh"
#include "libxipc/xrl_std_router.hh"
#include "libxorp/tlv.hh"

#include "xrl/interfaces/rl_service_xif.hh"

using namespace std;

/**
 * Get the list of neighbours
 */
class GetNatList {
public:
    typedef enum {kNone = 0, kStatic = 1, kDynamic = 2} Filter;

public:
    GetNatList(XrlStdRouter& xrl_router, Filter filter)
	: _xrl_router(xrl_router),
	  _done(false), _fail(false), _filter(filter)
    {
    }

    void start() {
	//	printf("GetNatList::start():\n");
	XrlRlServiceV0p1Client rl_service(&_xrl_router);
	rl_service.send_get_nat_rule_list("rl_service", _filter,
					   callback(this, &GetNatList::response));
    }

    bool busy() {
	return !_done;
    }

    bool fail() {
	return _fail;
    }

    list<string>& get() {
	return _nlist;
    }

private:
    void response(const XrlError& error, const XrlAtomList *atomlist) {
	//	printf("GetNatList::response\n");
	_done = true;
	if (XrlError::OKAY() != error) {
	    XLOG_WARNING("Attempt to get nat list failed");
	    _fail = true;
	    return;
	}
	const size_t size = atomlist->size();
	for (size_t i = 0; i < size; i++) {
	    string foo = atomlist->get(i).str().c_str();
	    printf("%s\n", foo.c_str());
	    _nlist.push_back(atomlist->get(i).str());
	}
	//	printf("GetNatList::response.... Leaving....\n");
    }
private:
    XrlStdRouter &_xrl_router;
    bool _done;
    bool _fail;
    Filter _filter;

    list<string> _nlist;
};


/**
 * The base class that needs to be implemented by all print routines.
 */ 

class Output {
public:
    virtual ~Output()
    {}

    virtual bool begin() {
	//	printf("Rule\n");
	return true;
    }

    void print_first_line(const uint32_t& number) {
	printf("%d", number);
	printf("\n");
    }

    virtual bool print(const uint32_t& number) {
	print_first_line(number);
	return true;
    }
};

int
usage(const char *myname)
{
    fprintf(stderr, "usage: %s\n", myname);

    return -1;
}

int 
main(int argc, char **argv)
{
    UNUSED(argc);
    XorpUnexpectedHandler x(xorp_unexpected_handler);

    //
    // Get option of static or dynamic to filter the rule
    //
    GetNatList::Filter filter = GetNatList::kNone;
    if (argc == 2) {
	if (string(argv[1]) == "static") {
	    filter = GetNatList::kStatic;
	}
	else if (string(argv[1]) == "dynamic") {
	    filter = GetNatList::kDynamic;
	}
	else {
	    filter = GetNatList::kNone;
	}
    }
    
    //
    // Initialize and start xlog
    //
    xlog_init(argv[0], NULL);
    xlog_set_verbose(XLOG_VERBOSE_LOW);		// Least verbose messages
    // XXX: verbosity of the error messages temporary increased
    xlog_level_set_verbose(XLOG_LEVEL_ERROR, XLOG_VERBOSE_HIGH);
    xlog_add_default_output();
    xlog_start();

    Output *output = new Output();

    try {
	EventLoop& eventloop = *EventLoopFactory::instance().create(eventloop_st);
	XrlStdRouter xrl_router(eventloop, "print_nat_rules");

	debug_msg("Waiting for router");
	xrl_router.finalize();
	wait_until_xrl_router_is_ready(eventloop, xrl_router);
	debug_msg("\n");

	GetNatList get_nat_list(xrl_router, filter);
	get_nat_list.start();
	while(get_nat_list.busy())
	    eventloop.run();

	if (get_nat_list.fail()) {
	    XLOG_ERROR("Failed to get nat list");
	    return -1;
	}

	if (!output->begin())
	    return -1;
	delete output;

    } catch (...) {
	xorp_catch_standard_exceptions();
    }

    xlog_stop();
    xlog_exit();

    return 0;
}
