/*
 * Module: rl_service_node.hh
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Michael Larson
 * Date: 2005
 * Description:
 *
 * **** End License ****
 *
 */
#ifndef __RL_SERVICE_RL_SYSTEM_NODE_HH__
#define __RL_SERVICE_RL_SYSTEM_NODE_HH__

#include <map>
#include <set>
#include <string>
#include <librl_common/rl_nat.hh>
#include <libxipc/xrl_error.hh>
#include <libxipc/xrl_atom_list.hh>
#include <libxorp/mac.hh>
#include <libxorp/ipv4.hh>
#include <libxorp/ipv4net.hh>

/**
 *
 *
 **/
class DHCPRelayData
{
public:
  typedef set<IPv4> ServerColl;
  typedef set<IPv4>::iterator ServerIter;

  DHCPRelayData() :
    _port(67),
    _max_size(576),
    _hop_count(10),
    _agents_packets("forward") {}

  ServerColl _server_coll;
  uint32_t _port;
  uint32_t _max_size;
  uint32_t _hop_count;
  string _agents_packets;
};

/**
 *
 *
 **/
class DHCPData
{
public:
  typedef map<IPv4, IPv4> AddrRange;
  typedef map<IPv4, IPv4>::iterator AddrIter;
  typedef vector<IPv4> IPColl;
  typedef vector<IPv4>::iterator IPIter;
  typedef map<IPv4, Mac> StaticMap;
  typedef map<IPv4, Mac>::iterator StaticIter;

  DHCPData() : _mask_length(0), _default_router_valid(false), _lease(0), _authoritative(false) {} 

  bool
  commit(FILE *wr, const string &name);

private:
  AddrRange compute_addr_range();

public:
  AddrRange _addr_range;
  IPColl _addr_exclude;
  StaticMap _static_map;
  uint32_t _mask_length;
  IPColl _dns_server;
  IPv4 _default_router;
  bool _default_router_valid;
  IPColl _wins_server;
  uint32_t _lease;
  string _interface;
  string _domain_name;
  bool _authoritative;
};

/**
 *
 *
 **/

/**
 *
 *
 **/
class RLServiceNode
{
public:
  typedef map<uint32_t, NATData> NATColl;
  typedef map<uint32_t, NATData>::iterator NATIter;

  typedef map<string, DHCPData> DHCPColl;
  typedef map<string, DHCPData>::iterator DHCPIter;

  typedef map<string, DHCPRelayData> DHCPRelayColl;
  typedef map<string, DHCPRelayData>::iterator DHCPRelayIter;

public:
  RLServiceNode();
  ~RLServiceNode();

  XrlCmdError
  start_dhcp(const string &name);

  XrlCmdError
  commit_dhcp(const string &name);



  XrlCmdError
  set_service_dhcp_server_name_start_stop(const string &name,
					 const IPv4 &start,
					 const IPv4 &stop);

  XrlCmdError
  delete_service_dhcp_server_name_start_stop(const string &name,
					    const IPv4 &start,
					    const IPv4 &stop);

  XrlCmdError
  set_service_dhcp_server_name_exclude(const string &name,
				       const IPv4 &exclude);

  XrlCmdError
  delete_service_dhcp_server_name_exclude(const string &name,
					  const IPv4 &exclude);

  XrlCmdError
  set_service_dhcp_server_name_static_mac(const string &name,
				       const IPv4 &static_addr,
					  const Mac &mac);

  XrlCmdError
  delete_service_dhcp_server_name_static_mac(const string &name,
					  const IPv4 &static_addr,
					     const Mac &mac);

  XrlCmdError
  delete_dhcp();

  XrlCmdError
  delete_dhcp(const string &name);

  XrlCmdError
  set_service_dhcp_server_name_default(const string &name,
				      const IPv4 &default_router);

  XrlCmdError
  delete_service_dhcp_server_name_default(const string &name,
					 const IPv4 &default_router);

  XrlCmdError
  set_dhcp_mask(const string &name,
		const uint32_t &mask);

  XrlCmdError
  delete_dhcp_mask(const string &name,
		   const uint32_t &mask);

  XrlCmdError
  set_service_dhcp_server_name_dns(const string &name,
				  const IPv4 &dns_server);

  XrlCmdError
  delete_service_dhcp_server_name_dns(const string &name,
				     const IPv4 &dns_server);

  XrlCmdError
  set_service_dhcp_server_name_wins(const string &name,
				   const IPv4 &wins_server);

  XrlCmdError
  delete_service_dhcp_server_name_wins(const string &name,
				      const IPv4 &wins_server);

  XrlCmdError
  set_dhcp_lease(const string &name,
		 const uint32_t &lease);

  XrlCmdError
  delete_dhcp_lease(const string &name,
		    const uint32_t &lease);
  
  XrlCmdError
  set_dhcp_interface(const string &name,
		     const string &interface);

  XrlCmdError
  delete_dhcp_interface(const string &name,
			const string &interface);

  XrlCmdError
  set_dhcp_domain_name(const string &name,
		     const string &domain_name);

  XrlCmdError
  delete_dhcp_domain_name(const string &name,
			const string &domain_name);

  XrlCmdError
  set_dhcp_auth(const string &name, 
		const string &enable);

  XrlCmdError
  delete_dhcp_relay(const string &interface);

  XrlCmdError
  start_dhcp_relay(const string &interface);

  XrlCmdError
  commit_dhcp_relay(const string &interface);

  XrlCmdError
  set_dhcp_relay_server(const string &interface, const IPv4 &server);

  XrlCmdError
  delete_dhcp_relay_server(const string &interface, const IPv4 &server);

  XrlCmdError
  set_dhcp_relay_port(const string &interface, const uint32_t &port);

  XrlCmdError
  delete_dhcp_relay_port(const string &interface, const uint32_t &port);

  XrlCmdError
  set_dhcp_relay_max_size(const string &interface, const uint32_t &max_size);

  XrlCmdError
  delete_dhcp_relay_max_size(const string &interface, const uint32_t &max_size);

  XrlCmdError
  set_dhcp_relay_hop_count(const string &interface, const uint32_t &hop_count);

  XrlCmdError
  delete_dhcp_relay_hop_count(const string &interface, const uint32_t &hop_count);
  
  XrlCmdError
  set_dhcp_relay_agents_packets(const string &interface, const string &relay_agents_packets);

  XrlCmdError
  delete_dhcp_relay_agents_packets(const string &interface, const string &relay_agents_packets);

  XrlCmdError
  set_ssh(const uint32_t &port,
	  const string &version);

  XrlCmdError
  delete_ssh(const uint32_t &port,
	     const string &version);

  XrlCmdError
  set_telnet(const uint32_t &port);

  XrlCmdError
  delete_telnet(const uint32_t &port);

  XrlCmdError
  set_xgdaemon();

  XrlCmdError
  delete_xgdaemon();


  XrlCmdError
  start_transaction(const uint32_t &rule);

  XrlCmdError
  end_transaction(const uint32_t &rule);

  XrlCmdError
  delete_rule(const uint32_t &rule);

  XrlCmdError
  set_nat_type(const string &type);

  XrlCmdError
  delete_nat_type(const string &type);

  XrlCmdError
  set_nat_translation(const string &translation_type);

  XrlCmdError
  delete_nat_translation(const string &translation_type);

  XrlCmdError
  set_nat_inbound_interface(const string &interface);

  XrlCmdError
  delete_nat_inbound_interface(const string &interface);

  XrlCmdError
  set_nat_outbound_interface(const string &interface);

  XrlCmdError
  delete_nat_outbound_interface(const string &interface);

  XrlCmdError
  set_nat_input_interface(const string &interface);

  XrlCmdError
  delete_nat_input_interface(const string &interface);

  XrlCmdError
  set_nat_protocol(const string &protocol);

  XrlCmdError
  delete_nat_protocol(const string &protocol);

  XrlCmdError
  set_nat_source_address(const IPv4 &address);

  XrlCmdError
  delete_nat_source_address(const IPv4 &address);

  XrlCmdError
  set_nat_source_address(const IPv4Net &network);

  XrlCmdError
  delete_nat_source_address(const IPv4Net &network);

  XrlCmdError
  set_nat_source_port(const uint32_t &port);

  XrlCmdError
  delete_nat_source_port(const uint32_t &port);

  XrlCmdError
  set_nat_source_port(const string &name);

  XrlCmdError
  delete_nat_source_port(const string &name);

  XrlCmdError
  set_nat_source_port(const uint32_t &start, const uint32_t &stop);

  XrlCmdError
  delete_nat_source_port(const uint32_t &start, const uint32_t &stop);

  XrlCmdError
  set_nat_destination_address(const IPv4 &address);

  XrlCmdError
  delete_nat_destination_address(const IPv4 &address);

  XrlCmdError
  set_nat_destination_address(const IPv4Net &network);

  XrlCmdError
  delete_nat_destination_address(const IPv4Net &network);

  XrlCmdError
  set_nat_destination_port(const uint32_t &port);

  XrlCmdError
  delete_nat_destination_port(const uint32_t &port);

  XrlCmdError
  set_nat_destination_port(const string &name);

  XrlCmdError
  delete_nat_destination_port(const string &name);

  XrlCmdError
  set_nat_destination_port(const uint32_t &start, const uint32_t &stop);

  XrlCmdError
  delete_nat_destination_port(const uint32_t &start, const uint32_t &stop);

  XrlCmdError
  set_nat_inside_address(const IPv4 &address);

  XrlCmdError
  delete_nat_inside_address(const IPv4 &address);

  XrlCmdError
  set_nat_inside_address(const IPv4Net &network);

  XrlCmdError
  delete_nat_inside_address(const IPv4Net &network);

  XrlCmdError
  set_nat_outside_address(const IPv4 &address);

  XrlCmdError
  delete_nat_outside_address(const IPv4 &address);

  XrlCmdError
  set_nat_outside_address(const IPv4Net &network);

  XrlCmdError
  delete_nat_outside_address(const IPv4Net &network);

  XrlCmdError
  set_nat_outside_address(const IPv4 &start, const IPv4 &stop);

  XrlCmdError
  delete_nat_outside_address(const IPv4 &start, const IPv4 &stop);

  XrlCmdError
  reset();

  XrlCmdError
  retrieve_rules(const uint32_t &filter, XrlAtomList &rules);

private: //members
  NATIter 
  get_nat(const uint32_t &rule);

private: //variables
  string _service_dhcp_file;
  string _service_pidfile;
  string _service_nat_save_file;
  string _service_lease_file;
  NATColl _nat_coll;
  NATData _cur_nat_data;

  DHCPColl _dhcp_coll;
  DHCPData _cur_dhcp_data;

  DHCPRelayColl _dhcp_relay;
};

#endif //__RL_SERVICE_RL_SYSTEM_NODE_HH_

