/*
 * Module: rl_service_node.cc
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Michael Larson
 * Date: 2005
 * Description:
 *
 * **** End License ****
 *
 */
#include <stdio.h>
#include <iostream> 
#include <fstream>
#include <sstream>
#include <stdlib.h>
#include <string>
#include "rl_service_module.h"
#include "libxorp/xlog.h"
#include "libxipc/xrl_atom.hh"
#include "librl_common/rl_str_proc.hh"
#include "librl_common/rl_nat.hh"
#include "librl_common/rl_fileaccess.hh"
#include "librl_common/rl_command.hh"
#include "librl_common/rl_interface.hh"
#include "rl_service_node.hh"

using namespace std;

/**
 *
 **/
RLServiceNode::RLServiceNode() :
  _service_dhcp_file(SYSCONFDIR "/dhcpd.conf"),
  _service_pidfile("/var/run/dnsmasq-"),
  _service_nat_save_file("/etc/iptables_nat"),
  _service_lease_file("/var/log/dhcpd.leases")
{
  reset();
}

/**
 *
 **/
RLServiceNode::~RLServiceNode()
{
  //  reset();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::reset()
{
  string cmd;
  //copy over all default files
  
  //remove all filelocks, if any
  rl_fileaccess::init(_service_dhcp_file);

  // NAT is now handled by new CLI
#if 0
  //flush nat commands
  string cmd("iptables -t nat -F");
  rl_command::execute(cmd); //ignore return codes
#endif

  //DHCP server is now handled by new CLI
#if 0
  //remove any files
  /* remove dhcp files */
  unlink(_service_lease_file.c_str());
#endif

  //SSH server is now handled by the New CLI
#if 0
  //stop any processes
  /* stop all http */
  /* stop all ssh */
  cmd = string(SBINDIR "/sshd.init stop");
  if (rl_command::execute(cmd) == false) {
    return XrlCmdError::COMMAND_FAILED();
  }
#endif

  //Telnet server is now handled by the New CLI
#if 0
  /* stop all telnet */
  cmd = string(SBINDIR "/telnetd.init stop");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("rl_service failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
#endif

  //hup any processes

  //DHCP server is now handled by new CLI
#if 0
  /* do we need to hup the dhcp service? */
  //Now restart the dhcpd process
  cmd = string(SBINDIR "/dhcpd.init stop");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLServiceNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
#endif

  //DHCP relay is now handled by new CLI
#if 0
  cmd = string(SBINDIR "/dhcrelay.init stop");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLServiceNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
#endif

  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_dhcp()
{
  _dhcp_coll.erase(_dhcp_coll.begin(), _dhcp_coll.end());

  string cmd = string(SBINDIR "/dhcpd.init stop");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLServiceNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  unlink(_service_dhcp_file.c_str());

  unlink(_service_lease_file.c_str());
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_dhcp(const string &name)
{
  _dhcp_coll.erase(name);
  if (_dhcp_coll.empty() == true) {
    return delete_dhcp();
  }
  
  {
    FILE *fd_rd, *fd_wr;
    rl_fileaccess fa(_service_dhcp_file);
    bool success = fa.get(fd_rd, fd_wr);
    if (success == false) {
      XLOG_ERROR("RLServiceNode: failed to open file %s", _service_dhcp_file.c_str());
      return XrlCmdError::COMMAND_FAILED();
    }
    
    
    DHCPIter iter = _dhcp_coll.begin();
    while (iter != _dhcp_coll.end()) {
      bool err = iter->second.commit(fd_wr, iter->first);
      if (err == true) {
	XLOG_ERROR("RLServiceNode: failed to write to dhcp file %s", _service_dhcp_file.c_str());
	return XrlCmdError::COMMAND_FAILED();
      }
      ++iter;
    }
  }


  //calling explicitly as this seems to be excluded from the transaction.
  //Now restart the dhcpd process
  string cmd(SBINDIR "/dhcpd.init restart& ");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLServiceNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::start_dhcp(const string &name)
{
  //on second thought--let's munge all the names
  /*
  bool number_flag = true;
  for (unsigned int i = 0; i < name.length(); ++i) {
    if (isdigit(name[i]) == 0) {
      number_flag = false;
      break;
    }
  }
  
  if (number_flag == true) {
    return XrlCmdError::COMMAND_FAILED("DHCP group '" + name + "' cannot be configured as a number");
  }
  */
  DHCPIter iter =_dhcp_coll.find(name);
  if (iter == _dhcp_coll.end()) {
    _dhcp_coll.insert(pair<string, DHCPData>(name, DHCPData()));
    iter = _dhcp_coll.find(name);
  }
  _cur_dhcp_data = iter->second;
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::commit_dhcp(const string &name)
{
  UNUSED(name);

  DHCPIter iter = _dhcp_coll.find(name);
  if (iter == _dhcp_coll.end()) {
    return XrlCmdError::OKAY();
  }

  string address;
  if (rl_interface::is_configured(_cur_dhcp_data._interface, address) == false) {
      XLOG_ERROR("RLServiceNode: need interface %s configured before starting dhcp services", _cur_dhcp_data._interface.c_str());
    return XrlCmdError::COMMAND_FAILED("Need to configure interface\n");
  }
  UNUSED(address);

  //commit changes....
  iter->second = _cur_dhcp_data;
  _cur_dhcp_data = DHCPData();

  bool config_err = false;
  {
    string line;
    FILE *fd_rd, *fd_wr;
    rl_fileaccess fa(_service_dhcp_file);
    
    bool success = fa.get(fd_rd, fd_wr);
    if (success == false) {
      XLOG_ERROR("RLServiceNode: failed to open file %s", _service_dhcp_file.c_str());
      return XrlCmdError::COMMAND_FAILED();
    }
    
    iter = _dhcp_coll.begin();
    while (iter != _dhcp_coll.end()) {
      config_err = iter->second.commit(fd_wr, iter->first);
      if (config_err == true) {
	XLOG_WARNING("RLServiceNode: invalid dhcpd configuration stopping dhcp: %s", _service_dhcp_file.c_str());
	break;
      }
      ++iter;
    }
  }

  string cmd;
  if (config_err == true) {
    cmd = SBINDIR "/dhcpd.init stop ";
  }
  else {
    cmd = SBINDIR "/dhcpd.init restart ";
  }

  //Now restart or stop the dhcpd process
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLServiceNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  return XrlCmdError::OKAY();
}


/**
 *
 **/
XrlCmdError
RLServiceNode::set_service_dhcp_server_name_default(const string &name,
						     const IPv4 &default_router)
{
  UNUSED(name);

  _cur_dhcp_data._default_router = default_router;
  _cur_dhcp_data._default_router_valid = true;
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_service_dhcp_server_name_default(const string &name,
							const IPv4 &default_router)
{
  _cur_dhcp_data._default_router = IPv4();
  _cur_dhcp_data._default_router_valid = false;
  return XrlCmdError::OKAY();
  UNUSED(default_router);
  UNUSED(name);
}

/**
 *
 **/
XrlCmdError
RLServiceNode::set_service_dhcp_server_name_start_stop(const string &name,
							const IPv4 &start,
							const IPv4 &stop)
{
  UNUSED(name);
  _cur_dhcp_data._addr_range.insert(pair<IPv4, IPv4>(start, stop));
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_service_dhcp_server_name_start_stop(const string &name,
							const IPv4 &start,
							const IPv4 &stop)
{
  UNUSED(name);
  UNUSED(stop);
  _cur_dhcp_data._addr_range.erase(start);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::set_service_dhcp_server_name_exclude(const string &name,
				       const IPv4 &exclude)
{
  UNUSED(name);
  _cur_dhcp_data._addr_exclude.push_back(exclude);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_service_dhcp_server_name_exclude(const string &name,
					  const IPv4 &exclude)
{
  UNUSED(name);
  DHCPData::IPIter iter = _cur_dhcp_data._addr_exclude.begin();
  while (iter != _cur_dhcp_data._addr_exclude.end()) {
    if (*iter == exclude) {
      _cur_dhcp_data._addr_exclude.erase(iter);
      return XrlCmdError::OKAY();
    }
    ++iter;
  }
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::set_service_dhcp_server_name_static_mac(const string &name,
				       const IPv4 &static_addr,
					  const Mac &mac)
{
  UNUSED(name);
  _cur_dhcp_data._addr_exclude.push_back(static_addr);
  _cur_dhcp_data._static_map.insert(pair<IPv4, Mac>(static_addr, mac));
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_service_dhcp_server_name_static_mac(const string &name,
					  const IPv4 &static_addr,
					     const Mac &mac)
{
  UNUSED(name);
  UNUSED(mac);
  _cur_dhcp_data._static_map.erase(static_addr);
  
  DHCPData::IPIter iter = _cur_dhcp_data._addr_exclude.begin();
  while (iter != _cur_dhcp_data._addr_exclude.end()) {
    if (*iter == static_addr) {
      _cur_dhcp_data._addr_exclude.erase(iter);
      return XrlCmdError::OKAY();      
    }
    ++iter;
  }
  return XrlCmdError::OKAY();
}



/**
 *
 **/
XrlCmdError
RLServiceNode::set_dhcp_mask(const string &name,
	      const uint32_t &mask)
{
  UNUSED(name);

  _cur_dhcp_data._mask_length = mask;
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_dhcp_mask(const string &name,
	      const uint32_t &mask)
{
  _cur_dhcp_data._mask_length = 0;
  return XrlCmdError::OKAY();
  UNUSED(name);
  UNUSED(mask);
}

/**
 *
 **/
XrlCmdError
RLServiceNode::set_service_dhcp_server_name_dns(const string &name,
						 const IPv4 &dns_server)
{
  _cur_dhcp_data._dns_server.push_back(dns_server);
  return XrlCmdError::OKAY();
  UNUSED(name);
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_service_dhcp_server_name_dns(const string &name,
						    const IPv4 &dns_server)
{
  DHCPData::IPIter iter = _cur_dhcp_data._dns_server.begin();
  while (iter != _cur_dhcp_data._dns_server.end()) {
    if (*iter == dns_server) {
      _cur_dhcp_data._dns_server.erase(iter);
      return XrlCmdError::OKAY();      
    }
    ++iter;
  }
  return XrlCmdError::OKAY();
  UNUSED(name);
}
 
/**
 *
 **/
XrlCmdError
RLServiceNode::set_service_dhcp_server_name_wins(const string &name,
						  const IPv4 &wins_server)
{
  _cur_dhcp_data._wins_server.push_back(wins_server);
  return XrlCmdError::OKAY();
  UNUSED(name);
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_service_dhcp_server_name_wins(const string &name,
						     const IPv4 &wins_server)
{
  DHCPData::IPIter iter = _cur_dhcp_data._wins_server.begin();
  while (iter != _cur_dhcp_data._wins_server.end()) {
    if (*iter == wins_server) {
      _cur_dhcp_data._wins_server.erase(iter);
      return XrlCmdError::OKAY();      
    }
    ++iter;
  }
  return XrlCmdError::OKAY();
  UNUSED(wins_server);
  UNUSED(name);
}

/**
 *
 **/
XrlCmdError
RLServiceNode::set_dhcp_lease(const string &name, const uint32_t &lease)
{
  _cur_dhcp_data._lease = lease;
  return XrlCmdError::OKAY();
  UNUSED(name);
}

XrlCmdError
RLServiceNode::delete_dhcp_lease(const string &name, const uint32_t &lease)
{
  _cur_dhcp_data._lease = 0;
  return XrlCmdError::OKAY();
  UNUSED(lease);
  UNUSED(name);
}

/**
 *
 **/
XrlCmdError
RLServiceNode::set_dhcp_interface(const string &name, const string &interface)
{
  _cur_dhcp_data._interface = interface;
  return XrlCmdError::OKAY();
  UNUSED(name);
}

XrlCmdError
RLServiceNode::delete_dhcp_interface(const string &name, const string &interface)
{
  _cur_dhcp_data._interface = string("");
  return XrlCmdError::OKAY();
  UNUSED(interface);
  UNUSED(name);
}

/**
 *
 **/
XrlCmdError
RLServiceNode::set_dhcp_domain_name(const string &name, const string &domain_name)
{
  _cur_dhcp_data._domain_name = domain_name;
  return XrlCmdError::OKAY();
  UNUSED(name);
}

XrlCmdError
RLServiceNode::delete_dhcp_domain_name(const string &name, const string &domain_name)
{
  _cur_dhcp_data._domain_name = string("");
  return XrlCmdError::OKAY();
  UNUSED(domain_name);
  UNUSED(name);
}

/**
 *
 **/ 
XrlCmdError
RLServiceNode::set_dhcp_auth(const string &name, 
	      const string &enable)
{
  UNUSED(name);
  string tmp = rl_utils::to_upper(enable);
  
  if (tmp == "ENABLE") {
    _cur_dhcp_data._authoritative = true;
  }
  else {
    _cur_dhcp_data._authoritative = false;
  }
  return XrlCmdError::OKAY();
}


XrlCmdError
RLServiceNode::delete_dhcp_relay(const string &interface)
{
  if (_dhcp_relay.find(interface) != _dhcp_relay.end()) {
    _dhcp_relay.erase(interface);
    
      string cmd = SBINDIR "/dhcrelay.init stop " + interface;
      if (rl_command::execute(cmd) == false) {
	XLOG_ERROR("rl_service failed to execute command: %s", cmd.c_str());
	return XrlCmdError::COMMAND_FAILED();
      }
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::start_dhcp_relay(const string &interface)
{
  if (_dhcp_relay.find(interface) == _dhcp_relay.end()) {
    _dhcp_relay.insert(pair<string, DHCPRelayData>(interface, DHCPRelayData()));
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::commit_dhcp_relay(const string &interface)
{
  //apply changes to the dhcp relay file and restart, see previous
  //incarnation

  /*
    INTERFACE=$2
    SERVER=$3
    COUNT=$4
    PORT=$5
    LENGTH=$6
    MOD=$7
  */
  string cmd;
  char buf[80];
  DHCPRelayIter iter = _dhcp_relay.find(interface);
  if (iter != _dhcp_relay.end()) {
    if (iter->second._server_coll.empty() == false) {
      string cmd(SBINDIR "/dhcrelay.init restart ");
      cmd += interface + " ";
      if (iter->second._hop_count > 0) {
	sprintf(buf, "%d", iter->second._hop_count);
	cmd += string(" ") + buf;
      } 
      
      if (iter->second._port > 0) {
	sprintf(buf, "%d", iter->second._port);
	cmd += string(" ") + buf;
      }
      
      if (iter->second._max_size > 0) {
	sprintf(buf, "%d", iter->second._max_size);
	cmd += string(" ") + buf;
      }
      
      if (iter->second._agents_packets.empty() == false) {
	cmd += " " + iter->second._agents_packets;
      }
      
      cmd += " '";
      DHCPRelayData::ServerIter s_iter = iter->second._server_coll.begin();
      while (s_iter != iter->second._server_coll.end()) {
	cmd += s_iter->str() + " ";
	++s_iter;
      }
      cmd += "'";
      
      if (rl_command::execute(cmd) == false) {
	XLOG_ERROR("rl_service failed to execute command: %s", cmd.c_str());
	return XrlCmdError::COMMAND_FAILED();
      }
    }
    else {
      cmd = SBINDIR "/dhcrelay.init stop " + interface;
      if (rl_command::execute(cmd) == false) {
	XLOG_ERROR("rl_service failed to execute command: %s", cmd.c_str());
	return XrlCmdError::COMMAND_FAILED();
      }
    }
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::set_dhcp_relay_server(const string &interface, const IPv4 &server)
{
  DHCPRelayIter iter = _dhcp_relay.find(interface);
  if (iter != _dhcp_relay.end()) {
    iter->second._server_coll.insert(server);
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_dhcp_relay_server(const string &interface, const IPv4 &server)
{
  UNUSED(server);
  DHCPRelayIter iter = _dhcp_relay.find(interface);
  if (iter != _dhcp_relay.end()) {
    iter->second._server_coll.erase(server);
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::set_dhcp_relay_port(const string &interface, const uint32_t &port)
{
  DHCPRelayIter iter = _dhcp_relay.find(interface);
  if (iter != _dhcp_relay.end()) {
    iter->second._port = port;
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_dhcp_relay_port(const string &interface, const uint32_t &port)
{
  UNUSED(port);
  DHCPRelayIter iter = _dhcp_relay.find(interface);
  if (iter != _dhcp_relay.end()) {
    iter->second._port = 67; //default
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::set_dhcp_relay_max_size(const string &interface, const uint32_t &max_size)
{
  DHCPRelayIter iter = _dhcp_relay.find(interface);
  if (iter != _dhcp_relay.end()) {
    iter->second._max_size = max_size;
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_dhcp_relay_max_size(const string &interface, const uint32_t &max_size)
{
  UNUSED(max_size);
  DHCPRelayIter iter = _dhcp_relay.find(interface);
  if (iter != _dhcp_relay.end()) {
    iter->second._max_size = 576; //default
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::set_dhcp_relay_hop_count(const string &interface, const uint32_t &hop_count)
{
  DHCPRelayIter iter = _dhcp_relay.find(interface);
  if (iter != _dhcp_relay.end()) {
    iter->second._hop_count = hop_count;
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_dhcp_relay_hop_count(const string &interface, const uint32_t &hop_count)
{
  UNUSED(hop_count);
  DHCPRelayIter iter = _dhcp_relay.find(interface);
  if (iter != _dhcp_relay.end()) {
    iter->second._hop_count = 10; //default
  }
  return XrlCmdError::OKAY();
}
  
/**
 *
 **/
XrlCmdError
RLServiceNode::set_dhcp_relay_agents_packets(const string &interface, const string &relay_agents_packets)
{
  DHCPRelayIter iter = _dhcp_relay.find(interface);
  if (iter != _dhcp_relay.end()) {
    iter->second._agents_packets = relay_agents_packets;
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_dhcp_relay_agents_packets(const string &interface, const string &relay_agents_packets)
{
  UNUSED(relay_agents_packets);
  DHCPRelayIter iter = _dhcp_relay.find(interface);
  if (iter != _dhcp_relay.end()) {
    iter->second._agents_packets = "-m forward";
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::set_ssh(const uint32_t &port,
			 const string &version)
{
    UNUSED(version);

    char buf[20];
    sprintf(buf, "%d", port);
    std::string cmd(string(SBINDIR "/sshd.init restart ") + version + " " + string(buf) + "&");
    if (rl_command::execute(cmd) == false) {
      /*
	XLOG_ERROR("rl_service failed to execute command: %s", cmd.c_str());
	return XrlCmdError::COMMAND_FAILED();
      */
    }
    return XrlCmdError::OKAY();
}


/**
 *
 **/
XrlCmdError
RLServiceNode::delete_ssh(const uint32_t &port,
			    const string &version)
{
    UNUSED(version);

    char buf[20];
    sprintf(buf, "%d", port);
    std::string cmd(string(SBINDIR "/sshd.init stop ") + version + " " + string(buf));
    if (rl_command::execute(cmd) == false) {
	XLOG_ERROR("rl_service failed to execute command: %s", cmd.c_str());
	return XrlCmdError::COMMAND_FAILED();
    }
    return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::set_telnet(const uint32_t &port)
{
  char buf[20];
  sprintf(buf, "%d", port);
  std::string cmd(string(SBINDIR "/telnetd.init restart ") + string(buf));
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("rl_service failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_telnet(const uint32_t &port)
{
  char buf[20];
  sprintf(buf, "%d", port);
  std::string cmd(string(SBINDIR "/telnetd.init stop ") + string(buf));
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("rl_service failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::set_xgdaemon()
{
  std::string cmd(SBINDIR "/xgdaemon.init start");

  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("rl_service failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_xgdaemon()
{
  std::string cmd(SBINDIR "/xgdaemon.init stop");

  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("rl_service failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::start_transaction(const uint32_t &rule)
{
  NATIter iter = get_nat(rule);
  if (iter == _nat_coll.end()) {
    _cur_nat_data = NATData();
  } 
  else {
    _cur_nat_data = iter->second;
  }
  _cur_nat_data._is_new = true;
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::end_transaction(const uint32_t &rule)
{
  bool rtn_err = false;

  //flush all rules, then iterate through and commit all rules again
  string cmd("iptables -t nat -F");
  rl_command::execute(cmd);
  NATColl tmp_nat_coll = _nat_coll;
  
  //now copy over the temporary nat data
  NATIter iter = tmp_nat_coll.find(rule);
  if (iter != tmp_nat_coll.end()) {
    iter->second = _cur_nat_data;
  }
  else {
    return XrlCmdError::COMMAND_FAILED();
  }

  iter = tmp_nat_coll.begin();
  while (iter != tmp_nat_coll.end()) {
    //Skip the new rules after first failure and only apply old rules.
    if (rtn_err == false || iter->second._is_new == false) {
      string rule;
      XrlCmdError error = iter->second.create_rule(rule);
      
      if (error != XrlCmdError::OKAY()) {
	XLOG_ERROR("rl_service error on command creation");
	return error;
      } 
      else if (rule.empty() == true) {
	XLOG_ERROR("rl_service cmd is empty");
	rtn_err = true;
      }
      
      //can we get away with a replace option???
      cmd = "iptables -t nat " + rule;
      if (rl_command::execute(cmd) == false) {
	XLOG_ERROR("rl_service failed to execute command: %s", cmd.c_str());
	rtn_err = true;
      } 
      else {
	iter->second._is_new = false;
      }
    }
    ++iter;
  }

  cmd = "iptables-save -t nat > " + _service_nat_save_file;
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("rl_service failed to execute command: %s", cmd.c_str());
    rtn_err = true;
  }

  //now finally perform the copy and transact over the changed collection
  //On error don't update collection. This should work as the transaction is
  //only on a per rule basis, which would be a rollback of a specific rule
  //failure.
  if (rtn_err == false) {
    _nat_coll.erase(_nat_coll.begin(), _nat_coll.end());
    _nat_coll = tmp_nat_coll;
  } 

  return rtn_err == true ? XrlCmdError::COMMAND_FAILED() : XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLServiceNode::delete_rule(const uint32_t &rule)
{
  NATIter iter = get_nat(rule);
  if (iter == _nat_coll.end()) {
    XLOG_ERROR("RLServiceNode: entry not found");
    return XrlCmdError::COMMAND_FAILED();
  }
  _nat_coll.erase(iter);

  //flush all rules, then iterate through and commit all rules again
  string cmd("iptables -t nat -F");
  rl_command::execute(cmd);

  iter = _nat_coll.begin();
  while (iter != _nat_coll.end()) {
    //Skip the new rules after first failure and only apply old rules.
    string rule;
    XrlCmdError error = iter->second.create_rule(rule);
    if (error != XrlCmdError::OKAY()) {
      XLOG_ERROR("rl_service error in applying NAT");
      return error;
    }
      
    //can we get away with a replace option???
    cmd = "iptables -t nat " + rule;
    if (rl_command::execute(cmd) == false) {
      XLOG_ERROR("rl_service failed to execute command: %s", cmd.c_str());
      return XrlCmdError::COMMAND_FAILED();
    }
    ++iter;
  }

  return XrlCmdError::OKAY();
}




XrlCmdError
RLServiceNode::set_nat_type(const string &type)
{
  _cur_nat_data._type = type;
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_type(const string &type)
{
  _cur_nat_data._type = string("");
  UNUSED(type);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::set_nat_translation(const string &translation_type)
{
  _cur_nat_data._trans_type = translation_type;
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_translation(const string &translation_type)
{
  _cur_nat_data._trans_type = string("");
  return XrlCmdError::OKAY();
  UNUSED(translation_type);
}

XrlCmdError
RLServiceNode::set_nat_inbound_interface(const string &interface)
{
  _cur_nat_data._in_interface = interface;
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_inbound_interface(const string &interface)
{
  _cur_nat_data._in_interface = string("");
  return XrlCmdError::OKAY();
  UNUSED(interface);
}

XrlCmdError
RLServiceNode::set_nat_outbound_interface(const string &interface)
{
  _cur_nat_data._out_interface = interface;
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_outbound_interface(const string &interface)
{
  _cur_nat_data._out_interface = string("");
  return XrlCmdError::OKAY();
  UNUSED(interface);
}

XrlCmdError
RLServiceNode::set_nat_input_interface(const string &interface)
{
  _cur_nat_data._in_interface = interface;
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_input_interface(const string &interface)
{
  _cur_nat_data._in_interface = interface;  
  return XrlCmdError::OKAY();
  UNUSED(_cur_nat_data);
}

XrlCmdError
RLServiceNode::set_nat_protocol(const string &protocol)
{
  _cur_nat_data._protocol = protocol;
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_protocol(const string &protocol)
{
  _cur_nat_data._protocol = string("");
  return XrlCmdError::OKAY();
  UNUSED(protocol);
}

XrlCmdError
RLServiceNode::set_nat_source_address(const IPv4 &address)
{
  _cur_nat_data._src_address = address;
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_source_address(const IPv4 &address)
{
  _cur_nat_data._src_address = IPv4();
  return XrlCmdError::OKAY();
  UNUSED(address);
}

XrlCmdError
RLServiceNode::set_nat_source_address(const IPv4Net &network)
{
  _cur_nat_data._src_network.set(network);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_source_address(const IPv4Net &network)
{
  _cur_nat_data._src_network.clear();
  return XrlCmdError::OKAY();
  UNUSED(network);
}

XrlCmdError
RLServiceNode::set_nat_source_port(const uint32_t &port)
{
  _cur_nat_data._src_ports.insert(port);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_source_port(const uint32_t &port)
{
  _cur_nat_data._src_ports.erase(port);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::set_nat_source_port(const string &name)
{
  _cur_nat_data._src_port_apps.insert(name);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_source_port(const string &name)
{
  _cur_nat_data._src_port_apps.erase(name);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::set_nat_source_port(const uint32_t &start, const uint32_t &stop)
{
  _cur_nat_data._src_port_start = start;
  _cur_nat_data._src_port_stop = stop;
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_source_port(const uint32_t &start, const uint32_t &stop)
{
  _cur_nat_data._src_port_start = 0;
  _cur_nat_data._src_port_stop = 0;
  return XrlCmdError::OKAY();
  UNUSED(start);
  UNUSED(stop);
}

XrlCmdError
RLServiceNode::set_nat_destination_address(const IPv4 &address)
{
  _cur_nat_data._dst_address = address;
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_destination_address(const IPv4 &address)
{
  _cur_nat_data._dst_address = IPv4();
  return XrlCmdError::OKAY();
  UNUSED(address);
}

XrlCmdError
RLServiceNode::set_nat_destination_address(const IPv4Net &network)
{
  _cur_nat_data._dst_network.set(network);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_destination_address(const IPv4Net &network)
{
  _cur_nat_data._dst_network.clear();
  return XrlCmdError::OKAY();
  UNUSED(network);
}

XrlCmdError
RLServiceNode::set_nat_destination_port(const uint32_t &port)
{
  _cur_nat_data._dst_ports.insert(port);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_destination_port(const uint32_t &port)
{
  _cur_nat_data._dst_ports.erase(port);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::set_nat_destination_port(const string &name)
{
  _cur_nat_data._dst_port_apps.insert(name);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_destination_port(const string &name)
{
  _cur_nat_data._dst_port_apps.erase(name);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::set_nat_destination_port(const uint32_t &start, const uint32_t &stop)
{
  _cur_nat_data._dst_port_start = start;
  _cur_nat_data._dst_port_stop = stop;
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_destination_port(const uint32_t &start, const uint32_t &stop)
{
  _cur_nat_data._dst_port_start = 0;
  _cur_nat_data._dst_port_stop = 0;
  return XrlCmdError::OKAY();
  UNUSED(start);
  UNUSED(stop);
}

XrlCmdError
RLServiceNode::set_nat_inside_address(const IPv4 &address)
{
  _cur_nat_data._in_address = address;
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_inside_address(const IPv4 &address)
{
  _cur_nat_data._in_address = IPv4();
  return XrlCmdError::OKAY();
  UNUSED(address);
}

XrlCmdError
RLServiceNode::set_nat_inside_address(const IPv4Net &network)
{
  _cur_nat_data._in_network.set(network);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_inside_address(const IPv4Net &network)
{
  _cur_nat_data._in_network.clear();
  return XrlCmdError::OKAY();
  UNUSED(network);
}

XrlCmdError
RLServiceNode::set_nat_outside_address(const IPv4 &address)
{
  _cur_nat_data._out_address = address;
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_outside_address(const IPv4 &address)
{
  _cur_nat_data._out_address = IPv4();
  return XrlCmdError::OKAY();
  UNUSED(address);
}

XrlCmdError
RLServiceNode::set_nat_outside_address(const IPv4Net &network)
{
  _cur_nat_data._out_network.set(network);
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_outside_address(const IPv4Net &network)
{
  _cur_nat_data._out_network.clear();
  return XrlCmdError::OKAY();
  UNUSED(network);
}

XrlCmdError
RLServiceNode::set_nat_outside_address(const IPv4 &start, const IPv4 &stop)
{
  _cur_nat_data._out_address_start = start;
  _cur_nat_data._out_address_stop = stop;
  return XrlCmdError::OKAY();
}

XrlCmdError
RLServiceNode::delete_nat_outside_address(const IPv4 &start, const IPv4 &stop)
{
  _cur_nat_data._out_address_start = IPv4();
  _cur_nat_data._out_address_stop = IPv4();
  return XrlCmdError::OKAY();
  UNUSED(start);
  UNUSED(stop);
}


/**
 *
 **/
RLServiceNode::NATIter 
RLServiceNode::get_nat(const uint32_t &rule)
{
  NATIter iter;

  iter = _nat_coll.find(rule);
  if (iter == _nat_coll.end()) {
    _nat_coll[rule] = NATData();
    pair<NATIter, bool> rtn_value = _nat_coll.insert(pair<uint32_t, NATData> (rule, NATData()));
    iter = rtn_value.first;
  }
  return iter;

}

/**
 *
 *
 **/
bool
DHCPData::commit(FILE *wr, const string &name)
{
  string line;

  if (_addr_range.empty() == true && _static_map.empty() == true) {
    return false;
  }

  line = "ddns-update-style ad-hoc;\n";
  fputs(line.c_str(), wr);

  line = "#" + name + "\n";
  fputs(line.c_str(), wr);

  line = "shared-network " + name + " {\n";
  fputs(line.c_str(), wr);

  AddrIter iter = _addr_range.begin();

  //first check interface address, then check first range address
  string addr;
  if (rl_interface::is_configured(_interface, addr) == false) {
    if (iter != _addr_range.end()) {
      addr = iter->first.str();
    }
  }

  if (addr.empty()) {
    return false;
  }

  IPv4 ip(addr.c_str());
  IPv4Net network(ip, _mask_length);
  IPv4Net mask(IPv4("255.255.255.255"), _mask_length);
  line = "subnet " + network.masked_addr().str() + " netmask " +  mask.masked_addr().str() + " {\n";
  fputs(line.c_str(), wr);

  if (iter != _addr_range.end()) {

    AddrRange computed_range_coll = compute_addr_range();
    iter = computed_range_coll.begin();
    while (iter != computed_range_coll.end()) {
      line = "range " + iter->first.str() + " " + iter->second.str() + ";\n";
      fputs(line.c_str(), wr);
      ++iter;
    }
  }

  if (_default_router_valid == true) {
    line = "option routers " + _default_router.str() + ";\n";
    fputs(line.c_str(), wr);
  }

  line = "option domain-name \"" + _domain_name + "\";\n";
  fputs(line.c_str(), wr);

  char buf[80];
  sprintf(buf, "%d", _lease);
  line = "default-lease-time " + string(buf) + ";\n";
  fputs(line.c_str(), wr);
  
  line = "option domain-name-servers ";
  IPIter ip_iter = _dns_server.begin();
  while (ip_iter != _dns_server.end()) {
    line += ip_iter->str() + ",";
    ++ip_iter;
  }
  line = line.substr(0, line.length()-1);
  line += ";\n";
  if (_dns_server.empty() == false) {
    fputs(line.c_str(), wr);
  }
  else {
    line = "ddns-updates off;\n";
    fputs(line.c_str(), wr);
  }

  line = "option netbios-name-servers ";
  ip_iter = _wins_server.begin();
  while (ip_iter != _wins_server.end()) {
    line += ip_iter->str() + ",";
    ++ip_iter;
  }
  line = line.substr(0, line.length()-1);
  line += ";\n";
  if (_wins_server.empty() == false) {
    fputs(line.c_str(), wr);
  }

  if (_authoritative) {
    line = "authoritative;\n";
  } 
  else {
    line = "not authoritative;\n";
  }
  fputs(line.c_str(), wr);

  line = "}\n"; //close subnet statement
  fputs(line.c_str(), wr);

  if (_static_map.empty() == false) {
    StaticIter iter_stat = _static_map.begin();
    while (iter_stat != _static_map.end()) {
      line = "host " + iter_stat->first.str() + " {\n";
      fputs(line.c_str(), wr);
      line = "  hardware ethernet " + iter_stat->second.str() + ";\n";
      fputs(line.c_str(), wr);
      line = "  fixed-address " + iter_stat->first.str() + ";\n";
      fputs(line.c_str(), wr);
      line = "}\n";
      fputs(line.c_str(), wr);
	
      ++iter_stat;
    }
  }

  line = "}\n"; //closed shared network statement
  fputs(line.c_str(), wr);
  return false;
}

/**
 *
 **/
DHCPData::AddrRange 
DHCPData::compute_addr_range()
{
  AddrRange computed_range;

  AddrIter iter = _addr_range.begin();
  while (iter != _addr_range.end()) {

    IPv4 start_addr = iter->first;
    IPv4 end_addr = iter->second;
    
    IPIter ex_iter = _addr_exclude.begin();
    while (ex_iter != _addr_exclude.end()) {
      if (start_addr == *ex_iter) {
	//skip this address as the exclusion is equal
	++start_addr;
      }
      else if (start_addr < *ex_iter && end_addr >= *ex_iter) {
	IPv4 addr = *ex_iter;
	computed_range.insert(pair<IPv4, IPv4>(start_addr, --addr));
	addr = *ex_iter;
	start_addr = ++addr;
      }
      ++ex_iter;
    }
    if (start_addr <= end_addr) {
      computed_range.insert(pair<IPv4, IPv4>(start_addr, end_addr));
    }
    ++iter;
  }

  return computed_range;
}

/**
 *
 **/
XrlCmdError
RLServiceNode::retrieve_rules(const uint32_t &filter, XrlAtomList &areas)
{
  string dl;

  //first pre-process the nat rules so that we can index
  //correctly into the vyatta defined rule set
  vector<string> ipt_pre_coll, ipt_post_coll;
  char buf[2048];
  string cmd = "iptables -t nat -L -v";
  bool pre_route = false;
  FILE *f = popen(cmd.c_str(), "r");
  if (f) {
    while(fgets(buf, 2047, f) != NULL) { 
      string line(buf);
      if (line.find("PREROUTING") != string::npos) {
	pre_route = true;
      }
      else if (line.find("POSTROUTING") != string::npos) {
	pre_route = false;
      }
      
      if (line.find("SNAT") != string::npos ||
	  line.find("DNAT") != string::npos ||
	  line.find("MASQUERADE") != string::npos) {
	StrProc proc_str(line, " ");
	if (proc_str.get(1).empty() == false) {
	  if (pre_route) {
	    ipt_pre_coll.push_back("pkts=" + proc_str.get(0) + ";bytes=" + proc_str.get(1));
	  }
	  else {
	    ipt_post_coll.push_back("pkts=" + proc_str.get(0) + ";bytes=" + proc_str.get(1));
	  }
	}
      }
    } 
    if (pclose(f) != 0) {
      return XrlCmdError::COMMAND_FAILED();
    }
  }

  uint32_t ct_pre = 0, ct_post = 0;

  NATIter iter = _nat_coll.begin();
  while (iter != _nat_coll.end()) {
    string line;
    char buf[80];

    sprintf(buf, "%d", iter->first);
    line += "rule_num=" + string(buf) + ";";

    //now process the ipt data here
    if (iter->second._type == "destination") {
      if (ct_pre < ipt_pre_coll.size()) {
	line += ipt_pre_coll[ct_pre] + ";";
      }
      ++ct_pre;
    }
    else { //source
      if (ct_post < ipt_post_coll.size()) {
	line += ipt_post_coll[ct_post] + ";";
      }
      ++ct_post;
    }

    line += "type=" + iter->second._type + ";";
    line += "trans_type=" + iter->second._trans_type + ";";
    line += "in_interface=" + iter->second._in_interface + ";";
    line += "out_interface=" + iter->second._out_interface + ";";
    line += "src_addr=" + iter->second._src_address.str() + ";";
    line += "src_network=" + iter->second._src_network.str() + ";";
    
    line += "src_ports=";
    NATData::PortIter port_iter = iter->second._src_ports.begin();
    while (port_iter != iter->second._src_ports.end()) {
      if (port_iter != iter->second._src_ports.begin()) line += ",";
      sprintf(buf, "%d", *port_iter);
      line += string(buf);
      ++port_iter;
    }
    line += ";";

    line += "src_ports_apps=";
    NATData::AppIter app_iter = iter->second._src_port_apps.begin();
    while (app_iter != iter->second._src_port_apps.end()) {
      if (app_iter != iter->second._src_port_apps.begin()) line += ",";
      line += *app_iter;
      ++app_iter;
    }
    line += ";";

    line += "src_port_start=";
    if (iter->second._src_port_start > 0) {
      sprintf(buf, "%d", iter->second._src_port_start);
      line += string(buf);
    }
    line += ";";

    line += "src_port_stop=";
    if (iter->second._src_port_stop > 0) {
      sprintf(buf, "%d", iter->second._src_port_stop);
      line += string(buf);
    }
    line += ";";

    line += "dst_addr=" + iter->second._dst_address.str() + ";";
    line += "dst_network=" + iter->second._dst_network.str() + ";";
    
    line += "dst_ports=";
    port_iter = iter->second._dst_ports.begin();
    while (port_iter != iter->second._dst_ports.end()) {
      if (port_iter != iter->second._dst_ports.begin()) line += ",";
      sprintf(buf, "%d", *port_iter);
      line += string(buf);
      ++port_iter;
    }
    line += ";";

    line += "dst_ports_apps=";
    app_iter = iter->second._dst_port_apps.begin();
    while (app_iter != iter->second._dst_port_apps.end()) {
      if (app_iter != iter->second._dst_port_apps.begin()) line += ",";
      line += *app_iter;
      ++app_iter;
    }
    line += ";";

    line += "dst_port_start=";
    if (iter->second._dst_port_start > 0) {
      sprintf(buf, "%d", iter->second._dst_port_start);
      line += string(buf);
    }
    line += ";";

    line += "dst_port_stop=";
    if (iter->second._dst_port_stop > 0) {
      sprintf(buf, "%d", iter->second._dst_port_stop);
      line += string(buf);
    }
    line += ";";

    line += "in_addr=" + iter->second._in_address.str() + ";";

    line += "in_network=" + iter->second._in_network.str() + ";";

    line += "out_addr=" + iter->second._out_address.str() + ";";

    line += "out_network=" + iter->second._out_network.str() + ";";
    
    line += "out_addr_start=" + iter->second._out_address_start.str() + ";";

    line += "out_addr_stop=" + iter->second._out_address_stop.str() + ";";


    //now push it on the list
    if (filter == 2 && iter->second._trans_type == "dynamic") {
      areas.append(XrlAtom(line));
    }
    else if (filter == 1 && iter->second._trans_type == "static") {
      areas.append(XrlAtom(line));
    }
    else if (filter == 0) {
      areas.append(XrlAtom(line));
    }

    ++iter;
  }
  return XrlCmdError::OKAY();
}
