/**
 * Module: xrl_rl_serial_node.hh
 *
 * Author: Michael Larson
 * Date: 2005
 */

#ifndef __RL_INTERFACES_XRL_RL_SERIAL_NODE_HH__
#define __RL_INTERFACES_XRL_RL_SERIAL_NODE_HH__


//
// RLSystem XRL-aware node definition.
//

#include "libxorp/status_codes.h"
#include "libxipc/xrl_std_router.hh"
#include "libfeaclient/ifmgr_xrl_mirror.hh"
#include "xrl/interfaces/finder_event_notifier_xif.hh"
#include "xrl/interfaces/fea_ifmgr_xif.hh"
#include "xrl/interfaces/rl_interface_firewall_xif.hh"
#include "xrl/targets/rl_serial_base.hh"
#include "libxorp/callback.hh"
#include "rl_serial_node.hh"
#include "rl_serial_fea_mgr.hh"

//
// The top-level class that wraps-up everything together under one roof
//
class XrlRLSerialNode :     public XrlStdRouter, /* used for dispatch of xrl messages */
			    public XrlRlSerialTargetBase { /* used for defined interface */
public:
  typedef map<string, RLFeaMgr*> RLFeaMgrColl;
  typedef map<string, RLFeaMgr*>::iterator RLFeaMgrIter;

public:
  XrlRLSerialNode(EventLoop&	eventloop,
		  const string&	class_name,
		  const string&	finder_hostname,
		  uint16_t	        finder_port,
		  const string&	finder_target,
		  const string&     fea_target,
		  const string &rl_interface_fw_target);
  ~XrlRLSerialNode();
  
  /**
   * Startup the node operation.
   *
   * @return true on success, false on failure.
   */
  bool	startup();
  
  /**
   * Shutdown the node operation.
   *
   * @return true on success, false on failure.
   */
  XrlCmdError	rl_serial_0_1_shutdown_interfaces();
  
  /**
   * Get a reference to the XrlRouter instance.
   *
   * @return a reference to the XrlRouter (@ref XrlRouter) instance.
   */
  XrlRouter&	xrl_router() { return *this; }
  
  /**
   * Test if the node processing is done.
   * 
   * @return true if the node processing is done, otherwise false.
   */
    bool	is_done() const { return (_status == PROC_DONE); }
  
protected:
  //
  // XRL target methods
  //

  /**
   *  Get name of Xrl Target
   */
  XrlCmdError common_0_1_get_target_name(
					 // Output values, 
					 string&	name);
  
  /**
   *  Get version string from Xrl Target
   */
  XrlCmdError common_0_1_get_version(
				     // Output values, 
				     string&	version);
  
  /**
   *  Get status of Xrl Target
   */
  XrlCmdError common_0_1_get_status(
				    // Output values, 
				    uint32_t&	status, 
				    string&	reason);
  
  /**
   *  Request clean shutdown of Xrl Target
   */
  XrlCmdError common_0_1_shutdown();
  
  /**
   *  Announce target birth to observer.
   *
   *  @param target_class the target class name.
   *
   *  @param target_instance the target instance name.
   */
  XrlCmdError finder_event_observer_0_1_xrl_target_birth(
							 // Input values,
							 const string&	target_class,
							 const string&	target_instance);
  
  /**
   *  Announce target death to observer.
   *
   *  @param target_class the target class name.
   *
   *  @param target_instance the target instance name.
   */
  XrlCmdError finder_event_observer_0_1_xrl_target_death(
							 // Input values,
							 const string&	target_class,
							 const string&	target_instance);
  
  /**
   *  Enable/disable/start/stop RLSerial
   *
   *  @param enable if true, then enable RLSerial, otherwise disable it.
   */
  XrlCmdError rl_serial_0_1_enable_rl_serial(
						     // Input values,
						     const bool&	enable);
  
  XrlCmdError rl_serial_0_1_start_rl_serial();
  
  XrlCmdError rl_serial_0_1_stop_rl_serial();
  
  /**
   *  Configure an interface
   *
   *  @param address of interface
   */
  XrlCmdError rl_serial_0_1_add_interface(
					      // Input values,
					      const IPv4&	address);
  
  /**
   *  Enable/disable the RLSerial trace log for all operations.
   *
   *  @param enable if true, then enable the trace log, otherwise disable it.
   */
  XrlCmdError rl_serial_0_1_enable_log_trace_all(
						     // Input values,
						     const bool&	enable);
  /**
   *
   */
  XrlCmdError rl_serial_0_1_delete_serial(
					      //Input values,
					      const string &serial);

  /**
   *
   */
  XrlCmdError rl_serial_0_1_set_interfaces_encapsulation( 
							     //Input values,
							     const string &serial,
							     const string &encapsulation);

  /**
   *
   */
  XrlCmdError rl_serial_0_1_delete_proto( 
							     //Input values,
							     const string &serial,
							     const string &proto);

  /**
   *
   */
  XrlCmdError rl_serial_0_1_delete_vif(
				       //Input values,
				       const string &serial,
				       const string &proto,
				       const uint32_t &vif);

  /**
   *
   */
  XrlCmdError rl_serial_0_1_set_interfaces_vif_vlan_id(
							   //Input values,
							   const string &serial,
							   const string &vif,
							   const uint32_t &vlan_id);

  XrlCmdError rl_serial_0_1_delete_interfaces_vif_vlan_id(
							      //Input values,
							      const string &serial,
							      const string &vif,
							      const uint32_t &vlan_id);


  /**
   *
   */
  XrlCmdError rl_serial_0_1_start_serial_transaction(
								 //Input values,
							 const string &serial);

  XrlCmdError rl_serial_0_1_commit_serial_transaction(
								  //Input values,
							  const string &serial);

  XrlCmdError complete_commit_serial_transaction(
						 //Input values,
						 const string &serial);

  /**
   *
   */
  XrlCmdError rl_serial_0_1_set_encapsulation(
						  //Input values,
						  const string &serial,
						  const string &encapsulation);

  /**
   *	
   */
  XrlCmdError rl_serial_0_1_create_address4(
			      //Input values,
			      const string &serial,
			      const string &proto,
			      const uint32_t &vif,
			      const IPv4 &address,
			      const IPv4 &remote_address,
			      const uint32_t &prefix_length);

  XrlCmdError rl_serial_0_1_delete_address4(
			      //Input values, 
			      const string &serial,
			      const string &proto,
			      const uint32_t &vif,
			      const IPv4 &address,
			      const IPv4 &remote_address,
			      const uint32_t &prefix_length);

  /**
   *
   **/
  XrlCmdError rl_serial_0_1_set_dlci(
					//Input values,
					const string &serial,
					const string &proto,
					const uint32_t &vif,
					const uint32_t &dlci);

  XrlCmdError rl_serial_0_1_set_cir(
					//Input values,
					const string &serial,
					const string &proto,
					const uint32_t &vif,
					const uint32_t &cir);

  XrlCmdError rl_serial_0_1_set_burst(
					  //Input values,
					  const string &serial,
					  const string &proto,
					  const uint32_t &vif,
					  const uint32_t &burst);

  XrlCmdError rl_serial_0_1_set_excess(
					   //Input values,
					   const string &serial,
					   const string &proto,
					   const uint32_t &vif,
					   const uint32_t &excess);

  XrlCmdError rl_serial_0_1_set_tx_inverse_arp(
						   //Input values,
						   const string &serial,
						   const string &proto,
						   const uint32_t &vif,
						   const uint32_t &tx_inverse_arp);

  XrlCmdError rl_serial_0_1_set_rx_inverse_arp(
						   //Input values,
						   const string &serial,
						   const string &proto,
						   const uint32_t &vif,
						   const string &rx_inverse_arp);

  /**
   *
   */
  XrlCmdError rl_serial_0_1_set_card_type(
					  //Input values,
					  const string &serial,
					  const string &card_type
					  );
					  
  /**
   *
   */	
  XrlCmdError rl_serial_0_1_delete_physical(
					    //Input values,
					    const string &serial
					    );
  
  /**
   *
   */
  XrlCmdError rl_serial_0_1_set_t1_options_lbo(
			       //Input values,
			       const string &serial, 
			       const string &lbo);
  XrlCmdError rl_serial_0_1_set_t1_options_line_coding(
				       //Input values,
				       const string &serial, 
				       const string &line_coding);
  XrlCmdError rl_serial_0_1_set_t1_options_timeslots(
				     //Input values,
				     const string &serial, 
				     const uint32_t &start,
				     const uint32_t &stop);
  XrlCmdError rl_serial_0_1_set_t1_options_mtu(
			       //Input values,
			       const string &serial, 
			       const uint32_t &mtu);
  XrlCmdError rl_serial_0_1_set_t1_options_clock(
				 //Input values,
				 const string &serial, 
				 const string &clock);
  
  /**
   *
   **/
  XrlCmdError rl_serial_0_1_set_e1_options_framing(
				   //Input values,
				   const string &serial, 
				   const string &line);
  XrlCmdError rl_serial_0_1_set_e1_options_line_coding(
				       //Input values,
				       const string &serial, 
				       const string &line_coding);
  XrlCmdError rl_serial_0_1_set_e1_options_timeslots(
				     //Input values,
				     const string &serial, 
				     const uint32_t &start,
				     const uint32_t &stop);
  XrlCmdError rl_serial_0_1_set_e1_options_mtu(
			       //Input values,
			       const string &serial, 
			       const uint32_t &mtu);
  XrlCmdError rl_serial_0_1_set_e1_options_clock(
				 //Input values,
				 const string &serial, 
				 const string &clock);
  
  /**
   *
   **/
  XrlCmdError rl_serial_0_1_set_t3_options_framing(
				   //Input values,
				   const string &serial, 
				   const string &line);
  XrlCmdError rl_serial_0_1_set_t3_options_line_coding(
				       //Input values,
				       const string &serial, 
				       const string &line_coding);

  /**
   *
   **/
  XrlCmdError rl_serial_0_1_set_e3_options_framing(
				   //Input values,
				   const string &serial, 
				   const string &line);
  XrlCmdError rl_serial_0_1_set_e3_options_line_coding(
				       //Input values,
				       const string &serial, 
				       const string &line_coding);

  /**
   *
   */
  XrlCmdError rl_serial_0_1_set_ppp_authentication(
				     //Input values,
				     const string &serial, 
				     const string &proto,
				     const string &type);
  XrlCmdError rl_serial_0_1_set_ppp_auth_parameters_user_id(
					      //Input values,
					      const string &serial, 
					      const string &proto,
					      const string &user_id);
  XrlCmdError rl_serial_0_1_set_ppp_auth_parameters_password(
					       //Input values,
					       const string &serial, 
					       const string &proto,
					       const string &password);
  XrlCmdError rl_serial_0_1_set_ppp_auth_parameters_system_name(
						  //Input values,
						  const string &serial, 
						  const string &proto,
						  const string &system_name);
  
  /**
   *
   */
  XrlCmdError rl_serial_0_1_set_cisco_hdlc_keepalives(
						  //Input values,
						       const string &serial, 
						       const string &proto,
						       const string &require_rx,
						       const uint32_t &timer);
  /**
   *
   */
  XrlCmdError rl_serial_0_1_set_frame_relay_signaling(
							  //Input values,
							  const string &serial, 
							  const string &proto,
							  const string &signaling);
  XrlCmdError rl_serial_0_1_set_frame_relay_signaling_options_n391dte(
									  //Input values,
									  const string &serial, 
									  const string &proto,
									  const uint32_t &n391dte);
  XrlCmdError rl_serial_0_1_set_frame_relay_signaling_options_n392dte(
									  //Input values,
									  const string &serial, 
									  const string &proto,
									  const uint32_t &n392dte);
  XrlCmdError rl_serial_0_1_set_frame_relay_signaling_options_n393dte(
									  //Input values,
									  const string &serial, 
									  const string &proto,
									  const uint32_t &n393dte);
  XrlCmdError rl_serial_0_1_set_frame_relay_signaling_options_t391dte(
									  //Input values,
									  const string &serial, 
									  const string &proto,
									  const uint32_t &t391dte);

  XrlCmdError rl_serial_0_1_set_serial_firewall_name( 
						     //Input values,
						     const string &serial,
						     const uint32_t &vif,
						     const string &proto,
						     const string &firewall_name,
						     const string &filter);

  XrlCmdError rl_serial_0_1_delete_serial_firewall_name( 
						     //Input values,
						     const string &serial,
						     const uint32_t &vif,
						     const string &proto,
						     const string &firewall_name,
						     const string &filter);

  XrlCmdError rl_serial_0_1_get_serial_data(
					    //Input values,
					    const string &name,
					    //output values,
					    XrlAtomList &rules);
  


  XrlCmdError complete_shutdown(const string &serial);  
  XrlCmdError complete_add(const string &serial);  
  XrlCmdError complete_delete(const string &serial);  
  XrlCmdError complete_delete_final(const string &serial);  

private:
  /**
   *
   **/
  void send_transaction_cb(const XrlError &xrl_error);

  /**
   * Called when Finder connection is established.
   *
   * Note that this method overwrites an XrlRouter virtual method.
   */
  virtual void finder_connect_event();
  
  /**
   * Called when Finder disconnect occurs.
   *
   * Note that this method overwrites an XrlRouter virtual method.
   */
  virtual void finder_disconnect_event();
  
  void fea_register_startup();
  void finder_register_interest_fea_cb(const XrlError& xrl_error);
  void fea_register_shutdown();
  void finder_deregister_interest_fea_cb(const XrlError& xrl_error);

  void rl_interface_fw_register_startup();
  void finder_register_interest_rl_interface_fw_cb(const XrlError& xrl_error);
  void rl_interface_fw_register_shutdown();
  void finder_deregister_interest_rl_interface_fw_cb(const XrlError& xrl_error);

  void firewall_cb(const XrlError &xrl_error);

  const string& my_xrl_target_name() {
    return XrlRlSerialTargetBase::name();
  }
  
  EventLoop&		_eventloop;
  const string	_class_name;
  const string	_instance_name;
  const string	_finder_target;
  const string	_fea_target;
  const string	_rl_interface_fw_target;

  IfMgrXrlMirror	_ifmgr;

  XrlFinderEventNotifierV0p1Client	_xrl_finder_client;
  
  static const TimeVal RETRY_TIMEVAL;
  
  bool		_is_finder_alive;

  bool		_is_fea_registered;
  bool		_is_fea_registering;
  bool		_is_fea_deregistering;
  XorpTimer		_fea_register_startup_timer;
  XorpTimer		_fea_register_shutdown_timer;
  XrlIfmgrV0p1Client	_xrl_fea_client;  
  bool		_is_fea_alive;

  bool		_is_rl_interface_fw_registered;
  bool		_is_rl_interface_fw_registering;
  bool		_is_rl_interface_fw_deregistering;
  XorpTimer		_rl_interface_fw_register_startup_timer;
  XorpTimer		_rl_interface_fw_register_shutdown_timer;
  XrlRlInterfaceFirewallV0p1Client _rl_interface_fw_client;
  bool		_is_rl_interface_fw_alive;

  RLSerialNode   _rl_serial_node;

  VifManager::IfaceDeltaColl _iface_add_coll;
  VifManager::IfaceDeltaColl _iface_del_coll;
  
  uint32_t _status;
  /*  RLFeaMgr *_fea_mgr_del;
      RLFeaMgr *_fea_mgr_add;*/

  RLFeaMgrColl _fea_mgr_del_coll;
  RLFeaMgrColl _fea_mgr_add_coll;
};

#endif // __RL_INTERFACES_XRL_RL_SERIAL_NODE_HH__
