/**
 * Module: xrl_rl_serial_node.hh
 *
 * Author: Michael Larson
 * Date: 2005
 */

#include "rl_serial_module.h"

#include "libxorp/xorp.h"
#include "libxorp/xlog.h"
#include "libxorp/debug.h"
#include "libxorp/ipvx.hh"
#include "libxorp/status_codes.h"

#include "xrl/interfaces/fea_ifmgr_xif.hh"
#include "xrl/interfaces/rl_interface_firewall_xif.hh"

#include "rl_serial_node.hh"

#include "rl_serial_fea_mgr.hh"

#include "xrl_rl_serial_node.hh"

const TimeVal XrlRLSerialNode::RETRY_TIMEVAL = TimeVal(1, 0);

XrlRLSerialNode::XrlRLSerialNode(EventLoop&	eventloop,
				 const string&	class_name,
				 const string&	finder_hostname,
				 uint16_t	finder_port,
				 const string&	finder_target,
				 const string& fea_target,
				 const string &rl_interface_fw_target)
    : XrlStdRouter(eventloop, class_name.c_str(), finder_hostname.c_str(),
		   finder_port),
      XrlRlSerialTargetBase(&xrl_router()),
      _eventloop(eventloop),
      _class_name(xrl_router().class_name()),
      _instance_name(xrl_router().instance_name()),
      _finder_target(finder_target),
      _fea_target(fea_target),
      _rl_interface_fw_target(rl_interface_fw_target),
      _ifmgr(eventloop, fea_target.c_str(), xrl_router().finder_address(),
	     xrl_router().finder_port()),
      _xrl_finder_client(&xrl_router()),
      _is_finder_alive(false),
      _is_fea_registered(false),
      _is_fea_registering(false),
      _is_fea_deregistering(false),
      _xrl_fea_client(&xrl_router()),
      _is_fea_alive(false),
      _rl_interface_fw_client(&xrl_router()),
      _status(PROC_STARTUP)
{
}

XrlRLSerialNode::~XrlRLSerialNode()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    rl_serial_0_1_shutdown_interfaces();
}

bool
XrlRLSerialNode::startup()
{
  XLOG_INFO("XrlRLSerialNode::startup() entered and now calling");
  
  fea_register_startup();
  _status = PROC_READY;
  return true;
}

XrlCmdError
XrlRLSerialNode::rl_serial_0_1_shutdown_interfaces()
{
  //handles shutdown of interfaces and deletion of nodes?
  VifManager::IfaceDeltaColl iface_coll;
  SerialData::FirewallDeltaColl fw_coll;
  XrlCmdError err = _rl_serial_node.reset(iface_coll, fw_coll);

  //we'll want to wire these into the current settings of the state
  VifManager::IfaceDeltaIter iter = iface_coll.begin();
  while (iter != iface_coll.end()) {
    RLFeaMgr *fea_mgr = new RLFeaMgr(_xrl_fea_client, _fea_target, *iter, true, this, &XrlRLSerialNode::complete_shutdown);
    _fea_mgr_del_coll.insert(pair<string,RLFeaMgr*>(iter->_serial, fea_mgr));
    fea_mgr->set_finder_state(_is_finder_alive);
    fea_mgr->set_fea_interest(_is_fea_registered);
    fea_mgr->process();
    ++iter;
  }
  //needs to handle proc_shutdown in the final callback from this object

  //now set the firewall if needed
  char buf[80];
  SerialData::FirewallDeltaIter fw_iter = fw_coll.begin();
  while (fw_iter != fw_coll.end()) {
    sprintf(buf, "%d", fw_iter->_vif);
    string interface = fw_iter->_interface + "." + string(buf);
    _rl_interface_fw_client.send_delete_interfaces_interface_firewall_name(_rl_interface_fw_target.c_str(), 
									   fw_iter->_firewall_name, 
									   fw_iter->_filter, 
									   interface,
									   callback(this, &XrlRLSerialNode::firewall_cb));
    ++fw_iter;
  }

  _status = PROC_SHUTDOWN;
  return err;
}

/**
 *
 **/
XrlCmdError
XrlRLSerialNode::complete_shutdown(const string &serial)
{
  //continue on with other shutdown work...

  //now mark the internal housekeeping as complete
  //and let the module to be shutdown by the rtrmgr.
  RLFeaMgrIter iter = _fea_mgr_del_coll.find(serial);

  if (iter != _fea_mgr_del_coll.end() && 
      iter->second != NULL) {
    delete iter->second;
    _fea_mgr_del_coll.erase(iter);
  }

  _status = PROC_DONE;
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
XrlRLSerialNode::complete_delete(const string &serial)
{
  //continue on with other shutdown work...

  //now mark the internal housekeeping as complete
  //and let the module to be shutdown by the rtrmgr.
  RLFeaMgrIter iter = _fea_mgr_del_coll.find(serial);

  if (iter != _fea_mgr_del_coll.end() && 
      iter->second != NULL) {
    delete iter->second;
    _fea_mgr_del_coll.erase(iter);
  }

  //now proceed with the add...
  return complete_commit_serial_transaction(serial);
}

/**
 *
 **/
XrlCmdError 
XrlRLSerialNode::complete_delete_final(const string &serial)
{
  //continue on with other shutdown work...

  //now mark the internal housekeeping as complete
  //and let the module to be shutdown by the rtrmgr.
  RLFeaMgrIter iter = _fea_mgr_del_coll.find(serial);

  if (iter != _fea_mgr_del_coll.end() && 
      iter->second != NULL) {
    delete iter->second;
    _fea_mgr_del_coll.erase(iter);
  }

  //no add in this case, this is it!
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
XrlRLSerialNode::complete_add(const string &serial)
{
  //continue on with other shutdown work...

  //now mark the internal housekeeping as complete
  //and let the module to be shutdown by the rtrmgr.
  RLFeaMgrIter iter = _fea_mgr_add_coll.find(serial);

  if (iter != _fea_mgr_add_coll.end() && 
      iter->second != NULL) {
    delete iter->second;
    _fea_mgr_add_coll.erase(iter);
  }
  return XrlCmdError::OKAY();
}

//
// Finder-related events
//
/**
 * Called when Finder connection is established.
 *
 * Note that this method overwrites an XrlRouter virtual method.
 */
void
XrlRLSerialNode::finder_connect_event()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    _is_finder_alive = true;
}

/**
 * Called when Finder disconnect occurs.
 *
 * Note that this method overwrites an XrlRouter virtual method.
 */
void
XrlRLSerialNode::finder_disconnect_event()
{
    XLOG_ERROR("Finder disconnect event. Exiting immediately...");
    //printf("%s\n",__PRETTY_FUNCTION__);

    _is_finder_alive = false;
}

//
// XRL target methods
//
//
// Register with the FEA
//
void
XrlRLSerialNode::fea_register_startup()
{
    bool success;

    _fea_register_startup_timer.unschedule();
    _fea_register_shutdown_timer.unschedule();

    if (! _is_finder_alive)
	return;		// The Finder is dead

    if (_is_fea_registered)
	return;		// Already registered

    //removing for now from static routes examples
    if (! _is_fea_registering) {
      //>	StaticRoutesNode::incr_startup_requests_n();	// XXX: for the ifmgr
	_is_fea_registering = true;
    }

    //
    // Register interest in the FEA with the Finder
    //
    success = _xrl_finder_client.send_register_class_event_interest(
	_finder_target.c_str(), _instance_name, _fea_target,
	callback(this, &XrlRLSerialNode::finder_register_interest_fea_cb));

    if (! success) {
	//
	// If an error, then start a timer to try again.
	//
	_fea_register_startup_timer = _eventloop.new_oneoff_after(
	    RETRY_TIMEVAL,
	    callback(this, &XrlRLSerialNode::fea_register_startup));
	return;
    }
}

void
XrlRLSerialNode::finder_register_interest_fea_cb(const XrlError& xrl_error)
{
  XLOG_INFO("XrlRLSerialNode::finder_register_interest_fea_cb: %d", xrl_error.error_code());
  

    switch (xrl_error.error_code()) {
    case OKAY:
	//
	// If success, then the FEA birth event will startup the ifmgr
	//
	_is_fea_registering = false;
	_is_fea_registered = true;
	break;

    case COMMAND_FAILED:
	//
	// If a command failed because the other side rejected it, this is
	// fatal.
	//
	XLOG_FATAL("Cannot register interest in finder events: %s",
		   xrl_error.str().c_str());
	break;

    case NO_FINDER:
    case RESOLVE_FAILED:
    case SEND_FAILED:
	//
	// A communication error that should have been caught elsewhere
	// (e.g., by tracking the status of the finder and the other targets).
	// Probably we caught it here because of event reordering.
	// In some cases we print an error. In other cases our job is done.
	//
	XLOG_ERROR("XRL communication error: %s", xrl_error.str().c_str());
	break;

    case BAD_ARGS:
    case NO_SUCH_METHOD:
    case INTERNAL_ERROR:
	//
	// An error that should happen only if there is something unusual:
	// e.g., there is XRL mismatch, no enough internal resources, etc.
	// We don't try to recover from such errors, hence this is fatal.
	//
	XLOG_FATAL("Fatal XRL error: %s", xrl_error.str().c_str());
	break;

    case REPLY_TIMED_OUT:
    case SEND_FAILED_TRANSIENT:
	//
	// If a transient error, then start a timer to try again
	// (unless the timer is already running).
	//
	if (! _fea_register_startup_timer.scheduled()) {
	    XLOG_ERROR("Failed to register interest in finder events: %s. "
		       "Will try again.",
		       xrl_error.str().c_str());
	    _fea_register_startup_timer = _eventloop.new_oneoff_after(
		RETRY_TIMEVAL,
		callback(this, &XrlRLSerialNode::fea_register_startup));
	}
	break;
    }
}

//
// De-register with the FEA
//
void
XrlRLSerialNode::fea_register_shutdown()
{
    bool success;

    _fea_register_startup_timer.unschedule();
    _fea_register_shutdown_timer.unschedule();

    if (! _is_finder_alive)
	return;		// The Finder is dead

    if (! _is_fea_alive)
	return;		// The FEA is not there anymore

    if (! _is_fea_registered)
	return;		// Not registered

    if (! _is_fea_deregistering) {
      //	StaticRoutesNode::incr_shutdown_requests_n();	// XXX: for the ifmgr

	_is_fea_deregistering = true;
    }

    //
    // De-register interest in the FEA with the Finder
    //
    success = _xrl_finder_client.send_deregister_class_event_interest(
	_finder_target.c_str(), _instance_name, _fea_target,
	callback(this, &XrlRLSerialNode::finder_deregister_interest_fea_cb));

    if (! success) {
	//
	// If an error, then start a timer to try again.
	//
	_fea_register_shutdown_timer = _eventloop.new_oneoff_after(
	    RETRY_TIMEVAL,
	    callback(this, &XrlRLSerialNode::fea_register_shutdown));
	return;
    }

    //
    // XXX: when the shutdown is completed, StaticRoutesNode::status_change()
    // will be called.
    //
    _ifmgr.shutdown();
}

void
XrlRLSerialNode::finder_deregister_interest_fea_cb(
    const XrlError& xrl_error)
{
    switch (xrl_error.error_code()) {
    case OKAY:
	//
	// If success, then we are done
	//
	_is_fea_deregistering = false;
	_is_fea_registered = false;
	XLOG_INFO("RLSerialNode:: finder_dergistered_interface_fea_cb");
	break;

    case COMMAND_FAILED:
	//
	// If a command failed because the other side rejected it, this is
	// fatal.
	//
	XLOG_FATAL("Cannot deregister interest in finder events: %s",
		   xrl_error.str().c_str());
	break;

    case NO_FINDER:
    case RESOLVE_FAILED:
    case SEND_FAILED:
	//
	// A communication error that should have been caught elsewhere
	// (e.g., by tracking the status of the finder and the other targets).
	// Probably we caught it here because of event reordering.
	// In some cases we print an error. In other cases our job is done.
	//
	_is_fea_deregistering = false;
	_is_fea_registered = false;
	XLOG_INFO("RLSerialNode:: SEND_FAILED");
	break;

    case BAD_ARGS:
    case NO_SUCH_METHOD:
    case INTERNAL_ERROR:
	//
	// An error that should happen only if there is something unusual:
	// e.g., there is XRL mismatch, no enough internal resources, etc.
	// We don't try to recover from such errors, hence this is fatal.
	//
	XLOG_FATAL("Fatal XRL error: %s", xrl_error.str().c_str());
	break;

    case REPLY_TIMED_OUT:
    case SEND_FAILED_TRANSIENT:
	//
	// If a transient error, then start a timer to try again
	// (unless the timer is already running).
	//
	if (! _fea_register_shutdown_timer.scheduled()) {
	    XLOG_ERROR("Failed to deregister interest in finder events: %s. "
		       "Will try again.",
		       xrl_error.str().c_str());
	    _fea_register_shutdown_timer = _eventloop.new_oneoff_after(
		RETRY_TIMEVAL,
		callback(this, &XrlRLSerialNode::fea_register_shutdown));
	}
	break;
    }
}


//
// XRL target methods
//
//
// Register with the FEA
//
void
XrlRLSerialNode::rl_interface_fw_register_startup()
{
    bool success;

    _rl_interface_fw_register_startup_timer.unschedule();
    _rl_interface_fw_register_shutdown_timer.unschedule();

    if (! _is_finder_alive)
	return;		// The Finder is dead

    if (_is_rl_interface_fw_registered)
	return;		// Already registered

    //removing for now from static routes examples
    if (! _is_rl_interface_fw_registering) {
      //>	StaticRoutesNode::incr_startup_requests_n();	// XXX: for the ifmgr
	_is_rl_interface_fw_registering = true;
    }

    //
    // Register interest in the FEA with the Finder
    //
    success = _xrl_finder_client.send_register_class_event_interest(
	_finder_target.c_str(), _instance_name, _rl_interface_fw_target,
	callback(this, &XrlRLSerialNode::finder_register_interest_rl_interface_fw_cb));

    if (! success) {
	//
	// If an error, then start a timer to try again.
	//
	_rl_interface_fw_register_startup_timer = _eventloop.new_oneoff_after(
	    RETRY_TIMEVAL,
	    callback(this, &XrlRLSerialNode::rl_interface_fw_register_startup));
	return;
    }
}

void
XrlRLSerialNode::finder_register_interest_rl_interface_fw_cb(const XrlError& xrl_error)
{
  XLOG_INFO("XrlRLSerialNode::finder_register_interest_rl_interface_fw_cb: %d", xrl_error.error_code());
  

    switch (xrl_error.error_code()) {
    case OKAY:
	//
	// If success, then the FEA birth event will startup the ifmgr
	//
	_is_rl_interface_fw_registering = false;
	_is_rl_interface_fw_registered = true;
	XLOG_INFO("RLSerialNode:: TESTA: REGISTERED FEA");
	break;

    case COMMAND_FAILED:
	//
	// If a command failed because the other side rejected it, this is
	// fatal.
	//
	XLOG_FATAL("Cannot register interest in finder events: %s",
		   xrl_error.str().c_str());
	break;

    case NO_FINDER:
    case RESOLVE_FAILED:
    case SEND_FAILED:
	//
	// A communication error that should have been caught elsewhere
	// (e.g., by tracking the status of the finder and the other targets).
	// Probably we caught it here because of event reordering.
	// In some cases we print an error. In other cases our job is done.
	//
	XLOG_ERROR("XRL communication error: %s", xrl_error.str().c_str());
	break;

    case BAD_ARGS:
    case NO_SUCH_METHOD:
    case INTERNAL_ERROR:
	//
	// An error that should happen only if there is something unusual:
	// e.g., there is XRL mismatch, no enough internal resources, etc.
	// We don't try to recover from such errors, hence this is fatal.
	//
	XLOG_FATAL("Fatal XRL error: %s", xrl_error.str().c_str());
	break;

    case REPLY_TIMED_OUT:
    case SEND_FAILED_TRANSIENT:
	//
	// If a transient error, then start a timer to try again
	// (unless the timer is already running).
	//
	if (! _rl_interface_fw_register_startup_timer.scheduled()) {
	    XLOG_ERROR("Failed to register interest in finder events: %s. "
		       "Will try again.",
		       xrl_error.str().c_str());
	    _rl_interface_fw_register_startup_timer = _eventloop.new_oneoff_after(
		RETRY_TIMEVAL,
		callback(this, &XrlRLSerialNode::rl_interface_fw_register_startup));
	}
	break;
    }
}

//
// De-register with the FEA
//
void
XrlRLSerialNode::rl_interface_fw_register_shutdown()
{
    bool success;
    XLOG_INFO("XrlRLSerialNode::rl_interface_fw_register_shutdown");

    _rl_interface_fw_register_startup_timer.unschedule();
    _rl_interface_fw_register_shutdown_timer.unschedule();

    if (! _is_finder_alive)
	return;		// The Finder is dead

    if (! _is_rl_interface_fw_alive)
	return;		// The FEA is not there anymore

    if (! _is_rl_interface_fw_registered)
	return;		// Not registered

    if (! _is_rl_interface_fw_deregistering) {
      //	StaticRoutesNode::incr_shutdown_requests_n();	// XXX: for the ifmgr

	_is_rl_interface_fw_deregistering = true;
    }

    //
    // De-register interest in the FEA with the Finder
    //
    success = _xrl_finder_client.send_deregister_class_event_interest(
	_finder_target.c_str(), _instance_name, _rl_interface_fw_target,
	callback(this, &XrlRLSerialNode::finder_deregister_interest_rl_interface_fw_cb));

    if (! success) {
	//
	// If an error, then start a timer to try again.
	//
	_rl_interface_fw_register_shutdown_timer = _eventloop.new_oneoff_after(
	    RETRY_TIMEVAL,
	    callback(this, &XrlRLSerialNode::rl_interface_fw_register_shutdown));
	return;
    }

    //
    // XXX: when the shutdown is completed, StaticRoutesNode::status_change()
    // will be called.
    //
    _ifmgr.shutdown();
}

void
XrlRLSerialNode::finder_deregister_interest_rl_interface_fw_cb(
    const XrlError& xrl_error)
{
    switch (xrl_error.error_code()) {
    case OKAY:
	//
	// If success, then we are done
	//
	_is_rl_interface_fw_deregistering = false;
	_is_rl_interface_fw_registered = false;
	XLOG_INFO("RLSerialNode:: TESTB: finder_dergistered_interface_rl_interface_fw_cb");
	break;

    case COMMAND_FAILED:
	//
	// If a command failed because the other side rejected it, this is
	// fatal.
	//
	XLOG_FATAL("Cannot deregister interest in finder events: %s",
		   xrl_error.str().c_str());
	break;

    case NO_FINDER:
    case RESOLVE_FAILED:
    case SEND_FAILED:
	//
	// A communication error that should have been caught elsewhere
	// (e.g., by tracking the status of the finder and the other targets).
	// Probably we caught it here because of event reordering.
	// In some cases we print an error. In other cases our job is done.
	//
	_is_rl_interface_fw_deregistering = false;
	_is_rl_interface_fw_registered = false;
	XLOG_INFO("RLSerialNode:: TESTC: SEND_FAILED");
	break;

    case BAD_ARGS:
    case NO_SUCH_METHOD:
    case INTERNAL_ERROR:
	//
	// An error that should happen only if there is something unusual:
	// e.g., there is XRL mismatch, no enough internal resources, etc.
	// We don't try to recover from such errors, hence this is fatal.
	//
	XLOG_FATAL("Fatal XRL error: %s", xrl_error.str().c_str());
	break;

    case REPLY_TIMED_OUT:
    case SEND_FAILED_TRANSIENT:
	//
	// If a transient error, then start a timer to try again
	// (unless the timer is already running).
	//
	if (! _rl_interface_fw_register_shutdown_timer.scheduled()) {
	    XLOG_ERROR("Failed to deregister interest in finder events: %s. "
		       "Will try again.",
		       xrl_error.str().c_str());
	    _rl_interface_fw_register_shutdown_timer = _eventloop.new_oneoff_after(
		RETRY_TIMEVAL,
		callback(this, &XrlRLSerialNode::rl_interface_fw_register_shutdown));
	}
	break;
    }
}


/**
 *  Get name of Xrl Target
 */
XrlCmdError
XrlRLSerialNode::common_0_1_get_target_name(
    // Output values, 
    string&	name)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    name = my_xrl_target_name();

    return XrlCmdError::OKAY();
}

/**
 *  Get version string from Xrl Target
 */
XrlCmdError
XrlRLSerialNode::common_0_1_get_version(
    // Output values, 
    string&	version)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    version = XORP_MODULE_VERSION;

    return XrlCmdError::OKAY();
}

/**
 *  Get status of Xrl Target
 */
XrlCmdError
XrlRLSerialNode::common_0_1_get_status(
    // Output values, 
    uint32_t&	status, 
    string&	reason)
{
  reason = " ";
  status = _status;

  //  if (PROC_SHUTDOWN == _status) {
  //    _status = PROC_DONE;
    //  }

  return XrlCmdError::OKAY();
}

/**
 *  Request clean shutdown of Xrl Target
 */
XrlCmdError
XrlRLSerialNode::common_0_1_shutdown()
{
    string error_msg;
    //printf("%s\n",__PRETTY_FUNCTION__);
    /*
    if (shutdown() != true) {
	error_msg = c_format("Failed to shutdown RLSerial");
	return XrlCmdError::COMMAND_FAILED(error_msg);
    }
    */
    return XrlCmdError::OKAY();
}

/**
 *  Announce target birth to observer.
 *
 *  @param target_class the target class name.
 *
 *  @param target_instance the target instance name.
 */
XrlCmdError
XrlRLSerialNode::finder_event_observer_0_1_xrl_target_birth(
    // Input values,
    const string&   target_class,
    const string&   target_instance)
{
    UNUSED(target_class);
    UNUSED(target_instance);

    if (target_class == _fea_target) {
	//
	// XXX: when the startup is completed,
	// IfMgrHintObserver::tree_complete() will be called.
	//
	_is_fea_alive = true;
	if (_ifmgr.startup() != true) {
	  /*	    StaticRoutesNode::ServiceBase::set_status(SERVICE_FAILED);
		    StaticRoutesNode::update_status();*/
	}
    }

    if (target_class == _rl_interface_fw_target) {
	//
	// XXX: when the startup is completed,
	// IfMgrHintObserver::tree_complete() will be called.
	//
	_is_rl_interface_fw_alive = true;
	if (_ifmgr.startup() != true) {
	  /*	    StaticRoutesNode::ServiceBase::set_status(SERVICE_FAILED);
		    StaticRoutesNode::update_status();*/
	}
    }

    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
}

/**
 *  Announce target death to observer.
 *
 *  @param target_class the target class name.
 *
 *  @param target_instance the target instance name.
 */
XrlCmdError
XrlRLSerialNode::finder_event_observer_0_1_xrl_target_death(
    // Input values,
    const string&   target_class,
    const string&   target_instance)
{
    UNUSED(target_class);
    UNUSED(target_instance);

    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
}

/**
 *  Enable/disable/start/stop RLSerial.
 *
 *  @param enable if true, then enable RLSerial, otherwise disable it.
 */
XrlCmdError
XrlRLSerialNode::rl_serial_0_1_enable_rl_serial(
    // Input values,
    const bool&	enable)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    UNUSED(enable);

    // XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

XrlCmdError
XrlRLSerialNode::rl_serial_0_1_start_rl_serial()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    // XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

XrlCmdError
XrlRLSerialNode::rl_serial_0_1_stop_rl_serial()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

/**
 *  Add an interface
 *
 *  @param add an interface
 */
XrlCmdError
XrlRLSerialNode::rl_serial_0_1_add_interface(
    // Input values,
    const IPv4&	address)
{
    UNUSED(address);

    //printf("%s\n",__PRETTY_FUNCTION__);
    string error_msg;

    //ADD INTERFACE EXEC HERE!
    //printf("XrlRLSerialNode::rl_serial_0_1_add_interface()--called!\n");

    return XrlCmdError::OKAY();
}

/**
 *  Enable/disable the RLSerial trace log for all operations.
 *
 *  @param enable if true, then enable the trace log, otherwise disable it.
 */
XrlCmdError
XrlRLSerialNode::rl_serial_0_1_enable_log_trace_all(
    // Input values,
    const bool&	enable)
{
    UNUSED(enable);

    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
}


  /**
   *
   */
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_interfaces_encapsulation( 
							     //Input values,
							     const string &interface,
							     const string &encapsulation)
{
  return _rl_serial_node.set_encap(interface, encapsulation);
}

  /**
   *
   */
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_interfaces_vif_vlan_id(
							   //Input values,
							   const string &interface,
							   const string &vif,
							   const uint32_t &vlan_id)
{
  return _rl_serial_node.set_vlan_id(interface, vif, vlan_id);
}

XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_delete_interfaces_vif_vlan_id(
							      //Input values,
							      const string &interface,
							      const string &vif,
							      const uint32_t &vlan_id)
{
  return _rl_serial_node.delete_vlan_id(interface, vif, vlan_id);
}

/**
 *
 */
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_delete_serial(
				//Input values,
				const string &serial)
{
  //needs to be revisited along with commit etc.
  VifManager::IfaceDeltaColl coll;
  _rl_serial_node.delete_serial(serial, coll);
  VifManager::IfaceDeltaIter iter = coll.begin();
  while (iter != coll.end()) {
    RLFeaMgr *fea_mgr = new RLFeaMgr(_xrl_fea_client, _fea_target, *iter, true, this, &XrlRLSerialNode::complete_delete_final);
    _fea_mgr_del_coll.insert(pair<string,RLFeaMgr*>(serial, fea_mgr));
    fea_mgr->set_finder_state(_is_finder_alive);
    fea_mgr->set_fea_interest(_is_fea_registered);
    fea_mgr->process();
    ++iter;
  }
  
  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_start_serial_transaction(
								//Input values,
								const string &serial)
{
  return _rl_serial_node.start_serial_transaction(serial);
}

/**
 *
 */
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_commit_serial_transaction(
							 //Input values,
							 const string &serial)
{
  XLOG_INFO("XrlRLSerialNode::rl_serial_0_1_commit_serial_transaction");
  //if the configuration has not changed, then we are done.
  if (_rl_serial_node.config_change(serial) == false) {
    return XrlCmdError::OKAY();
  }

  //clear out the previous set of processed data
  _iface_add_coll.clear();
  _iface_del_coll.clear();

  XrlCmdError err = _rl_serial_node.pre_serial_commit(serial, _iface_del_coll);
  if (err != XrlCmdError::OKAY()) {
    return err;
  }

  if (_iface_del_coll.empty()) {
    //skip the deletion and just add...
    complete_delete(serial);
  }
  else {
    XLOG_INFO("XrlRLSerialNode::commit_serial_transaction--deleting: %d", _iface_del_coll.size());
    VifManager::IfaceDeltaIter iter = _iface_del_coll.begin();
    while (iter != _iface_del_coll.end()) {
      RLFeaMgr *fea_mgr = new RLFeaMgr(_xrl_fea_client, _fea_target, *iter, true, this, &XrlRLSerialNode::complete_delete);
      //we'll want to wire these into the current settings of the state
      _fea_mgr_del_coll.insert(pair<string,RLFeaMgr*>(iter->_serial, fea_mgr));
      fea_mgr->set_finder_state(_is_finder_alive);
      fea_mgr->set_fea_interest(_is_fea_registered);
      fea_mgr->process();
      ++iter;
    }
  }

  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
XrlRLSerialNode::complete_commit_serial_transaction(const string &serial) 
{
  XLOG_INFO("RLSerialNode::commit_serial_transaction--committing");
  //  return coll of added and deleted vifs/ifnames and apply below!!!!!!!!
  SerialData::FirewallDeltaColl fw_add_coll, fw_del_coll;
  XrlCmdError err2 = _rl_serial_node.commit_serial_transaction(serial, _iface_add_coll, fw_add_coll, fw_del_coll);
  if (err2 != XrlCmdError::OKAY()) {
    return err2;
  }

  VifManager::IfaceDeltaIter iter = _iface_add_coll.begin();
  while (iter != _iface_add_coll.end()) {
    RLFeaMgr *fea_mgr = new RLFeaMgr(_xrl_fea_client, _fea_target, *iter, false, this, &XrlRLSerialNode::complete_add);
    //we'll want to wire these into the current settings of the state
    _fea_mgr_add_coll.insert(pair<string,RLFeaMgr*>(iter->_serial, fea_mgr));
    fea_mgr->set_finder_state(_is_finder_alive);
    fea_mgr->set_fea_interest(_is_fea_registered);
    fea_mgr->process();
    ++iter;
  }


 //now set the firewall if needed
  char buf[80];
  SerialData::FirewallDeltaIter fw_iter = fw_add_coll.begin();
  while (fw_iter != fw_add_coll.end()) {
    sprintf(buf, "%d", fw_iter->_vif);
    string interface = fw_iter->_interface + "." + string(buf);
    _rl_interface_fw_client.send_set_interfaces_interface_firewall_name(_rl_interface_fw_target.c_str(), 
									fw_iter->_firewall_name, 
									fw_iter->_filter, 
									interface,
									callback(this, &XrlRLSerialNode::firewall_cb));
    //    XLOG_INFO("RLSerialNode::commit, adding fw: %s to %s", fw_iter->_firewall_name.c_str(), fw_iter->_filter.c_str());
    ++fw_iter;
  }

  fw_iter = fw_del_coll.begin();
  while (fw_iter != fw_del_coll.end()) {
    sprintf(buf, "%d", fw_iter->_vif);
    string interface = fw_iter->_interface + "." + string(buf);
    _rl_interface_fw_client.send_delete_interfaces_interface_firewall_name(_rl_interface_fw_target.c_str(), 
									fw_iter->_firewall_name, 
									fw_iter->_filter, 
									interface,
									callback(this, &XrlRLSerialNode::firewall_cb));
    //    XLOG_INFO("RLSerialNode::commit, deleting fw: %s to %s", fw_iter->_firewall_name.c_str(), fw_iter->_filter.c_str());
    ++fw_iter;
  }
  return XrlCmdError::OKAY();
}

/**
 *
 */
void
XrlRLSerialNode::firewall_cb(const XrlError &xrl_error)
{
  UNUSED(xrl_error);
}

/**
 *
 */
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_encapsulation(
							 //Input values,
							 const string &serial,
							 const string &encapsulation)
{
  return _rl_serial_node.set_encap(serial, encapsulation);
}

/**
 *
 */
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_delete_proto( 
					    //Input values,
					    const string &serial,
					    const string &proto)
{
  return _rl_serial_node.delete_proto(serial, proto);
}


/**
 *
 */
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_delete_vif(
					  //Input values,
					  const string &serial,
					  const string &proto,
					  const uint32_t &vif)
{
  return _rl_serial_node.delete_vif(serial, proto, vif);
}

/**
 *	
 */
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_create_address4(
			      //Input values,
			      const string &serial,
			      const string &proto,
			      const uint32_t &vif,
			      const IPv4 &address,
			      const IPv4 &remote_address,
			      const uint32_t &prefix_length)
{
  return _rl_serial_node.create_address(serial, proto, vif, address, remote_address, prefix_length);
}

XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_delete_address4(
			      //Input values, 
			      const string &serial,
			      const string &proto,
			      const uint32_t &vif,
			      const IPv4 &address,	
			      const IPv4 &remote_address,
			      const uint32_t &prefix_length)
{
  return _rl_serial_node.delete_address(serial, proto, vif, address, remote_address, prefix_length);
}

XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_dlci(
					//Input values,
					const string &serial,
					const string &proto,
					const uint32_t &vif,
					const uint32_t &dlci)
{
  return _rl_serial_node.set_dlci(serial, proto, vif, dlci);
}

XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_cir(
					//Input values,
					const string &serial,
					const string &proto,
					const uint32_t &vif,
					const uint32_t &cir)
{
  return _rl_serial_node.set_cir(serial, proto, vif, cir);
}

XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_burst(
					  //Input values,
					  const string &serial,
					  const string &proto,
					  const uint32_t &vif,
					  const uint32_t &burst)
{
  return _rl_serial_node.set_burst(serial, proto, vif, burst);
}

XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_excess(
						  //Input values,
					   const string &serial,
					   const string &proto,
					   const uint32_t &vif,
					   const uint32_t &excess)
{
  return _rl_serial_node.set_excess(serial, proto, vif, excess);
}

XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_tx_inverse_arp(
						   //Input values,
						   const string &serial,
						   const string &proto,
						   const uint32_t &vif,
						   const uint32_t &tx_inverse_arp)
{
  return _rl_serial_node.set_tx_inverse_arp(serial, proto, vif, tx_inverse_arp);
}

XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_rx_inverse_arp(
						   //Input values,
						   const string &serial,
						   const string &proto,
						   const uint32_t &vif,
						   const string &rx_inverse_arp)
{
  return _rl_serial_node.set_rx_inverse_arp(serial, proto, vif, rx_inverse_arp);
}


/**
 *
 */	
XrlCmdError   
XrlRLSerialNode::rl_serial_0_1_set_card_type(
			       //Input values,
			       const string &serial, 
			       const string &card_type)
{
  UNUSED(serial);
  UNUSED(card_type);

  return XrlCmdError::OKAY();
  //return _rl_serial_node.set_card(serial, card_type);
}

/**
 *
 */	
XrlCmdError   
XrlRLSerialNode::rl_serial_0_1_delete_physical(
				 //Input values,
				 const string &serial)
{
  return _rl_serial_node.delete_physical(serial);
}

XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_t1_options_lbo(
						  //Input values,
						  const string &serial, 
						  const string &lbo)
{
  return _rl_serial_node.set_t1_lbo(serial, lbo);
}
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_t1_options_line_coding(
							  //Input values,
							  const string &serial, 
							  const string &line_coding)
{
  return _rl_serial_node.set_t1_line_coding(serial, line_coding);
}
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_t1_options_timeslots(
							//Input values,
							const string &serial, 
							const uint32_t &start,
							const uint32_t &stop)
{
  return _rl_serial_node.set_t1_timeslots(serial, start, stop);
}
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_t1_options_mtu(
						  //Input values,
						  const string &serial, 
						  const uint32_t &mtu)
{
  return _rl_serial_node.set_t1_mtu(serial, mtu);
}
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_t1_options_clock(
						    //Input values,
						    const string &serial, 
						    const string &clock)
{
  return _rl_serial_node.set_t1_clock(serial, clock);
}
  

XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_e1_options_framing(
				   //Input values,
				   const string &serial, 
				   const string &framing)
{
  return _rl_serial_node.set_e1_framing(serial, framing);
}
  XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_e1_options_line_coding(
				       //Input values,
				       const string &serial, 
				       const string &line_coding)
{
  return _rl_serial_node.set_e1_line_coding(serial, line_coding);
}
  XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_e1_options_timeslots(
				     //Input values,
				     const string &serial, 
				     const uint32_t &start,
				     const uint32_t &stop)
{
  return _rl_serial_node.set_e1_timeslots(serial, start, stop);
}
  XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_e1_options_mtu(
			       //Input values,
			       const string &serial, 
			       const uint32_t &mtu)
{
  return _rl_serial_node.set_e1_mtu(serial, mtu);
}
  XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_e1_options_clock(
				 //Input values,
				 const string &serial, 
				 const string &clock)
{
  return _rl_serial_node.set_e1_clock(serial, clock);
}
  




XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_t3_options_framing(
				   //Input values,
				   const string &serial, 
				   const string &framing)
{
  return _rl_serial_node.set_t3_framing(serial, framing);
}
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_t3_options_line_coding(
							  //Input values,
							  const string &serial, 
							  const string &line_coding)
{
  return _rl_serial_node.set_t3_line_coding(serial, line_coding);
}
  
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_e3_options_framing(
				   //Input values,
				   const string &serial, 
				   const string &framing)
{
  return _rl_serial_node.set_e3_framing(serial, framing);
}
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_e3_options_line_coding(
							  //Input values,
							  const string &serial, 
							  const string &line_coding)
{
  return _rl_serial_node.set_e3_line_coding(serial, line_coding);
}
  



  /**
   *
   */
  XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_ppp_authentication(
				     //Input values,
				     const string &serial, 
				     const string &proto,
				     const string &type)
{
  return _rl_serial_node.set_ppp_authentication(serial, proto, type);
}
  XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_ppp_auth_parameters_user_id(
					      //Input values,
					      const string &serial, 
					      const string &proto,
					      const string &user_id)
{
  return _rl_serial_node.set_auth_parameters_user_id(serial, proto, user_id);
}
  XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_ppp_auth_parameters_password(
					       //Input values,
					       const string &serial, 
					       const string &proto,
					       const string &password)
{
  return _rl_serial_node.set_auth_parameters_password(serial, proto, password);
}
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_ppp_auth_parameters_system_name(
							 //Input values,
							 const string &serial, 
							 const string &proto,
							 const string &system_name)
{
  return _rl_serial_node.set_auth_parameters_system_name(serial, proto, system_name);
}


/**
 *
 */
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_cisco_hdlc_keepalives(
									  //Input values,
						     const string &serial, 
						     const string &proto,
						     const string &require_rx,
						     const uint32_t &timer)
{
  return _rl_serial_node.set_keepalives(serial, proto, require_rx, timer);
}

/**
 *
 */
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_frame_relay_signaling(
					  //Input values,
					  const string &serial, 
					  const string &proto,
					  const string &signaling)
{
  return _rl_serial_node.set_signaling(serial, proto, signaling);
}
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_frame_relay_signaling_options_n391dte(
							  //Input values,
							  const string &serial, 
							  const string &proto,
							  const uint32_t &n391dte)
{
  return _rl_serial_node.set_n391dte(serial, proto, n391dte);
}
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_frame_relay_signaling_options_n392dte(
							  //Input values,
							  const string &serial, 
							  const string &proto,
							  const uint32_t &n392dte)
{
  return _rl_serial_node.set_n392dte(serial, proto, n392dte);
}
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_frame_relay_signaling_options_n393dte(
							  //Input values,
							  const string &serial, 
							  const string &proto,
							  const uint32_t &n393dte)
{
  return _rl_serial_node.set_n393dte(serial, proto, n393dte);
}
XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_frame_relay_signaling_options_t391dte(
							  //Input values,
							  const string &serial, 
							  const string &proto,
							  const uint32_t &t391dte)
{
  return _rl_serial_node.set_t391dte(serial, proto, t391dte);
}



XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_set_serial_firewall_name( 
						   //Input values,
						     const string &serial,
						     const uint32_t &vif,
						     const string &proto,
						     const string &firewall_name,
						     const string &filter)
{
  return _rl_serial_node.set_firewall(serial, vif, proto, firewall_name, filter);
}

XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_delete_serial_firewall_name( 
						      //Input values,
						     const string &serial,
						     const uint32_t &vif,
						     const string &proto,
						     const string &firewall_name,
						     const string &filter)
{
  return _rl_serial_node.delete_firewall(serial, vif, proto, firewall_name, filter);
}


void 
XrlRLSerialNode::send_transaction_cb(const XrlError &xrl_error)

{
  UNUSED(xrl_error);

}


XrlCmdError 
XrlRLSerialNode::rl_serial_0_1_get_serial_data(
					       //Input values,
					       const string &name,
					       //output values,
					       XrlAtomList &rules)
{
  return _rl_serial_node.get_serial_data(name, rules);

}
