// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
// vim:set sts=4 ts=8:

// Copyright (c) 2001-2005 International Computer Science Institute
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software")
// to deal in the Software without restriction, subject to the conditions
// listed in the XORP LICENSE file. These conditions include: you must
// preserve this copyright notice, and you cannot mention the copyright
// holders in advertising related to the Software without their permission.
// The Software is provided WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED. This
// notice is a summary of the XORP LICENSE file; the license in that file is
// legally binding.

// #define DEBUG_LOGGING
// #define DEBUG_PRINT_FUNCTION_NAME

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string>
#include <set>
#include <list>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "rl_serial/rl_serial_module.h"

#include "librl_common/rl_str_proc.hh"

#include "libxorp/debug.h"
#include "libxorp/xlog.h"

#include "libxorp/status_codes.h"
#include "libxorp/eventloop_factory.hh"
#include "libxipc/xrl_std_router.hh"
#include "libxorp/tlv.hh"

#include "xrl/interfaces/rl_serial_xif.hh"

using namespace std;

/**
 * 
 */
class GetSerialList {
public:
    GetSerialList(XrlStdRouter& xrl_router, const string &name, const string &interface)
	: _xrl_router(xrl_router),
	  _done(false), _fail(false), _name(name), _interface(interface)
    {
    }

    void start() {
	//	printf("GetSerialList::start():\n");
	XrlRlSerialV0p1Client rl_serial(&_xrl_router);
	rl_serial.send_get_serial_data("rl_serial",
				       _name, 
				       callback(this, &GetSerialList::response));
    }

    bool busy() {
	return !_done;
    }

    bool fail() {
	return _fail;
    }

    list<string>& get() {
	return _nlist;
    }

private:
    void response(const XrlError& error, const XrlAtomList *atomlist) {
	//	printf("GetSerialList::response\n");
	_done = true;
	if (XrlError::OKAY() != error) {
	    XLOG_WARNING("Attempt to get serial list failed");
	    _fail = true;
	    return;
	}
	const size_t size = atomlist->size();
	for (size_t i = 0; i < size; i++) {
	    string foo = atomlist->get(i).str().c_str();
	    string token("%3B");

	    printf("%s\n", foo.c_str());
	    _nlist.push_back(atomlist->get(i).str());
	}

	//	printf("GetSerialList::response.... Leaving....\n");
    }

private:
    XrlStdRouter &_xrl_router;
    bool _done;
    bool _fail;
    string _name;

    list<string> _nlist;

    string _interface;
};


/**
 * 
 */
class GetSerialInterfaceList {
public:
    GetSerialInterfaceList(XrlStdRouter& xrl_router, const string &name)
	: _xrl_router(xrl_router),
	  _done(false), _fail(false), _name(name)
    {
    }

    void start() {
	//	printf("GetSerialList::start():\n");
	XrlRlSerialV0p1Client rl_serial_interface(&_xrl_router);
	rl_serial_interface.send_get_serial_data("rl_interface_serial",
						_name, 
						callback(this, &GetSerialInterfaceList::response));
    }

    bool busy() {
	return !_done;
    }

    bool fail() {
	return _fail;
    }

    string get() {return _interface;}

private:
    void response(const XrlError& error, const XrlAtomList *atomlist) {
	//	printf("GetSerialList::response\n");
	_done = true;
	if (XrlError::OKAY() != error) {
	    XLOG_WARNING("Attempt to get interface serial list failed");
	    _fail = true;
	    return;
	}
	const size_t size = atomlist->size();
	for (size_t i = 0; i < size; i++) {
	    string foo = atomlist->get(i).str().c_str();
	    _interface = foo;
	    //	    _nlist.push_back(atomlist->get(i).str());
	}
	//	printf("GetSerialList::response.... Leaving....\n");
    }
private:
    XrlStdRouter &_xrl_router;
    bool _done;
    bool _fail;
    string _name;


    string _interface;
};


/**
 * The base class that needs to be implemented by all print routines.
 */ 
class Output {
public:
    virtual ~Output()
    {}

    virtual bool begin() {
	//	printf("Rule\n");
	return true;
    }

    void print_first_line(const uint32_t& number) {
	printf("%d", number);
	printf("\n");
    }

    virtual bool print(const uint32_t& number) {
	print_first_line(number);
	return true;
    }
};

int
usage(const char *myname)
{
    fprintf(stderr, "usage: %s\n", myname);

    return -1;
}

int 
main(int argc, char **argv)
{
    if (argc < 2) {
	printf("Usage: print_serial_rules name\n");
	exit(1);
    }

    XorpUnexpectedHandler x(xorp_unexpected_handler);
    //
    // Initialize and start xlog
    //
    xlog_init(argv[0], NULL);
    xlog_set_verbose(XLOG_VERBOSE_LOW);		// Least verbose messages
    // XXX: verbosity of the error messages temporary increased
    xlog_level_set_verbose(XLOG_LEVEL_ERROR, XLOG_VERBOSE_HIGH);
    xlog_add_default_output();
    xlog_start();

    Output *output = new Output();

    string name(argv[1]);

    try {
	EventLoop& eventloop = *EventLoopFactory::instance().create(eventloop_st);
	XrlStdRouter xrl_router(eventloop, "print_serial_rules");

	debug_msg("Waiting for router");
	xrl_router.finalize();
	wait_until_xrl_router_is_ready(eventloop, xrl_router);
	debug_msg("\n");

	//now get the interface information;
	GetSerialInterfaceList get_serial_interface_list(xrl_router, name);
	get_serial_interface_list.start();
	while(get_serial_interface_list.busy())
	    eventloop.run();


	GetSerialList get_serial_list(xrl_router, name, get_serial_interface_list.get());
	get_serial_list.start();
	while(get_serial_list.busy())
	    eventloop.run();



	if (get_serial_list.fail()) {
	    XLOG_ERROR("Failed to get serial list");
	    return -1;
	}

	if (!output->begin())
	    return -1;
	delete output;

    } catch (...) {
	xorp_catch_standard_exceptions();
    }

    xlog_stop();
    xlog_exit();

    return 0;
}
