/*
 * Module: rl_serial_node.hh
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Michael Larson
 * Date: 2005
 * Description:
 *
 * **** End License ****
 *
 */

#ifndef __RL_SERIAL_NODE_HH__
#define __RL_SERIAL_NODE_HH__

#include <string>
#include <vector>
#include <map>
#include <set>
#include <libxipc/xrl_error.hh>
#include <libxorp/ipv4.hh>
#include <libxorp/ipv4net.hh>


/**
 *
 **/
class IfaceDelta
{
public:
  IfaceDelta(const string &serial,
	     const uint32_t &vif,
	     const IPv4 &address,
	     const uint32_t &prefix_length) :
    _serial(serial),
    _vif(vif),
    _address(address),
    _prefix_length(prefix_length) {}
  string _serial;
  uint32_t _vif;
  IPv4 _address;
  uint32_t _prefix_length;
};

/**
 *
 **/
class FirewallDelta
{
public:
  FirewallDelta(const string &interface, const uint32_t &vif, const string &firewall, const string &name);

  bool
  operator==(const FirewallDelta &delta) const;

public:
  string _interface;
  uint32_t _vif;
  string _firewall_name;
  string _filter;
};


/**
 *
 **/
class VifData {
public:
  typedef map<uint32_t, VifData> VifDataColl;
  typedef map<uint32_t, VifData>::iterator VifDataIter;

  typedef enum {NEW, CURRENT, CHANGED, DELETE} VifState;

public:
  VifData();

  IPv4 _address;
  uint32_t _prefix_length;
  IPv4 _pointtopoint;

  //used for passing parameters to the firewall module
  string _in_firewall_name;
  string _out_firewall_name;
  string _local_firewall_name;
  uint32_t _dlci;
  uint32_t _cir;
  uint32_t _burst;
  uint32_t _excess;
  uint32_t _tx_inverse_arp;
  string _rx_inverse_arp;

  VifState _state;
};

/**
 * Set up to handle the complexities of managing the different logical vifs
 *
 **/
class VifManager {
public:
  typedef vector<IfaceDelta> IfaceDeltaColl;
  typedef vector<IfaceDelta>::iterator IfaceDeltaIter;

public:
  bool
  get_vif(const string &proto, const uint32_t &vif, VifData::VifDataIter &iter);

  void
  erase(const string &proto, const uint32_t &vif);

  void
  erase(const string &proto);

  VifData::VifDataColl 
  expose(const string &proto);

  void
  commit(const string &serial, const string &proto, IfaceDeltaColl &add_coll, IfaceDeltaColl &del_coll);

private:
  VifData::VifDataColl _ppp_vif_coll;
  VifData::VifDataColl _fr_vif_coll;
  VifData::VifDataColl _hdlc_vif_coll;
};

/**
 *
 **/
class SerialData
{
public:
  typedef map<string, SerialData> SerialDataColl;
  typedef map<string, SerialData>::iterator SerialDataIter;

  typedef vector<FirewallDelta> FirewallDeltaColl;
  typedef vector<FirewallDelta>::iterator FirewallDeltaIter;
  typedef vector<FirewallDelta>::const_iterator FirewallDeltaConstIter;

public:
  //  SerialData();
  SerialData(const string &card);

  bool   _config_changed;

  string _encapsulation;
  //note proto and encap should always agree on active configurations but are different values
  string _proto; //hdlc,ppp,frame

  VifManager _vif_manager;

  //track node changes across vifs
  bool _proto_transition;

  string _card;     //major card type
  string _card_name; //specific card type
  string _physical;
  string _pci_bus; //this is assigned
  string _pci_slot; //this is assigned
  string _cpu;      //assigned
  unsigned _port;     //port this interface is assigned to 
  unsigned _num_ports; //number of ports on this card

  //t1
  string _t1_lbo;
  string _t1_line_coding;
  string _t1_timeslots;
  uint32_t _t1_mtu;
  string _t1_clock;

  //e1
  string _e1_framing;
  string _e1_line_coding;
  string _e1_timeslots;
  uint32_t _e1_mtu;
  string _e1_clock;

  //t3
  string _t3_line_coding;
  string _t3_framing;

  //e3
  string _e3_line_coding;
  string _e3_framing;

  //ppp
  string _authentication;
  string _user_id;
  string _password;
  string _system_name;

  //cisco hdlc
  string   _require_rx;
  uint32_t _timer;

  //frame relay
  string _signaling;
  uint32_t _n391dte;
  uint32_t _n392dte;
  uint32_t _n393dte;
  uint32_t _t391dte;

  //keeps track of the active firewalls from the last transaction
  FirewallDeltaColl _fw_active_coll;
};


/**
 *
 **/
class RLSerialNode
{
public:
  RLSerialNode();
  ~RLSerialNode();

  XrlCmdError 
  reset(VifManager::IfaceDeltaColl &iface_coll, SerialData::FirewallDeltaColl &fw_coll);

  XrlCmdError
  set_vlan_id(
	      //Input values,
	      const string &interface,
	      const string &vif,
	      const uint32_t &vlan_id);

  XrlCmdError
  delete_vlan_id(
	      //Input values,
	      const string &interface,
	      const string &vif,
	      const uint32_t &vlan_id);

  XrlCmdError
  delete_serial(
		//Input values,
		const string &serial,
		//Output values,
		VifManager::IfaceDeltaColl &coll);

  XrlCmdError
  reset_serial(
		//Input values,
		const string &serial);

  XrlCmdError
  set_encap(
	    //Input values,
	    const string &serial,
	    const string &encapsulation);

  XrlCmdError
  delete_proto(
	       //Input values,
	       const string &serial,
	       const string &proto);

  XrlCmdError
  delete_vif(
	     //Input values,
	     const string &serial,
	     const string &proto,
	     const uint32_t &vif);

  XrlCmdError
  start_serial_transaction(
			    //Input values,
			   const string &serial);

  bool
  config_change(
		const string &serial);

  XrlCmdError
  pre_serial_commit(
		    const string &serial,
		    VifManager::IfaceDeltaColl &iface_del_coll);

  XrlCmdError
  commit_serial_transaction(
			     //Input values,
			    const string &serial,
			    VifManager::IfaceDeltaColl &iface_add_coll,
			    SerialData::FirewallDeltaColl &fw_add_coll,
			    SerialData::FirewallDeltaColl &fw_del_coll);
  
  XrlCmdError
  post_serial_commit(
		     //Input values,
		     const string &serial);
  
  XrlCmdError
  create_address(
		 //Input values,
			     const string &serial,
			     const string &proto,
			     const uint32_t &vif, 
			     const IPv4 &address,
			     const IPv4 &remote_address,
			     const uint32_t &prefix_length);

  XrlCmdError
  delete_address(
		 //Input values,
			     const string &serial,
			     const string &proto,
			     const uint32_t &vif, 
			     const IPv4 &address,
			     const IPv4 &remote_address,
			     const uint32_t &prefix_length);


  XrlCmdError
  set_dlci(
	   //Input values,
	   const string &serial,
	   const string &proto,
	   const uint32_t &vif,
	   const uint32_t &dlci);
  
  XrlCmdError
  set_cir(
	   //Input values,
	   const string &serial,
	   const string &proto,
	   const uint32_t &vif,
	   const uint32_t &cir);
  
  XrlCmdError
  set_burst(
	   //Input values,
	   const string &serial,
	   const string &proto,
	   const uint32_t &vif,
	   const uint32_t &burst);
  
  XrlCmdError
  set_excess(
	   //Input values,
	   const string &serial,
	   const string &proto,
	   const uint32_t &vif,
	   const uint32_t &excess);
  
  XrlCmdError
  set_tx_inverse_arp(
		     //Input values,
		     const string &serial,
		     const string &proto,
		     const uint32_t &vif,
		     const uint32_t &tx_inverse_arp);
  
  XrlCmdError
  set_rx_inverse_arp(
		     //Input values,
		     const string &serial,
		     const string &proto,
		     const uint32_t &vif,
		     const string &rx_inverse_arp);

  XrlCmdError
  set_card(
	   //Input values,
	   const string &serial,
	   const string &card);

  XrlCmdError
  delete_physical(
	    //Input values,
	    const string &serial);

  
  XrlCmdError 
  set_t1_lbo(
		   //Input values,
		   const string &serial, 
		   const string &lbo);
  XrlCmdError 
  set_t1_line_coding(
			   //Input values,
			   const string &serial, 
			   const string &line_coding);
  XrlCmdError 
  set_t1_timeslots(
			 //Input values,
			 const string &serial, 
			 const uint32_t &start,
			 const uint32_t &stop);
  XrlCmdError 
  set_t1_mtu(
		   //Input values,
		   const string &serial, 
		   const uint32_t &mtu);
  XrlCmdError 
  set_t1_clock(
		     //Input values,
		     const string &serial, 
		     const string &clock);

  XrlCmdError 
  set_e1_framing(
		       //Input values,
		       const string &serial, 
		       const string &line);
  XrlCmdError 
  set_e1_line_coding(
			   //Input values,
			   const string &serial, 
			   const string &line_coding);
  XrlCmdError 
  set_e1_timeslots(
			 //Input values,
			 const string &serial, 
			 const uint32_t &start,
			 const uint32_t &stop);
  XrlCmdError 
  set_e1_mtu(
		   //Input values,
		   const string &serial, 
		   const uint32_t &mtu);
  XrlCmdError 
  set_e1_clock(
		     //Input values,
		     const string &serial, 
		     const string &clock);
  
  XrlCmdError 
  set_t3_framing(
		       //Input values,
		       const string &serial, 
		       const string &line);
  XrlCmdError 
  set_t3_line_coding(
			   //Input values,
			   const string &serial, 
			   const string &line_coding);
  XrlCmdError 
  set_e3_framing(
		       //Input values,
		       const string &serial, 
		       const string &line);
  XrlCmdError 
  set_e3_line_coding(
			   //Input values,
			   const string &serial, 
			   const string &line_coding);

  XrlCmdError 
  set_ppp_authentication(
			 //Input values,
			 const string &serial, 
			 const string &proto,
			 const string &ppp);
  XrlCmdError 
  set_auth_parameters_user_id(
			      //Input values,
			      const string &serial, 
			      const string &proto,
			      const string &user_id);
  XrlCmdError 
  set_auth_parameters_password(
		     //Input values,
			       const string &serial, 
			       const string &proto,
			       const string &password);
  XrlCmdError 
  set_auth_parameters_system_name(
		     //Input values,
				  const string &serial, 
				  const string &proto,
				  const string &system_name);

  XrlCmdError
  set_keepalives(
		    //Input values,
		    const string &serial,
		    const string &proto,
		    const string &require_rx,
		    const uint32_t &timer);


  XrlCmdError
  set_signaling(
		    //Input values,
		    const string &serial,
		    const string &proto,
		    const string &signaling);

  XrlCmdError
  set_n391dte(
		    //Input values,
		    const string &serial,
		    const string &proto,
		    const uint32_t &n391dte);

  XrlCmdError
  set_n392dte(
		    //Input values,
		    const string &serial,
		    const string &proto,
		    const uint32_t &n392dte);

  XrlCmdError
  set_n393dte(
		    //Input values,
		    const string &serial,
		    const string &proto,
		    const uint32_t &n393dte);

  XrlCmdError
  set_t391dte(
		    //Input values,
		    const string &serial,
		    const string &proto,
		    const uint32_t &t391dte);

  XrlCmdError
  get_serial_data(
		  //Input values,
		  const string &name,
		  //Output values,
		  XrlAtomList &rules);

  XrlCmdError
  set_firewall(
	       //Input values,
	       const string &serial,
	       const uint32_t &vif,
	       const string &proto,
	       const string &firewall_name,
	       const string &filter);

  XrlCmdError
  delete_firewall(
	       //Input values,
	       const string &serial,
	       const uint32_t &vif,
	       const string &proto,
	       const string &firewall_name,
	       const string &filter);

private: //methods
  bool
  validate(const string &serial, string &reason);

  bool
  pci_bus_find(const string &serial, string &slot, string &pci, unsigned &port, unsigned &num_ports, string &cpu, string &card_type);

  string
  find_card_type(const string &serial);

  SerialData::FirewallDeltaColl
  fw_diff(const SerialData::FirewallDeltaColl &coll1, const SerialData::FirewallDeltaColl &coll2);

  XrlCmdError
  write_ppp_conf(const string &serial, long int index, const string &ip_line, const string &file);

  XrlCmdError
  write_hdlc_conf(const string &serial, long int index, const string &file);

  XrlCmdError
  write_frame_conf(const string &serial, long int index, const string &file);

private: //variables
  string _wanpipe_interfaces_dir;
  string _wanpipe_file;

  SerialData::SerialDataColl _pre_trans_serial_data_coll;

  SerialData::SerialDataColl _serial_data_coll;

  VifManager::IfaceDeltaColl _iface_add_coll;
};

#endif //__RL_INTERFACES_RL_SERIAL_NODE_HH__

