/*
 * Module: rl_serial_node.cc
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Michael Larson
 * Date: 2005
 * Description:
 *
 * **** End License ****
 *
 */

#include <string>
#include <dirent.h>
#include "rl_serial_module.h"
#include "libxipc/xrl_atom.hh"
#include "libxorp/xlog.h"
#include "librl_common/rl_interface.hh"
#include "librl_common/rl_fileaccess.hh"
#include "librl_common/rl_command.hh"
#include "librl_common/rl_str_proc.hh"
#include "rl_serial_node.hh"
#include "libxorp/xlog.h"

#define	EXEC_RET(_cmd) do { \
    if (rl_command::execute(_cmd) == false) { \
      string cmd(_cmd); \
      XLOG_ERROR("rl_interface failed to execute command: %s", cmd.c_str()); \
      return XrlCmdError::COMMAND_FAILED(); \
    } \
  } while(0)

#define	EXEC(_cmd) do { \
    if (rl_command::execute(_cmd) == false) { \
      string cmd(_cmd); \
      XLOG_ERROR("rl_interface failed to execute command: %s", cmd.c_str()); \
    } \
  } while(0)

/**
 *
 */
FirewallDelta::FirewallDelta(const string &interface, const uint32_t &vif, const string &firewall_name, const string &filter) :
  _interface(interface),
  _vif(vif),
  _firewall_name(firewall_name),
  _filter(filter)
{
}

/**
 *
 */
bool
FirewallDelta::operator==(const FirewallDelta &delta) const
{
  return (_interface == delta._interface &&
	  _vif == delta._vif &&
	  _firewall_name == delta._firewall_name &&
	  _filter == delta._filter);
}

/**
 *
 */
VifData::VifData() :
  _prefix_length(32),
  _dlci(16),
  _cir(0),
  _burst(0),
  _excess(0),
  _tx_inverse_arp(0),
  _rx_inverse_arp("disable"),
  _state(NEW)
{
}

/**
 *
 */
SerialData::SerialData(const string &card) : 
  _config_changed(false),
  _proto_transition(false),
  _card(card),
  _physical("NONE"),
  _t1_lbo("0-110ft"),
  _t1_line_coding("b8zs"),
  _t1_timeslots("ALL"),
  _t1_mtu(1500),
  _t1_clock("external"),
  _e1_framing("g704"),
  _e1_line_coding("hdb3"),
  _e1_timeslots("ALL"),
  _e1_mtu(1500),
  _e1_clock("external"),
  _t3_line_coding("b3zs"),
  _t3_framing("c-bit"),
  _e3_line_coding("hdb3"),
  _e3_framing("g751"),
  _require_rx("enable"),
  _timer(10),
  _signaling("auto"),
  _n391dte(6),
  _n392dte(6),
  _n393dte(4),
  _t391dte(10)
{

}

/**
 *
 */
RLSerialNode::RLSerialNode() :
  _wanpipe_interfaces_dir("/etc/wanpipe/interfaces/"),
  _wanpipe_file("/etc/wanpipe/wanpipe")
{
  //set up 8021q support in the kernel
  EXEC("modprobe 8021q");
  //  reset();
}

/**
 *
 */
RLSerialNode::~RLSerialNode()
{
}

/**
 *
 */
XrlCmdError
RLSerialNode::reset(VifManager::IfaceDeltaColl &iface_coll, SerialData::FirewallDeltaColl &fw_coll)
{
  XLOG_INFO("RLSerialNode::reset(), resetting module");

  //copy over all default files

  //remove any files

  //shut down interfaces
  EXEC("/etc/wanpipe/scripts/vyatta-wan stop");

  //remove all filelocks, if any
  rl_fileaccess::init(_wanpipe_interfaces_dir + "*");
  rl_fileaccess::init(_wanpipe_file + "*.conf");

  //stop any processes
  /* kill all vrrpd processes */

  //remove wanpipe files
  EXEC("rm -f " + _wanpipe_interfaces_dir + "*");
  //need to start wanrouter now so that default configs are created
  EXEC("/etc/wanpipe/scripts/vyatta-wan start");

  //now return up the interfaces and fw that need to be reset via the appropriate components
 
  //first move the new default files to the right location
  SerialData::SerialDataIter iter;
  for(iter = _serial_data_coll.begin(); iter!=_serial_data_coll.end(); ++iter) {
    fw_coll = iter->second._fw_active_coll;
    iter->second._fw_active_coll.erase(iter->second._fw_active_coll.begin(), iter->second._fw_active_coll.end());

    iter->second._vif_manager.erase(iter->second._encapsulation);
    VifManager::IfaceDeltaColl add_coll; //only care about the deletes at this point
    iter->second._vif_manager.commit(iter->first, iter->second._encapsulation, add_coll, iface_coll);
  }

   // bring up ppp
   //EXEC("/sbin/pallon");

  return XrlCmdError::OKAY();
}


/**
 *
 */
XrlCmdError
RLSerialNode::set_vlan_id(
			  //Input values,
			  const string &interface,
			  const string &vif,
			  const uint32_t &vlan_id)
{
  UNUSED(vif);

  char buf[40];
  sprintf(buf, "%d", vlan_id);

  EXEC_RET("vconfig add " + interface + " " + buf);
  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
RLSerialNode::delete_vlan_id(
				   //Input values,
				   const string &interface,
				   const string &vif,
				   const uint32_t &vlan_id)
{
  UNUSED(vif);

  char buf[40];

  sprintf(buf, "%d", vlan_id);
  EXEC_RET("vconfig rem " + interface + " " + buf);
  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
RLSerialNode::delete_serial(
	      //Input values,
	      const string &serial,
	      //Output values
	      VifManager::IfaceDeltaColl &coll)
{
  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter == _serial_data_coll.end())
    return XrlCmdError::OKAY();

  VifData::VifDataColl vif_coll = iter->second._vif_manager.expose(iter->second._proto);
  VifData::VifDataIter vif_iter;
  for(vif_iter = vif_coll.begin(); vif_iter != vif_coll.end(); ++vif_iter)
    coll.push_back(IfaceDelta(serial, vif_iter->first, vif_iter->second._address, vif_iter->second._prefix_length));

  //build line token
  long int index;
  const char num_str(serial[3]);
  index = strtol(&num_str, NULL, 10);
  index++;
  char buf[80];
  sprintf(buf, "%ld", index);
  string index_str(buf);

  //stop the driver first!
  EXEC("/etc/wanpipe/scripts/vyatta-wan stop " + index_str);

  //delete all interfaces file
  string files = _wanpipe_interfaces_dir + serial + "*";
  unlink(files.c_str());

  //clean entry in wanpipe file
  /*
    files = _wanpipe_file + index_str + ".conf";
    unlink(files.c_str());
  */
  //call firsttime script to reset router back to init state on deletion
  EXEC("/etc/wanpipe/scripts/vyatta-wan start " + index_str);
  _serial_data_coll.erase(serial);

  return XrlCmdError::OKAY();
}


/**
 *
 */
XrlCmdError
RLSerialNode::reset_serial(
	      //Input values,
	      const string &serial)
{
  //build line token
  long int index;
  const char num_str(serial[3]);
  index = strtol(&num_str, NULL, 10);
  index++;
  char buf[80];
  sprintf(buf, "%ld", index);
  string index_str(buf);

  //call firsttime script to reset router back to init state on deletion
  EXEC("/etc/wanpipe/scripts/vyatta-wan start " + index_str);
  return XrlCmdError::OKAY();
}


/**
 *
 */
XrlCmdError
RLSerialNode::set_encap(
			    //Input values,
			    const string &serial,
			    const string &encapsulation)
{
  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._encapsulation = encapsulation;
    iter->second._config_changed = true;
  }

  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
RLSerialNode::delete_vif(
			 //Input values,
			 const string &serial,
			 const string &proto,
			 const uint32_t &vif)
{
  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._vif_manager.erase(proto, vif);
    iter->second._config_changed = true;
  }
  return XrlCmdError::OKAY();
}


 /**
  *
  */
XrlCmdError
RLSerialNode::delete_proto(
			   const string &serial,
			   const string &proto)
{
  UNUSED(proto);
  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._proto_transition = false;
    iter->second._config_changed = true;
    if (iter->second._proto == proto) {
      //only clear this out if the proto wasn't set to a different proto
      iter->second._proto = "";
    }
    //always tell the vif manager though
    iter->second._vif_manager.erase(proto);
  }
  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
RLSerialNode::start_serial_transaction(
					    //Input values,
					   const string &serial)
{
  //copy the pre-transaction state in case we fail somewhere down the line...
  _pre_trans_serial_data_coll.erase(_pre_trans_serial_data_coll.begin(),
				    _pre_trans_serial_data_coll.end());
  _pre_trans_serial_data_coll = _serial_data_coll;

  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter == _serial_data_coll.end()) {
    string card_type = rl_interface::find_serial_card_type(serial);
    _serial_data_coll.insert(pair<string, SerialData>(serial, SerialData(card_type)));
  }
  else {
    iter->second._config_changed = false;
  }
  return XrlCmdError::OKAY();
}

/**
 *
 */
bool
RLSerialNode::config_change(
			    const string &serial)
{
  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    return iter->second._config_changed;
  }
  return true;
}

/**
 *
 */
XrlCmdError
RLSerialNode::pre_serial_commit(
			 const string &serial,
			 VifManager::IfaceDeltaColl &iface_del_coll)
{
  string reason;
  if (!validate(serial, reason)) {
    //failure, copy over the pre-transaction state
    _serial_data_coll.erase(_serial_data_coll.begin(), _serial_data_coll.end());
    _serial_data_coll = _pre_trans_serial_data_coll;
    XLOG_ERROR("RLSerialNode::pre_serial_commit: validation failed: %s", serial.c_str());
    return XrlCmdError::COMMAND_FAILED(reason);
  }

  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._vif_manager.commit(iter->first, iter->second._encapsulation, _iface_add_coll, iface_del_coll);
  }

  return XrlCmdError::OKAY();
}

int
myput(const string &str, FILE *strm)
{
	return fputs(str.c_str(), strm);
}

int
myput(const char *str, FILE *strm)
{
	return fputs(str, strm);
}

int
myput(char chr, FILE *strm)
{
	return fputc(chr, strm);
}

/**
 *
 */
XrlCmdError
RLSerialNode::commit_serial_transaction(
					     //Input values,
					    const string &serial,
					    VifManager::IfaceDeltaColl &iface_add_coll,
					    SerialData::FirewallDeltaColl &fw_add_coll,
					    SerialData::FirewallDeltaColl &fw_del_coll)
{
  bool config_success = false;

  //first validate that the logical name is the next in the configuration
  long index = strtol(serial.substr(3).c_str(), NULL, 10);
  /*
      if ((unsigned(index)+1) != _serial_data_coll.size()) {
    _serial_data_coll.erase(serial);
    char buf[512];
    sprintf(buf, "The serial interface name should be the next logical name: wan%d\n", _serial_data_coll.size());
    XLOG_ERROR(buf);
    return XrlCmdError::COMMAND_FAILED(buf);
  }
  */
  char buf[80];
  sprintf(buf, "%ld", ++index);
  string index_str(buf);
  XLOG_INFO("RLSerialNode::commit_serial_transaction(): %s, %c, %ld, %s", serial.c_str(), serial[3], index, index_str.c_str());

  iface_add_coll = _iface_add_coll;
  _iface_add_coll.clear();

  //stop the driver
  EXEC("/etc/wanpipe/scripts/vyatta-wan stop " + index_str);
  EXEC("rm -f " + _wanpipe_interfaces_dir + serial + "*");

  //first move the new default files to the right location
  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    string wanpipe_file(_wanpipe_file); //set to 1 for now--need to check with mike
    sprintf(buf, "%ld.conf", index);
    wanpipe_file += buf;

    //now resolve the slot location
    string pci_slot, pci_bus, cpu, card_name;
    unsigned port = 1, num_ports = 0;
    if (pci_bus_find(wanpipe_file, pci_slot, pci_bus, port, num_ports, cpu, card_name) != true) {
      XLOG_ERROR("RLSerialNode: failed to find card bus and slot: %s (%s)", serial.c_str(), wanpipe_file.c_str());
#ifdef	RJB
      string temp = "Failed to find serial card and slot: " + serial + "\n";
      _serial_data_coll.erase(serial);
      return XrlCmdError::COMMAND_FAILED(temp);
#endif
    }
    XLOG_INFO("rl_serial_node.cc::commit_serial_transaction() assigning bus: %s, slot: %s, port: %d", pci_bus.c_str(), pci_slot.c_str(), port);

    //need to maintain the slot/bus details
    iter->second._pci_bus = pci_bus;
    iter->second._pci_slot = pci_slot;
    iter->second._port = port;
    iter->second._num_ports = num_ports;
    iter->second._cpu = cpu;
    iter->second._card_name = card_name;

    VifData::VifDataColl vif_coll = iter->second._vif_manager.expose(iter->second._encapsulation);
    VifData::VifDataIter s_iter = vif_coll.begin();

    string ip_line;
    for(s_iter=vif_coll.begin(); s_iter!=vif_coll.end(); ++s_iter) {
	config_success = true; //means that we have a vif for now

	char buf[80];
	string wan_ip_filename(_wanpipe_interfaces_dir + serial);
	sprintf(buf, ".%d", s_iter->first);
	wan_ip_filename += buf;

	rl_fileaccess fa(wan_ip_filename);
	FILE *fd_rd, *fd_wr;
	bool success = fa.get(fd_rd, fd_wr);
	if (success == false) {
	  XLOG_ERROR("rl_interfaces_node: failed to open file: %s", wan_ip_filename.c_str());
	  _serial_data_coll.erase(serial);
	  return XrlCmdError::COMMAND_FAILED();
	}

      if (iter->second._encapsulation == "ppp") {
  	ip_line = s_iter->second._address.str() + ':' + s_iter->second._pointtopoint.str() + '\n';
      } else {
  	IPv4 ip("255.255.255.255");
  	IPv4 mask = ip.make_prefix(s_iter->second._prefix_length);

  	myput("DEVICE=" + serial + buf + "\n", fd_wr); //vif ending
  	myput("IPADDR=" + s_iter->second._address.str() + "\n", fd_wr);
  	myput("NETMASK=" + mask.str() + "\n", fd_wr);
  	myput("POINTOPOINT=" + s_iter->second._pointtopoint.str() + "\n", fd_wr);
  	myput("ONBOOT=yes\n", fd_wr);
      }
    }

    //need to select the right reference
    string ref_conf_file;
    if (iter->second._encapsulation == "ppp") {
      config_success = true; //we don't care if we have a vif: the ip address can be specified by the peer
      write_ppp_conf(serial, index, ip_line, wanpipe_file);
    }
    else if (iter->second._encapsulation == "frame-relay") {
      write_frame_conf(serial, index, wanpipe_file);
    }
    else if (iter->second._encapsulation == "cisco-hdlc") {
      write_hdlc_conf(serial, index, wanpipe_file);
    }
    else {
      _serial_data_coll.erase(serial);
      XLOG_ERROR("Invalid protocol type specified");
      return XrlCmdError::COMMAND_FAILED("Invalid protocol type specified\n");
    }
 
    //finally, build the return collection of interfaces that need to be removed or active
    SerialData::FirewallDeltaColl tmp_coll;
 
    //now add the new set
    vif_coll = iter->second._vif_manager.expose(iter->second._encapsulation);
    s_iter = vif_coll.begin();
    while (s_iter != vif_coll.end()) {
      if (s_iter->second._in_firewall_name.empty() == false) {
	tmp_coll.push_back(FirewallDelta(iter->first, s_iter->first, s_iter->second._in_firewall_name, "in"));
      }
      if (s_iter->second._out_firewall_name.empty() == false) {
	tmp_coll.push_back(FirewallDelta(iter->first, s_iter->first, s_iter->second._out_firewall_name, "out"));
      }
      if (s_iter->second._local_firewall_name.empty() == false) {
	tmp_coll.push_back(FirewallDelta(iter->first, s_iter->first, s_iter->second._local_firewall_name, "local"));
      }
      ++s_iter;
    }
 
    //then remove the dups and return
    fw_add_coll = fw_diff(tmp_coll, iter->second._fw_active_coll);
    fw_del_coll = fw_diff(iter->second._fw_active_coll, tmp_coll);
    iter->second._fw_active_coll = tmp_coll;
  }
  XLOG_INFO("RLSerialNode::commit_serial_transaction(), successful (index=%ld)", index);
  if (config_success == false) {
    return XrlCmdError::COMMAND_FAILED("Commit failed to configure serial card\n");
  }
  return post_serial_commit(serial);
}


/**
 *
 */
XrlCmdError
RLSerialNode::post_serial_commit(
		     //Input values,
		     const string &serial)
{
  char buf[180];

  sprintf(buf, "/etc/wanpipe/scripts/vyatta-wan start %ld", strtol(serial.substr(3).c_str(), NULL, 10) + 1);

  //now restart the driver
  if (rl_command::execute(buf) == false) {
    XLOG_ERROR("rl_interface failed to execute command: %s", buf);
    _serial_data_coll.erase(serial);
    return XrlCmdError::COMMAND_FAILED("Serial card failed to start\n");
  }
  else {
    XLOG_ERROR("rl_interface EXECUTED command: %s", buf);
  }

  return XrlCmdError::OKAY();
}

static
void
set_secrets(const char *fn, SerialData::SerialDataIter &iter)
{
    FILE *fd_wr = fopen(fn, "w");
#if 0
    if (iter->second._user_id.empty() == false)
	    myput("\"" + iter->second._user_id + '"', fd_wr);

    if (iter->second._system_name.empty() == false)
	    myput("\t\"" + iter->second._system_name + '"', fd_wr);

    if (iter->second._password.empty() == false)
	    myput("\t\"" + iter->second._password + '"', fd_wr);

#else
    myput("*\t*\t\"" + iter->second._password + "\"\t*\n", fd_wr);
#endif
    fchmod(fileno(fd_wr), 0600);
    fclose(fd_wr);
}

/**
 *
 **/
XrlCmdError
RLSerialNode::write_ppp_conf(const string &serial, long int index, const string &ip_line, const string &file)
{
  //first construct the file
  if (file.empty() == false) {
    rl_fileaccess fa(file, file);
 
    FILE *fd_rd = NULL, *fd_wr = NULL;
    bool success = fa.get(fd_rd, fd_wr);
    if (success == false) {
      XLOG_ERROR("RLSerialNode: failed to open file: %s", file.c_str());
      return XrlCmdError::COMMAND_FAILED();
    }
 
    //start writing out
    string new_line;

    SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
    if (iter == _serial_data_coll.end()) {
      return XrlCmdError::COMMAND_FAILED();
    }

    time_t t = time(NULL);
    new_line = "# Date: ";

    myput("#================================================\n", fd_wr);
    myput("# WANPIPE1 Configuration File\n", fd_wr);
    myput("#================================================\n", fd_wr);
    myput("#\n", fd_wr);
    myput(new_line+ctime(&t), fd_wr);
    myput("#\n", fd_wr);
    myput("# Note: This file was generated automatically\n", fd_wr);
    myput("#       by Vyatta, you should not manually edit\n", fd_wr);
    myput("#       this file.  If you want to modify your\n", fd_wr);
    myput("#       serial interface settings, you should\n", fd_wr);
    myput("#       use the Vyatta CLI or GUI.\n", fd_wr);
    myput("#\n", fd_wr);
    myput("#================================================\n", fd_wr);
    myput("\n\n", fd_wr);
    myput("##" + iter->second._card_name + "\n", fd_wr);

    //[devices]
    myput("[devices]\n", fd_wr);

    char buf[80];
    sprintf(buf, "%ld", index);
    new_line = "wanpipe";
    new_line += buf;
    if (iter->second._physical == "T1" || iter->second._physical == "E1") {
    if (strstr(iter->second._card_name.c_str(),"-SH")) {
	new_line += " = WAN_AFT_TE1";
      } else {
	new_line += " = WAN_AFT";
      }
    }
    else {
      new_line += " = WAN_AFT_TE3";
    }
    myput(new_line, fd_wr);
    myput(", Comment\n\n\n", fd_wr);

    //[interfaces]
    myput("[interfaces]\n", fd_wr);
    myput(serial + " = wanpipe" + buf + ", , STACK, Comment\n", fd_wr);
 
    sprintf(buf, "%s.1 = wanpipe%ld, %ld, TTY, tty, %s.ppp\n\n\n", serial.c_str(), index, index-1, serial.c_str());
    myput(buf, fd_wr);
 
    //[wanpipe1]
    sprintf(buf, "[wanpipe%ld]\n", index);
    myput(buf, fd_wr);
    myput("CARD_TYPE       = AFT\n", fd_wr);
    myput("S514CPU = " + iter->second._cpu + "\n", fd_wr);
    myput("CommPort        = PRI\n", fd_wr);
    myput("AUTO_PCISLOT = NO\n", fd_wr);
    myput("PCISLOT = " + iter->second._pci_slot + "\n", fd_wr);
    myput("PCIBUS = " + iter->second._pci_bus + "\n", fd_wr);

    if (iter->second._physical == "T1") {
      myput("FE_MEDIA = T1\n", fd_wr);
      myput("FE_FRAME = ESF\n", fd_wr);

      myput("LBO = " + iter->second._t1_lbo + "\n", fd_wr);

      if (iter->second._t1_line_coding == "ami") {
	new_line = "FE_LCODE = AMI\n";
      }
      else { //B8ZS
	new_line = "FE_LCODE = B8ZS\n";
      }
      myput(new_line, fd_wr);

      myput("ACTIVE_CH = " + iter->second._t1_timeslots + "\n", fd_wr);

      sprintf(buf, "%d", iter->second._t1_mtu);
      myput("MTU = " + string(buf) + "\n", fd_wr);

      if (iter->second._t1_clock == "internal") {
	new_line = "TE_CLOCK = MASTER\n";
      }
      else {
	new_line = "TE_CLOCK = NORMAL\n";
      }
      myput(new_line, fd_wr);
    }
    else if (iter->second._physical == "E1") {
      myput("FE_MEDIA = E1\n", fd_wr);

      if (iter->second._e1_framing == "g704") {
	new_line = "FE_FRAME = CRC4\n";
      }
      else if (iter->second._e1_framing == "g704-no-crc4") {
	new_line = "FE_FRAME = NCRC4\n";
      }
      else { //unframed
	new_line = "FE_FRAME = Unframed\n";
      }
      myput(new_line, fd_wr);

      if (iter->second._e1_line_coding == "ami") {
	new_line = "FE_LCODE = AMI\n";
      }
      else { //hdb3
	new_line = "FE_LCODE = HDB3\n";
      }
      myput(new_line, fd_wr);

      myput("ACTIVE_CH = " + iter->second._e1_timeslots + "\n", fd_wr);

      sprintf(buf, "%d", iter->second._e1_mtu);
      myput("MTU = " + string(buf) + "\n", fd_wr);

      if (iter->second._e1_clock == "internal") {
	new_line = "TE_CLOCK = MASTER\n";
      }
      else {
	new_line = "TE_CLOCK = NORMAL\n";
      }
      myput(new_line, fd_wr);
      myput("LBO = 120OH\n", fd_wr);
      myput("TE_SIG_MODE = CCS\n", fd_wr);
    }
    else if (iter->second._physical == "T3") {
      myput("FE_MEDIA = DS3\n", fd_wr);
 
      if (iter->second._t3_framing == "c-bit") {
	new_line = "FE_FRAME = C-BIT\n";
      }
      else {
	new_line = "FE_FRAME = M13\n";
      }
      myput(new_line, fd_wr);
    
      if (iter->second._t3_line_coding == "ami") {
	new_line = "FE_LCODE = AMI\n";
      }
      else {
	new_line = "FE_LCODE = B3ZS\n";
      }
      myput(new_line, fd_wr);
    
      myput("TE_CLOCK = NORMAL\n", fd_wr);
      myput("LBO = 0DB\n", fd_wr);
      sprintf(buf, "%d", iter->second._t1_mtu);
      myput("MTU = " + string(buf) + "\n", fd_wr);	//TODO: VERIFY!
    }
    else if (iter->second._physical == "E3") {
      myput("FE_MEDIA = E3\n", fd_wr);
    
      if (iter->second._e3_framing == "c-bit") {
	new_line = "FE_FRAME = G.751\n";
      }
      else {
	new_line = "FE_FRAME = G.832\n";
      }
      myput(new_line, fd_wr);
    
      if (iter->second._e3_line_coding == "ami") {
	new_line = "FE_LCODE = AMI\n";
      }
      else {
	new_line = "FE_LCODE = HDB3\n";
      }
      myput(new_line, fd_wr);
    
      myput("TE_CLOCK = NORMAL\n", fd_wr);
      myput("LBO = 0DB\n", fd_wr);
      sprintf(buf, "%d", iter->second._e1_mtu);
      myput("MTU = " + string(buf) + "\n", fd_wr);	//TODO: VERIFY!
    }

    sprintf(buf, "%d", iter->second._port);
    myput("FE_LINE = " + string(buf) + "\n", fd_wr);
    myput("TE_REF_CLOCK    = 0\n", fd_wr);
    myput("TE_HIGHIMPEDANCE        = NO\n", fd_wr);
    myput("FE_TXTRISTATE   = NO\n", fd_wr);
    myput("UDPPORT         = 9000\n", fd_wr);
    myput("TTL             = 255\n", fd_wr);
    myput("IGNORE_FRONT_END = NO\n", fd_wr);
    myput("\n", fd_wr);

    //[wan1.1]
    sprintf(buf, "[%s.1]\n", serial.c_str());
    myput(buf, fd_wr);
    myput("IPX     = NO\n", fd_wr);
    myput("MULTICAST       = NO\n", fd_wr);
    myput("DYN_INTR_CFG    = NO\n", fd_wr);
    myput("TRUE_ENCODING_TYPE      = NO\n", fd_wr);
    myput("GATEWAY = NO\n", fd_wr);
    myput("\n", fd_wr);
//////////////////////////////////////////////////////////////////////////////////////////////
    //[wan0.ppp]
    myput("[" + serial + ".ppp]\n", fd_wr);
    if (iter->second._authentication == "pap") {
      myput("#\tPAP = YES\n", fd_wr);
      myput("#\tCHAP = NO\n", fd_wr);
    } else if (iter->second._authentication == "chap") {
      myput("#\tPAP = NO\n", fd_wr);
      myput("#\tCHAP = YES\n", fd_wr);
    } else if (iter->second._authentication == "eap") {
      myput("#\tPAP = NO\n", fd_wr);
      myput("#\tCHAP = YES\n", fd_wr);
    } else if (iter->second._authentication == "mschap") {
      myput("#\tPAP = NO\n", fd_wr);
      myput("#\tCHAP = YES\n", fd_wr);
    } else if (iter->second._authentication == "mschap-v2") {
      myput("#\tPAP = NO\n", fd_wr);
      myput("#\tCHAP = YES\n", fd_wr);
    } else if (iter->second._authentication == "none") {
      myput("#\tPAP = NO\n", fd_wr);
      myput("#\tCHAP = NO\n", fd_wr);
    } else if (iter->second._authentication == "any") {
      myput("#\tPAP = NO\n", fd_wr);
      myput("#\tCHAP = NO\n", fd_wr);
    }

    if (iter->second._user_id.empty() == false)
      myput("#\tUSERID = " + iter->second._user_id + "\n", fd_wr);

    if (iter->second._password.empty() == false)
      myput("#\tPASSWD = " + iter->second._password + "\n", fd_wr);

    myput("\n", fd_wr);
  
/////////////////////////////////////////////////////////////////////////////////////////////////////
    {
	    new_line = "/etc/ppp/peers/" + serial;
	    FILE *fd_wr = fopen(new_line.c_str(), "w");

	    if(NULL==fd_wr)
		XLOG_ERROR("RLSerialNode: failed to open file: %s (errno=%d)", new_line.c_str(), errno);

	    myput("/dev/ttyWP" + serial.substr(3) + "\nsync\n" + ip_line, fd_wr);
	    myput("ipparam wan" + serial.substr(3) + ".1\n", fd_wr);

	    if (iter->second._user_id.empty() == false)
		myput("user \"" + iter->second._user_id + "\"\n", fd_wr);

	    if(iter->second._authentication == "none") {
	      myput("noauth", fd_wr);
	    } else if(	    iter->second._authentication == "pap"
			    ||
		    iter->second._authentication == "chap"
			    ||
		    iter->second._authentication == "eap"
			    ||
		    iter->second._authentication == "mschap"
			    ||
		    iter->second._authentication == "mschap-v2"
	   ) {
	      myput("auth\nrequire-" + iter->second._authentication, fd_wr);
	    } else {	// "any"
	      myput("auth", fd_wr);
	    }

	    myput('\n', fd_wr);
	    fclose(fd_wr);

	    set_secrets("/etc/ppp/pap-secrets", iter);
	    set_secrets("/etc/ppp/chap-secrets", iter);
    }
/////////////////////////////////////////////


    //[wan0]
    myput("[" + serial + "]\n", fd_wr);
    myput("HDLC_STREAMING  = YES\n", fd_wr);
    myput("ACTIVE_CH       = ALL\n", fd_wr);
    myput("MTU             = 1500\n", fd_wr);
    myput("MRU             = 1500\n", fd_wr);
    myput("DATA_MUX        = NO\n", fd_wr);
  }

  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLSerialNode::write_hdlc_conf(const string &serial, long int index, const string &file)
{
  //first construct the file
  if (file.empty() == false) {
    rl_fileaccess fa(file, file);
  
    FILE *fd_rd = NULL, *fd_wr = NULL;
    bool success = fa.get(fd_rd, fd_wr);
    if (success == false) {
      XLOG_ERROR("RLSerialNode: failed to open file: %s", file.c_str());
      return XrlCmdError::COMMAND_FAILED();
    }
  
    //start writing out
    string new_line;
 
    SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
    if (iter == _serial_data_coll.end()) {
      return XrlCmdError::COMMAND_FAILED();
    }


    time_t t = time(NULL);
    new_line = "# Date: ";
    myput("#================================================\n", fd_wr);
    myput("# WANPIPE1 Configuration File\n", fd_wr);
    myput("#================================================\n", fd_wr);
    myput("#\n", fd_wr);
    myput(new_line+ctime(&t), fd_wr);
    myput("#\n", fd_wr);
    myput("# Note: This file was generated automatically\n", fd_wr);
    myput("#       by Vyatta, you should not manually edit\n", fd_wr);
    myput("#       this file.  If you want to modify your\n", fd_wr);
    myput("#       serial interface settings, you should\n", fd_wr);
    myput("#       use the Vyatta CLI or GUI.\n", fd_wr);
    myput("#\n", fd_wr);
    myput("#================================================\n", fd_wr);
    myput("\n\n", fd_wr);
    myput("##" + iter->second._card_name + "\n", fd_wr);

    //[devices]
    myput("[devices]\n", fd_wr);
    char buf[80];
    sprintf(buf, "%ld", index);
    if (iter->second._physical == "T1" || iter->second._physical == "E1") {
      if (strstr(iter->second._card_name.c_str(),"-SH")) {
	new_line = "wanpipe" + string(buf) + " = WAN_AFT_TE1, Comment\n\n"; //only needs this for the 4 port card
      }
      else {
	new_line = "wanpipe" + string(buf) + " = WAN_AFT, Comment\n\n";
      }
    }
    else {
      new_line = "wanpipe" + string(buf) + " = WAN_AFT_TE3, Comment\n\n";
    }
    myput(new_line, fd_wr);
    myput("\n", fd_wr);

    //[interfaces]
    myput("[interfaces]\n", fd_wr);
  
    sprintf(buf, "%ld", index);
    myput(serial + " = wanpipe" + buf + ", , STACK, Comment\n", fd_wr);
  
    sprintf(buf, "%s.1 = wanpipe%ld, 1, WANPIPE, chdlc, %s.chdl\n\n", serial.c_str(), index, serial.c_str());
    myput(buf, fd_wr);
    myput("\n", fd_wr);
  
    //[wanpipe1]
    sprintf(buf, "[wanpipe%ld]\n", index);
    myput(buf, fd_wr);
    myput("CARD_TYPE       = AFT\n", fd_wr);
    myput("S514CPU = " + iter->second._cpu + "\n", fd_wr);
    myput("CommPort        = PRI\n", fd_wr);
    myput("AUTO_PCISLOT = NO\n", fd_wr);
    myput("PCISLOT = " + iter->second._pci_slot + "\n", fd_wr);
    myput("PCIBUS = " + iter->second._pci_bus + "\n", fd_wr);

    if (iter->second._physical == "T1") {
      myput("FE_MEDIA = T1\n", fd_wr);
      myput("FE_FRAME = ESF\n", fd_wr);
      myput("LBO = " + iter->second._t1_lbo + "\n", fd_wr);
    
      if (iter->second._t1_line_coding == "ami") {
	new_line = "FE_LCODE = AMI\n";
      }
      else { //hdb3
	new_line = "FE_LCODE = B8ZS\n";
      }
      myput(new_line, fd_wr);
    
      myput("ACTIVE_CH = " + iter->second._t1_timeslots + "\n", fd_wr);
    
      sprintf(buf, "%d", iter->second._t1_mtu);
      myput("MTU = " + string(buf) + "\n", fd_wr);
    
      if (iter->second._t1_clock == "internal") {
	new_line = "TE_CLOCK = MASTER\n";
      }
      else {
	new_line = "TE_CLOCK = NORMAL\n";
      }
      myput(new_line, fd_wr);
    }
    else if (iter->second._physical == "E1") {
      myput("FE_MEDIA = E1\n", fd_wr);
    
      if (iter->second._e1_framing == "g704") {
	new_line = "FE_FRAME = CRC4\n";
      }
      else if (iter->second._e1_framing == "g704-no-crc4") {
	new_line = "FE_FRAME = NCRC4\n";
      }
      else { //unframed
	new_line = "FE_FRAME = Unframed\n";
      }
      myput(new_line, fd_wr);
    
      if (iter->second._e1_line_coding == "ami") {
	new_line = "FE_LCODE = AMI\n";
      }
      else { //hdb3
	new_line = "FE_LCODE = HDB3\n";
      }
      myput(new_line, fd_wr);
      myput("ACTIVE_CH = " + iter->second._e1_timeslots + "\n", fd_wr);
    
      sprintf(buf, "%d", iter->second._e1_mtu);
      myput("MTU = " + string(buf) + "\n", fd_wr);
    
      if (iter->second._e1_clock == "internal") {
	new_line = "TE_CLOCK = MASTER\n";
      }
      else {
	new_line = "TE_CLOCK = NORMAL\n";
      }
      myput(new_line, fd_wr);
    
    }
    else if (iter->second._physical == "T3") {
      myput("FE_MEDIA = DS3\n", fd_wr);
    
      if (iter->second._t3_framing == "c-bit") {
	new_line = "FE_FRAME = C-BIT\n";
      }
      else {
	new_line = "FE_FRAME = M13\n";
      }
      myput(new_line, fd_wr);
    
      if (iter->second._t3_line_coding == "ami") {
	new_line = "FE_LCODE = AMI\n";
      }
      else {
	new_line = "FE_LCODE = B3ZS\n";
      }
      myput(new_line, fd_wr);
    
      myput("TE_CLOCK = NORMAL\n", fd_wr);
      myput("LBO = 0DB\n", fd_wr);
      sprintf(buf, "%d", iter->second._t1_mtu);
      myput("MTU = " + string(buf) + "\n", fd_wr);
    }
    else if (iter->second._physical == "E3") {
      myput("FE_MEDIA = E3\n", fd_wr);
    
      if (iter->second._e3_framing == "c-bit") {
	new_line = "FE_FRAME = G.751\n";
      }
      else {
	new_line = "FE_FRAME = G.832\n";
      }
      myput(new_line, fd_wr);
    
      if (iter->second._e3_line_coding == "ami") {
	new_line = "FE_LCODE = AMI\n";
      }
      else {
	new_line = "FE_LCODE = HDB3\n";
      }
      myput(new_line, fd_wr);
    
      myput("TE_CLOCK = NORMAL\n", fd_wr);
      myput("LBO = 0DB\n", fd_wr);
      sprintf(buf, "%d", iter->second._e1_mtu);
      myput("MTU = " + string(buf) + "\n", fd_wr);
    }

    sprintf(buf, "%d", iter->second._port);
    myput("FE_LINE = " + string(buf) + "\n", fd_wr);
    myput("TE_REF_CLOCK    = 0\n", fd_wr);
    myput("TE_HIGHIMPEDANCE        = NO\n", fd_wr);
    myput("FE_TXTRISTATE   = NO\n", fd_wr);
    myput("UDPPORT         = 9000\n", fd_wr);
    myput("TTL             = 255\n", fd_wr);
    myput("IGNORE_FRONT_END = NO\n", fd_wr);
    myput("\n", fd_wr);


    //[wan1.1]
    sprintf(buf, "[%s.1]\n", serial.c_str());
    myput(buf, fd_wr);
    myput("KEEPALIVE_ERROR_MARGIN  = 3\n", fd_wr);
    if (iter->second._require_rx != "disable") {
      sprintf(buf, "%d", iter->second._timer);
      new_line = "KEEPALIVE_TIMER = " + string(buf) + "\n";
    }
    else {
      new_line = "KEEPALIVE_TIMER = 0\n";
    }
    myput(new_line, fd_wr);

    myput("IGNORE_KEEPALIVE = NO"  + string("\n"), fd_wr);
    myput("MULTICAST       = NO\n", fd_wr);
    myput("DYN_INTR_CFG    = NO\n", fd_wr);
    myput("TRUE_ENCODING_TYPE      = NO\n", fd_wr);
    myput("GATEWAY = NO\n", fd_wr);
    myput("\n", fd_wr);


    //[wan0.hdlc]
    myput("[" + serial + ".chdl]\n", fd_wr);
    myput("\n", fd_wr);


    //[wan0]
    myput("[" + serial + "]\n", fd_wr);
    myput("HDLC_STREAMING  = YES\n", fd_wr);
    myput("ACTIVE_CH       = ALL\n", fd_wr);
    myput("MTU             = 1500\n", fd_wr);
    myput("MRU             = 1500\n", fd_wr);
    myput("DATA_MUX        = NO\n", fd_wr);
  }
  return XrlCmdError::OKAY();
}


/**
 *
 **/
XrlCmdError
RLSerialNode::write_frame_conf(const string &serial, long int index, const string &file)
{
  //first construct the file
  if (file.empty() == false) {
    rl_fileaccess fa(file, file);
  
    FILE *fd_rd = NULL, *fd_wr = NULL;
    bool success = fa.get(fd_rd, fd_wr);
    if (success == false) {
      XLOG_ERROR("RLSerialNode: failed to open file: %s", file.c_str());
      return XrlCmdError::COMMAND_FAILED();
    }
  
    //start writing out
    string new_line;
 
    SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
    if (iter == _serial_data_coll.end()) {
      return XrlCmdError::COMMAND_FAILED();
    }


    time_t t = time(NULL);
    new_line = "# Date: ";
    myput("#================================================\n", fd_wr);
    myput("# WANPIPE1 Configuration File\n", fd_wr);
    myput("#================================================\n", fd_wr);
    myput("#\n", fd_wr);
    myput(new_line + ctime(&t), fd_wr);
    myput("#\n", fd_wr);
    myput("# Note: This file was generated automatically\n", fd_wr);
    myput("#       by Vyatta, you should not manually edit\n", fd_wr);
    myput("#       this file.  If you want to modify your\n", fd_wr);
    myput("#       serial interface settings, you should\n", fd_wr);
    myput("#       use the Vyatta CLI or GUI.\n", fd_wr);
    myput("#\n", fd_wr);
    myput("#================================================\n", fd_wr);
    myput("\n\n", fd_wr);
    myput("##" + iter->second._card_name + "\n", fd_wr);

    //[devices]
    myput("[devices]\n", fd_wr);
    char buf[80];
    sprintf(buf, "%ld", index);
    if (iter->second._physical == "T1" || iter->second._physical == "E1") {
      if (strstr(iter->second._card_name.c_str(),"-SH")) {
	new_line = "wanpipe" + string(buf) + " = WAN_AFT_TE1, Comment\n\n";
      }
      else {
	new_line = "wanpipe" + string(buf) + " = WAN_AFT, Comment\n\n";
      }
    }
    else {
      new_line = "wanpipe" + string(buf) + " = WAN_AFT_TE3, Comment\n\n";
    }
    myput(new_line, fd_wr);
    myput("\n", fd_wr);

    //[interfaces]
    myput("[interfaces]\n", fd_wr);

    sprintf(buf, "%ld", index);
    myput(serial + " = wanpipe" + buf + ", , STACK, Comment\n", fd_wr);
  
    VifData::VifDataColl vif_coll = iter->second._vif_manager.expose(iter->second._encapsulation);
    VifData::VifDataIter s_iter = vif_coll.begin();
    while (s_iter != vif_coll.end()) {
      sprintf(buf, "%s.%d = wanpipe%ld, %d, WANPIPE, fr, %s.fr\n", serial.c_str(), s_iter->first, index, s_iter->first, serial.c_str());
      myput(buf, fd_wr);
      ++s_iter;
    }
    myput("\n", fd_wr);

    //[wanpipe1]
    sprintf(buf, "[wanpipe%ld]\n", index);
    myput(buf, fd_wr);
    myput("CARD_TYPE       = AFT\n", fd_wr);
    myput("S514CPU = " + iter->second._cpu + "\n", fd_wr);
    myput("CommPort        = PRI\n", fd_wr);
    myput("AUTO_PCISLOT = NO\n", fd_wr);
    myput("PCISLOT = " + iter->second._pci_slot + "\n", fd_wr);
    myput("PCIBUS = " + iter->second._pci_bus + "\n", fd_wr);

    if (iter->second._physical == "T1") {
      myput("FE_MEDIA = T1\n", fd_wr);
      myput("FE_FRAME = ESF\n", fd_wr);
      myput("LBO = " + iter->second._t1_lbo + "\n", fd_wr);
    
      if (iter->second._t1_line_coding == "ami") {
	new_line = "FE_LCODE = AMI\n";
      }
      else { //b8zs
	new_line = "FE_LCODE = B8ZS\n";
      }
      myput(new_line, fd_wr);
    
      myput("ACTIVE_CH = " + iter->second._t1_timeslots + "\n", fd_wr);
    
      sprintf(buf, "%d", iter->second._t1_mtu);
      myput("MTU = " + string(buf) + "\n", fd_wr);
    
      if (iter->second._t1_clock == "internal") {
	new_line = "TE_CLOCK = MASTER\n";
      }
      else {
	new_line = "TE_CLOCK = NORMAL\n";
      }
      myput(new_line, fd_wr);
    }
    else if (iter->second._physical == "E1") {
      myput("FE_MEDIA = E1\n", fd_wr);
    
      if (iter->second._e1_framing == "g704") {
	new_line = "FE_FRAME = CRC4\n";
      }
      else if (iter->second._e1_framing == "g704-no-crc4") {
	new_line = "FE_FRAME = NCRC4\n";
      }
      else { //unframed
	new_line = "FE_FRAME = Unframed\n";
      }
      myput(new_line, fd_wr);
    
      if (iter->second._e1_line_coding == "ami") {
	new_line = "FE_LCODE = AMI\n";
      }
      else { //hdb3
	new_line = "FE_LCODE = HDB3\n";
      }
      myput(new_line, fd_wr);
    
      myput("ACTIVE_CH = " + iter->second._e1_timeslots + "\n", fd_wr);
    
      sprintf(buf, "%d", iter->second._e1_mtu);
      myput("MTU = " + string(buf) + "\n", fd_wr);
    
      if (iter->second._e1_clock == "internal") {
	new_line = "TE_CLOCK = MASTER\n";
      }
      else {
	new_line = "TE_CLOCK = NORMAL\n";
      }
      myput(new_line, fd_wr);
    }
    else if (iter->second._physical == "T3") {
      myput("FE_MEDIA = DS3\n", fd_wr);
    
      if (iter->second._t3_framing == "c-bit") {
	new_line = "FE_FRAME = C-BIT\n";
      }
      else {
	new_line = "FE_FRAME = M13\n";
      }
      myput(new_line, fd_wr);
    
      if (iter->second._t3_line_coding == "ami") {
	new_line = "FE_LCODE = AMI\n";
      }
      else {
	new_line = "FE_LCODE = B3ZS\n";
      }
      myput(new_line, fd_wr);
    
      myput("TE_CLOCK = NORMAL\n", fd_wr);
      myput("LBO = 0DB\n", fd_wr);
      sprintf(buf, "%d", iter->second._t1_mtu);
      myput("MTU = " + string(buf) + "\n", fd_wr);
    }
    else if (iter->second._physical == "E3") {
      myput("FE_MEDIA = E3\n", fd_wr);
    
      if (iter->second._e3_framing == "c-bit") {
	new_line = "FE_FRAME = G.751\n";
      }
      else {
	new_line = "FE_FRAME = G.832\n";
      }
      myput(new_line, fd_wr);
    
      if (iter->second._e3_line_coding == "ami") {
	new_line = "FE_LCODE = AMI\n";
      }
      else {
	new_line = "FE_LCODE = HDB3\n";
      }
      myput(new_line, fd_wr);
    
      myput("TE_CLOCK = NORMAL\n", fd_wr);
      myput("LBO = 0DB\n", fd_wr);
      sprintf(buf, "%d", iter->second._e1_mtu);
      myput("MTU = " + string(buf) + "\n", fd_wr);
    }

    sprintf(buf, "%d", iter->second._port);
    myput("FE_LINE = " + string(buf) + "\n", fd_wr);
    myput("TE_REF_CLOCK    = 0\n", fd_wr);
    myput("TE_HIGHIMPEDANCE        = NO\n", fd_wr);
    myput("FE_TXTRISTATE   = NO\n", fd_wr);
    myput("UDPPORT         = 9000\n", fd_wr);
    myput("TTL             = 255\n", fd_wr);
    myput("IGNORE_FRONT_END = NO\n", fd_wr);
    myput("\n", fd_wr);


    //[wan0.fr]
    myput("[" + serial + ".fr]" + "\n", fd_wr);

    fprintf(fd_wr, "NUMBER_OF_DLCI = %d\n", vif_coll.size());
  
    myput("Station = CPE\n", fd_wr);
    myput("SIGNALLING = " + iter->second._signaling + "\n", fd_wr);
  
    sprintf(buf, "%d", iter->second._n391dte);
    myput("N391 = " + string(buf) + "\n", fd_wr);
  
    sprintf(buf, "%d", iter->second._n392dte);
    myput("N392 = " + string(buf) + "\n", fd_wr);
  
    sprintf(buf, "%d", iter->second._n393dte);
    myput("N393 = " + string(buf) + "\n", fd_wr);
  
    sprintf(buf, "%d", iter->second._t391dte);
    myput("T391 = " + string(buf) + "\n", fd_wr);

    myput("FR_ISSUE_FS = YES\n\n", fd_wr);
  
    //now roll through and add the vifs
    s_iter = vif_coll.begin();
    while (s_iter != vif_coll.end()) {
      fprintf(fd_wr, "[%s.%d]\n", serial.c_str(), s_iter->first);
      fprintf(fd_wr, "CIR = %d\n", s_iter->second._cir);
      fprintf(fd_wr, "BC = %d\n", s_iter->second._burst);
      fprintf(fd_wr, "BE = %d\n", s_iter->second._excess);
      if (s_iter->second._tx_inverse_arp != 0) {
	myput("INARP = YES\n", fd_wr);
	fprintf(fd_wr, "INARPINTERVAL = %d\n", s_iter->second._tx_inverse_arp);
      }
      else {
	myput("INARP = NO\n", fd_wr);
	myput("INARPINTERVAL = 0\n", fd_wr);
      }
    
      if (s_iter->second._rx_inverse_arp == "enable") {
	myput("INARP_RX = YES\n", fd_wr);
      }
      else {
	myput("INARP_RX = NO\n", fd_wr);
      }
    
      myput("IPX = NO\n", fd_wr);
      myput("MULTICAST = NO\n", fd_wr);
      myput("DYN_INTR_CFG = NO\n", fd_wr);
      myput("TRUE_ENCODING_TYPE = NO\n", fd_wr);
      myput("GATEWAY = NO\n", fd_wr);
      myput('\n', fd_wr);
      ++s_iter;
    }
    myput('\n', fd_wr);


    //[wan1.1]
    fprintf(fd_wr, "[%s.%ld]\n", serial.c_str(), index);
    myput("IPX     = NO\n", fd_wr);
    myput("MULTICAST       = NO\n", fd_wr);
    myput("DYN_INTR_CFG    = NO\n", fd_wr);
    myput("TRUE_ENCODING_TYPE      = NO\n", fd_wr);
    myput("GATEWAY = NO\n", fd_wr);
    myput("\n", fd_wr);

    //[wan0.ppp]
    myput("[" + serial + ".fr]\n", fd_wr);
    myput("\n", fd_wr);


    //[wan0]
    myput("[" + serial + "]\n", fd_wr);
    myput("HDLC_STREAMING  = YES\n", fd_wr);
    myput("ACTIVE_CH       = ALL\n", fd_wr);
    myput("MTU             = 1500\n", fd_wr);
    myput("MRU             = 1500\n", fd_wr);
    myput("DATA_MUX        = NO\n", fd_wr);
  }
  return XrlCmdError::OKAY();
}

/**
 *
 */
SerialData::FirewallDeltaColl
RLSerialNode::fw_diff(const SerialData::FirewallDeltaColl &coll1, const SerialData::FirewallDeltaColl &coll2)
{
  SerialData::FirewallDeltaColl diff;
  SerialData::FirewallDeltaConstIter iter1;

  //a slow search, we can upgrade if performance becomes an issue
  for(iter1=coll1.begin(); iter1!=coll1.end(); ++iter1) {
    SerialData::FirewallDeltaConstIter iter2;

    for(iter2=coll2.begin(); ; ++iter2) {
      if(iter2==coll2.end()) {
	 diff.push_back(*iter1);
	 break;
      }

      if(*iter2==*iter1)
	break;
    }
  }

  return diff;
}

/**
 *
 */
XrlCmdError
RLSerialNode::create_address(
				 //Input values,
				 const string &serial,
				 const string &proto,
				 const uint32_t &vif, 
				 const IPv4 &address,
				 const IPv4 &remote_address,
				 const uint32_t &prefix_length)
{
  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    VifData::VifDataIter s_iter;
    if (iter->second._vif_manager.get_vif(proto, vif, s_iter) == true) {
      s_iter->second._state = VifData::CHANGED;
      s_iter->second._address = address;
      s_iter->second._pointtopoint = remote_address;
      s_iter->second._prefix_length = prefix_length;
    }
  }

  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
RLSerialNode::delete_address(
				 //Input values,
				 const string &serial,
				 const string &proto,
				 const uint32_t &vif, 
				 const IPv4 &address,
				 const IPv4 &remote_address,
				 const uint32_t &prefix_length)
{
  UNUSED(remote_address);
  UNUSED(address);
  UNUSED(prefix_length);
  UNUSED(proto);

  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
    VifData::VifDataIter s_iter;
    if (iter->second._vif_manager.get_vif(proto, vif, s_iter) == true) {
      iter->second._vif_manager.erase(proto, vif);
    }
  }
  return XrlCmdError::OKAY();
}


/**
 *
 */
XrlCmdError
RLSerialNode::set_dlci(
			   //Input values,
			   const string &serial,
			   const string &proto,
			   const uint32_t &vif,
			   const uint32_t &dlci)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    VifData::VifDataIter s_iter;
    if (iter->second._vif_manager.get_vif(proto, vif, s_iter) == true) {
      s_iter->second._dlci = dlci;
    }
  }
  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
RLSerialNode::set_cir(
			  //Input values,
			  const string &serial,
			  const string &proto,
			  const uint32_t &vif,
			  const uint32_t &cir)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    VifData::VifDataIter s_iter;
    if (iter->second._vif_manager.get_vif(proto, vif, s_iter) == true) {
      s_iter->second._cir = cir;
    }
  }
  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
RLSerialNode::set_burst(
			    //Input values,
			    const string &serial,
			    const string &proto,
			    const uint32_t &vif,
			    const uint32_t &burst)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    VifData::VifDataIter s_iter;
    if (iter->second._vif_manager.get_vif(proto, vif, s_iter) == true) {
      s_iter->second._burst = burst;
    }
  }
  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
RLSerialNode::set_excess(
			    //Input values,
			    const string &serial,
			    const string &proto,
			    const uint32_t &vif,
			    const uint32_t &excess)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    VifData::VifDataIter s_iter;
    if (iter->second._vif_manager.get_vif(proto, vif, s_iter) == true) {
      s_iter->second._excess = excess;
    }
  }
  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
RLSerialNode::set_tx_inverse_arp(
				     //Input values,
				     const string &serial,
				     const string &proto,
				     const uint32_t &vif,
				     const uint32_t &tx_inverse_arp)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    VifData::VifDataIter s_iter;
    if (iter->second._vif_manager.get_vif(proto, vif, s_iter) == true) {
      s_iter->second._tx_inverse_arp = tx_inverse_arp;
    }
  }
  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
RLSerialNode::set_rx_inverse_arp(
				     //Input values,
				     const string &serial,
				     const string &proto,
				     const uint32_t &vif,
				     const string &rx_inverse_arp)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    VifData::VifDataIter s_iter;
    if (iter->second._vif_manager.get_vif(proto, vif, s_iter) == true) {
      s_iter->second._rx_inverse_arp = rx_inverse_arp;
    }
  }
  return XrlCmdError::OKAY();
}

XrlCmdError
RLSerialNode::delete_physical(
	  //Input values,
	  const string &serial)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
    iter->second._physical = "NONE";
  }
  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_t1_lbo(
				   //Input values,
				   const string &serial, 
				   const string &lbo)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._physical != "T1" && iter->second._physical != "NONE") {
      return XrlCmdError::COMMAND_FAILED(iter->second._physical + " is already enabled.");
    }
    iter->second._physical = "T1";

    iter->second._t1_lbo = lbo;
  }

  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_t1_line_coding(
					   //Input values,
					   const string &serial, 
					   const string &line_coding)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
 
   if (iter->second._physical != "T1" && iter->second._physical != "NONE") {
      return XrlCmdError::COMMAND_FAILED(iter->second._physical + " is already enabled.");
    }
    iter->second._physical = "T1";

    iter->second._t1_line_coding = line_coding;
  }

  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_t1_timeslots(
					 //Input values,
					 const string &serial, 
					 const uint32_t &start,
					 const uint32_t &stop)
{
  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
    char buf[80];
    if (stop > start) {
      sprintf(buf, "%d-%d", start, stop);
    }
    else if (start == stop) {
      sprintf(buf, "%d", start);
    }
    else {
      return XrlCmdError::COMMAND_FAILED("Timeslot stop must be greater or equal to timeslot start value.");
    }

    if (iter->second._physical != "T1" && iter->second._physical != "NONE") {
      return XrlCmdError::COMMAND_FAILED(iter->second._physical + " is already enabled.");
    }
    iter->second._physical = "T1";

    //iter->second._t1_timeslots = string(buf);
    iter->second._t1_timeslots = buf;
  }
  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_t1_mtu(
				   //Input values,
				   const string &serial, 
				   const uint32_t &mtu)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._physical != "T1" && iter->second._physical != "NONE") {
      return XrlCmdError::COMMAND_FAILED(iter->second._physical + " is already enabled.");
    }
    iter->second._physical = "T1";

    iter->second._t1_mtu = mtu;
  }

  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_t1_clock(
				     //Input values,
				     const string &serial, 
				     const string &clock)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._physical != "T1" && iter->second._physical != "NONE") {
      return XrlCmdError::COMMAND_FAILED(iter->second._physical + " is already enabled.");
    }
    iter->second._physical = "T1";

    iter->second._t1_clock = clock;
  }

  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_e1_framing(
				       //Input values,
				       const string &serial, 
				       const string &framing)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._physical != "E1" && iter->second._physical != "NONE") {
      return XrlCmdError::COMMAND_FAILED(iter->second._physical + " is already enabled.");
    }
    iter->second._physical = "E1";

    iter->second._e1_framing = framing;
  }

  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_e1_line_coding(
					   //Input values,
					   const string &serial, 
					   const string &line_coding)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._physical != "E1" && iter->second._physical != "NONE") {
      return XrlCmdError::COMMAND_FAILED(iter->second._physical + " is already enabled.");
    }
    iter->second._physical = "E1";

    iter->second._e1_line_coding = line_coding;
  }

  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_e1_timeslots(
					 //Input values,
					 const string &serial, 
					 const uint32_t &start,
					 const uint32_t &stop)
{
  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
    char buf[80];
    if (stop > start) {
      sprintf(buf, "%d-%d", start, stop);
    }
    else if (start == stop) {
      sprintf(buf, "%d", start);
    }
    else {
      return XrlCmdError::COMMAND_FAILED("Timeslot stop must be greater or equal to timeslot start value.");
    }

    if (iter->second._physical != "E1" && iter->second._physical != "NONE") {
      return XrlCmdError::COMMAND_FAILED(iter->second._physical + " is already enabled.");
    }
    iter->second._physical = "E1";

    iter->second._e1_timeslots = buf;
  }
  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_e1_mtu(
				   //Input values,
				   const string &serial, 
				   const uint32_t &mtu)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._physical != "E1" && iter->second._physical != "NONE") {
      return XrlCmdError::COMMAND_FAILED(iter->second._physical + " is already enabled.");
    }
    iter->second._physical = "E1";

      iter->second._e1_mtu = mtu;
  }

  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_e1_clock(
				     //Input values,
				     const string &serial, 
				     const string &clock)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._physical != "E1" && iter->second._physical != "NONE") {
      return XrlCmdError::COMMAND_FAILED(iter->second._physical + " is already enabled.");
    }
    iter->second._physical = "E1";

    iter->second._e1_clock = clock;
  }

  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_t3_framing(
				       //Input values,
				       const string &serial, 
				       const string &framing)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._physical != "T3" && iter->second._physical != "NONE") {
      return XrlCmdError::COMMAND_FAILED(iter->second._physical + " is already enabled.");
    }
    iter->second._physical = "T3";

    iter->second._t3_framing = framing;
  }

  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_t3_line_coding(
					   //Input values,
					   const string &serial, 
					   const string &line_coding)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._physical != "T3" && iter->second._physical != "NONE") {
      return XrlCmdError::COMMAND_FAILED(iter->second._physical + " is already enabled.");
    }
    iter->second._physical = "T3";

    iter->second._t3_line_coding = line_coding;
  }

  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_e3_framing(
				       //Input values,
				       const string &serial, 
				       const string &framing)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._physical != "E3" && iter->second._physical != "NONE") {
      return XrlCmdError::COMMAND_FAILED(iter->second._physical + " is already enabled.");
    }
    iter->second._physical = "E3";

    iter->second._e3_framing = framing;
  }

  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_e3_line_coding(
					   //Input values,
					   const string &serial, 
					   const string &line_coding)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;

    if (iter->second._physical != "E3" && iter->second._physical != "NONE") {
      return XrlCmdError::COMMAND_FAILED(iter->second._physical + " is already enabled.");
    }
    iter->second._physical = "E3";

    iter->second._e3_line_coding = line_coding;
  }

  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_ppp_authentication(
					 //Input values,
					 const string &serial, 
					 const string &proto,
					 const string &authentication)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;
  
    iter->second._authentication = authentication;
  }
  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_auth_parameters_user_id(
					      //Input values,
					      const string &serial, 
					      const string &proto,
					      const string &user_id)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    iter->second._user_id = user_id;
  }
  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_auth_parameters_password(
					       //Input values,
					       const string &serial, 
					       const string &proto,
					       const string &password)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    iter->second._password = password;
  }
  return XrlCmdError::OKAY();
}

XrlCmdError 
RLSerialNode::set_auth_parameters_system_name(
						  //Input values,
						  const string &serial, 
						  const string &proto,
						  const string &system_name)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    iter->second._system_name = system_name;
  }
  return XrlCmdError::OKAY();
}

XrlCmdError
RLSerialNode::set_keepalives(
		    //Input values,
		    const string &serial,
		    const string &proto,
		    const string &require_rx,
		    const uint32_t &timer)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;
    iter->second._require_rx = require_rx;
    iter->second._timer = timer;
  }
  return XrlCmdError::OKAY();
}



XrlCmdError
RLSerialNode::set_signaling(
			    //Input values,
			    const string &serial,
			    const string &proto,
			    const string &signaling)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    iter->second._signaling = signaling;
  }
  return XrlCmdError::OKAY();
}

XrlCmdError
RLSerialNode::set_n391dte(
			  //Input values,
			  const string &serial,
			  const string &proto,
			  const uint32_t &n391dte)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    iter->second._n391dte = n391dte;
  }
  return XrlCmdError::OKAY();
}

XrlCmdError
RLSerialNode::set_n392dte(
			  //Input values,
			  const string &serial,
			  const string &proto,
			  const uint32_t &n392dte)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    iter->second._n392dte = n392dte;
  }
  return XrlCmdError::OKAY();
}

XrlCmdError
RLSerialNode::set_n393dte(
			  //Input values,
			  const string &serial,
			  const string &proto,
			  const uint32_t &n393dte)
{
 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    iter->second._n393dte = n393dte;
  }
  return XrlCmdError::OKAY();
}

XrlCmdError
RLSerialNode::set_t391dte(
			  //Input values,
			  const string &serial,
			  const string &proto,
			  const uint32_t &t391dte)
{
  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    iter->second._config_changed = true;
    if (iter->second._proto != proto && iter->second._proto.empty() == false) {
      iter->second._proto_transition = true;
    }
    iter->second._proto = proto;

    iter->second._t391dte = t391dte;
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
bool
RLSerialNode::validate(const string &serial, string &reason)
{
  //first move the new default files to the right location
  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    if (iter->second._encapsulation.empty()) {
      reason = "Encapsulation has not been specified\n";
      return false;
    }
	
    if (iter->second._proto.empty()) {
      reason = "Protocol has not been fully configured\n";
      return false;
    }

    if (iter->second._encapsulation != iter->second._proto) {
      reason = "Encapsulation type is not the same as the protocol or the protocol has not been fully configured\n";
      return false;
    }

    if (iter->second._proto_transition == true) {
      reason = "Only one protocol can be configured at a time\n";
      return false;
    }

    if ((iter->second._physical == "T1" || iter->second._physical == "E1") && iter->second._card == "sangoma-t3") {
      reason = "Card support T3 only\n";
      return false;
    }
	
    if ((iter->second._physical == "T3" || iter->second._physical == "E3") && iter->second._card == "sangoma-t1/e1") {
      reason = "Card support T1/E1 only\n";
      return false;
    }

    if (iter->second._physical == "NONE" || iter->second._physical.empty()) {
      reason = "t1/e1/t3-option is not specified\n";
      return false;
    }

    if (iter->second._encapsulation != "frame-relay") {
      VifData::VifDataColl vif_coll = iter->second._vif_manager.expose(iter->second._encapsulation);
      if (vif_coll.size() > 1) {
	reason = "Only one vif is allowed for ppp or hdlc\n";
	return false;
      }
    }

    VifData::VifDataColl vif_coll = iter->second._vif_manager.expose(iter->second._encapsulation);
    VifData::VifDataIter s_iter = vif_coll.begin();
    while (s_iter != vif_coll.end()) {

      if (iter->second._physical == "t1") {
	/*
	if (iter->second._t1_framing != "esf" &&
	    iter->second._t1_framing != "sf") {
	  reason = "Only esf or sf framing are allowed with a t1 line\n";
	  return false;
	}
	*/
	if (iter->second._t1_timeslots != "all") {
	  int slot = strtol(iter->second._t1_timeslots.c_str(), NULL, 10);
	  if (slot > 24) {
	    reason = "Timeslot has to be less than 24 for a t1 line\n";
	    return false;
	  }
	}
      }
      else if (iter->second._physical == "e1") { //e1
	if (iter->second._e1_framing == "esf" ||
	    iter->second._e1_framing == "sf") {
	  reason = "Only g704, g704-no-crc4 or unframed framing are allowed with a e1 line\n";
	  return false;
	}
      }
      ++s_iter;
    }
  }
  return true;
}

/**
 *
 *
 **/
bool
RLSerialNode::pci_bus_find(const string &wanfile, string &slot, string &bus, unsigned &port, unsigned &num_ports, string &cpu, string &card_name)
{
  rl_fileaccess fa(wanfile);
  FILE *fd_rd, *fd_wr;
  bool success = fa.get(fd_rd, fd_wr);
  if (success == false) {
    XLOG_ERROR("pci_bus_find: failed to open file: %s", wanfile.c_str());
    return false;
  }

  char line[1025];
  num_ports = 1;

  while (fgets(line, 1024, fd_rd) != NULL) {
    myput(line, fd_wr); //note we'll need to add a new method to fileaccess for safe reading only
    //    XLOG_INFO("confline: %s",line);
    string output(line);
    StrProc proc(output, " ");
    if (output.find("PCISLOT") != string::npos && output.find("AUTO_PCISLOT") == string::npos) {
      slot = proc.get(2);
    }
    else if (output.find("PCIBUS") != string::npos) {
      bus = proc.get(2);
    }
    else if (output.find("A102") != string::npos) {
      num_ports = 2;
    }
    else if (output.find("A104") != string::npos) {
      num_ports = 4;
    }
    else if (output.find("FE_LINE") != string::npos) {
      port = strtoul(proc.get(2).c_str(), NULL, 10);
    }
    else if (output.find("CPU") != string::npos) {
      cpu = proc.get(2);
    }

    if (output.find("##") != string::npos) {
      card_name = proc.get(0).substr(2);
    }
  }

  XLOG_INFO("pci_bus_find: %s, %s, %s, %d, %d, %s, %s",wanfile.c_str(),slot.c_str(),bus.c_str(),num_ports,port,cpu.c_str(),card_name.c_str());
  if (!card_name.empty() && !slot.empty() && !bus.empty() && !cpu.empty() && port < 5) {
    return true;
  }
  return false;
}


/**
 *
 **/
XrlCmdError
RLSerialNode::get_serial_data(
			      //Input values,
			      const string &name,
			      //Output values,
			      XrlAtomList &rules)
{
  SerialData::SerialDataIter iter = _serial_data_coll.find(name);
  if (iter != _serial_data_coll.end()) {
  
    string line = ";ifname=" + name + ";";
    line += "encap=" + iter->second._encapsulation + ";";

    //find the encap type here
    VifData::VifDataColl vif_coll = iter->second._vif_manager.expose(iter->second._encapsulation);
    VifData::VifDataIter s_iter = vif_coll.begin();
    while (s_iter != vif_coll.end()) {
      char buf[80];
      sprintf(buf, "%d", s_iter->first);
      line += "vif=";
      line += buf;
      line += ";";
      ++s_iter;
    }
    rules.append(XrlAtom(line));
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLSerialNode::set_firewall(
	     //Input values,
	     const string &serial,
	     const uint32_t &vif,
	     const string &proto,
	     const string &firewall_name,
	     const string &filter)
{
  UNUSED(proto);

 SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {

    VifData::VifDataIter s_iter;
    if (iter->second._vif_manager.get_vif(proto, vif, s_iter) == true) {
      if (filter == "in") {
	s_iter->second._in_firewall_name = firewall_name;
      }
      else if (filter == "out") {
	s_iter->second._out_firewall_name = firewall_name;
      }
      else if (filter == "local") {
	s_iter->second._local_firewall_name = firewall_name;
      }
    }
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLSerialNode::delete_firewall(
			      //Input values,
			      const string &serial,
			      const uint32_t &vif,
			      const string &proto,
			      const string &firewall_name,
			      const string &filter)
{
  UNUSED(firewall_name);

  SerialData::SerialDataIter iter = _serial_data_coll.find(serial);
  if (iter != _serial_data_coll.end()) {
    VifData::VifDataIter s_iter;
    if (iter->second._vif_manager.get_vif(proto, vif, s_iter) == true) {
      if (filter == "in") {
	s_iter->second._in_firewall_name = "";
      }
      else if (filter == "out") {
	s_iter->second._out_firewall_name = "";
      }
      else if (filter == "local") {
	s_iter->second._local_firewall_name = "";
      }
    }
  }
  return XrlCmdError::OKAY();
}

/**
 *
 */
bool
VifManager::get_vif(const string &proto, const uint32_t &vif, VifData::VifDataIter &iter)
{
  VifData::VifDataColl *vcoll;
  if (proto == "ppp") {
    vcoll = &_ppp_vif_coll;
  }
  else if (proto == "frame-relay") {
    vcoll = &_fr_vif_coll;
  }
  else if (proto == "cisco-hdlc") {
    vcoll = &_hdlc_vif_coll;
  }
  else {
    return false;
  }

  iter = vcoll->find(vif);
  if (iter == vcoll->end()) {
XLOG_INFO("Inserting vif, protocol %s (vif #%d)", proto.c_str(), vif);
    vcoll->insert(pair<uint32_t, VifData>(vif, VifData()));
    iter = vcoll->find(vif);
    //????    _add_vif_coll.insert(pair<IPv4, IfaceDelta>()));
  }
  return true;
}

/**
 *
 **/
VifData::VifDataColl
VifManager::expose(const string &proto)
{
  VifData::VifDataColl *vcoll, tmp_coll;
  if (proto == "ppp") {
    vcoll = &_ppp_vif_coll;
  }
  else if (proto == "frame-relay") {
    vcoll = &_fr_vif_coll;
  }
  else if (proto == "cisco-hdlc") {
    vcoll = &_hdlc_vif_coll;
  }
  else {
    return tmp_coll;
  }

  VifData::VifDataIter iter = vcoll->begin();
  while (iter != vcoll->end()) {
    if (iter->second._state != VifData::DELETE) {
      tmp_coll.insert(pair<uint32_t, VifData>(iter->first, iter->second));
    }
    ++iter;
  }
  return tmp_coll;
}

/**
 *
 **/
void
VifManager::erase(const string &proto, const uint32_t &vif)
{
  if (proto == "ppp") {
    VifData::VifDataIter iter = _ppp_vif_coll.find(vif);
    if (iter != _ppp_vif_coll.end()) {
      iter->second._state = VifData::DELETE;
    }
  }
  else if (proto == "frame-relay") {
    VifData::VifDataIter iter = _fr_vif_coll.find(vif);
    if (iter != _fr_vif_coll.end()) {
      iter->second._state = VifData::DELETE;
    }
  }
  else if (proto == "cisco-hdlc") {
    VifData::VifDataIter iter = _hdlc_vif_coll.find(vif);
    if (iter != _hdlc_vif_coll.end()) {
      iter->second._state = VifData::DELETE;
    }
  }
}

/**
 *
 **/
void
VifManager::erase(const string &proto)
{
  if (proto == "ppp") {
    VifData::VifDataIter iter = _ppp_vif_coll.begin();
    while (iter != _ppp_vif_coll.end()) {
      iter->second._state = VifData::DELETE;
      ++iter;
    }
  }
  else if (proto == "frame-relay") {
    VifData::VifDataIter iter = _fr_vif_coll.begin();
    while (iter != _fr_vif_coll.end()) {
      iter->second._state = VifData::DELETE;
      ++iter;
    }
  }
  else if (proto == "cisco-hdlc") {
    VifData::VifDataIter iter = _hdlc_vif_coll.begin();
    while (iter != _hdlc_vif_coll.end()) {
      iter->second._state = VifData::DELETE;
      ++iter;
    }
  }
}

/**
 *
 **/
void
VifManager::commit(const string &serial, const string &proto, IfaceDeltaColl &add_coll, IfaceDeltaColl &del_coll)
{
  UNUSED(proto);

  VifData::VifDataIter iter = _ppp_vif_coll.begin();
  while (iter != _ppp_vif_coll.end()) {
    if (iter->second._state == VifData::NEW) {
      add_coll.push_back(IfaceDelta(serial, iter->first, iter->second._address, iter->second._prefix_length));
      iter->second._state = VifData::CURRENT;
      ++iter;
    }
    else if (iter->second._state == VifData::DELETE) {
      del_coll.push_back(IfaceDelta(serial, iter->first, iter->second._address, iter->second._prefix_length));
      _ppp_vif_coll.erase(iter++);
    }
    else if (iter->second._state == VifData::CHANGED) {
      del_coll.push_back(IfaceDelta(serial, iter->first, iter->second._address, iter->second._prefix_length));
      add_coll.push_back(IfaceDelta(serial, iter->first, iter->second._address, iter->second._prefix_length));
      iter->second._state = VifData::CURRENT;
    }
    else {
      ++iter;
    }
  }

  iter = _fr_vif_coll.begin();
  while (iter != _fr_vif_coll.end()) {
    if (iter->second._state == VifData::NEW) {
      add_coll.push_back(IfaceDelta(serial, iter->first, iter->second._address, iter->second._prefix_length));
      iter->second._state = VifData::CURRENT;
      ++iter;
    }
    else if (iter->second._state == VifData::DELETE) {
      del_coll.push_back(IfaceDelta(serial, iter->first, iter->second._address, iter->second._prefix_length));
      _fr_vif_coll.erase(iter++);
    }
    else if (iter->second._state == VifData::CHANGED) {
      del_coll.push_back(IfaceDelta(serial, iter->first, iter->second._address, iter->second._prefix_length));
      add_coll.push_back(IfaceDelta(serial, iter->first, iter->second._address, iter->second._prefix_length));
      iter->second._state = VifData::CURRENT;
    }
    else {
      ++iter;
    }
  }

  iter = _hdlc_vif_coll.begin();
  while (iter != _hdlc_vif_coll.end()) {
    if (iter->second._state == VifData::NEW) {
      add_coll.push_back(IfaceDelta(serial, iter->first, iter->second._address, iter->second._prefix_length));
      iter->second._state = VifData::CURRENT;
      ++iter;
    }
    else if (iter->second._state == VifData::DELETE) {
      del_coll.push_back(IfaceDelta(serial, iter->first, iter->second._address, iter->second._prefix_length));
      _hdlc_vif_coll.erase(iter++);
    }
    else if (iter->second._state == VifData::CHANGED) {
      del_coll.push_back(IfaceDelta(serial, iter->first, iter->second._address, iter->second._prefix_length));
      add_coll.push_back(IfaceDelta(serial, iter->first, iter->second._address, iter->second._prefix_length));
      iter->second._state = VifData::CURRENT;
    }
    else {
      ++iter;
    }
  }
}
