/**
 * Module: xrl_rl_protocols_node.hh
 *
 * Author: Michael Larson
 * Date: 2005
 */

#include "rl_protocols_module.h"

#include "libxorp/xorp.h"
#include "libxorp/xlog.h"
#include "libxorp/debug.h"
#include "libxorp/ipvx.hh"
#include "libxorp/status_codes.h"

#include "librl_common/rl_validate.hh"
#include "rl_protocols_node.hh"

#include "xrl_rl_protocols_node.hh"

XrlRLProtocolsNode::XrlRLProtocolsNode(EventLoop&	eventloop,
				 const string&	class_name,
				 const string&	finder_hostname,
				 uint16_t	finder_port,
				 const string&	finder_target)
    : XrlStdRouter(eventloop, class_name.c_str(), finder_hostname.c_str(),
		   finder_port),
      XrlRlProtocolsTargetBase(&xrl_router()),
      _eventloop(eventloop),
      _class_name(xrl_router().class_name()),
      _instance_name(xrl_router().instance_name()),
      _finder_target(finder_target),
      _xrl_finder_client(&xrl_router()),
      _is_finder_alive(false),
      _status(PROC_STARTUP)
{
}

XrlRLProtocolsNode::~XrlRLProtocolsNode()
{
    rl_protocols_0_1_shutdown_protocols();
}

bool
XrlRLProtocolsNode::startup()
{
  _status = PROC_READY;
  return true;
}

XrlCmdError
XrlRLProtocolsNode::rl_protocols_0_1_shutdown_protocols()
{
  //handles shutdown of protocols and deletion of nodes?
  _status = PROC_SHUTDOWN;
  _rl_protocols_node.reset();
  return XrlCmdError::OKAY();
}

//
// Finder-related events
//
/**
 * Called when Finder connection is established.
 *
 * Note that this method overwrites an XrlRouter virtual method.
 */
void
XrlRLProtocolsNode::finder_connect_event()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    _is_finder_alive = true;
}

/**
 * Called when Finder disconnect occurs.
 *
 * Note that this method overwrites an XrlRouter virtual method.
 */
void
XrlRLProtocolsNode::finder_disconnect_event()
{
    XLOG_ERROR("Finder disconnect event. Exiting immediately...");
    //printf("%s\n",__PRETTY_FUNCTION__);

    _is_finder_alive = false;
}

//
// XRL target methods
//

/**
 *  Get name of Xrl Target
 */
XrlCmdError
XrlRLProtocolsNode::common_0_1_get_target_name(
    // Output values, 
    string&	name)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    name = my_xrl_target_name();

    return XrlCmdError::OKAY();
}

/**
 *  Get version string from Xrl Target
 */
XrlCmdError
XrlRLProtocolsNode::common_0_1_get_version(
    // Output values, 
    string&	version)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    version = XORP_MODULE_VERSION;

    return XrlCmdError::OKAY();
}

/**
 *  Get status of Xrl Target
 */
XrlCmdError
XrlRLProtocolsNode::common_0_1_get_status(
    // Output values, 
    uint32_t&	status, 
    string&	reason)
{
  reason = " ";
  status = _status;

  if (PROC_SHUTDOWN == _status) {
    _status = PROC_DONE;
  }

  return XrlCmdError::OKAY();
}

/**
 *  Request clean shutdown of Xrl Target
 */
XrlCmdError
XrlRLProtocolsNode::common_0_1_shutdown()
{
    string error_msg;
    return XrlCmdError::OKAY();
}

/**
 *  Announce target birth to observer.
 *
 *  @param target_class the target class name.
 *
 *  @param target_instance the target instance name.
 */
XrlCmdError
XrlRLProtocolsNode::finder_event_observer_0_1_xrl_target_birth(
    // Input values,
    const string&   target_class,
    const string&   target_instance)
{
    UNUSED(target_class);
    UNUSED(target_instance);

    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
}

/**
 *  Announce target death to observer.
 *
 *  @param target_class the target class name.
 *
 *  @param target_instance the target instance name.
 */
XrlCmdError
XrlRLProtocolsNode::finder_event_observer_0_1_xrl_target_death(
    // Input values,
    const string&   target_class,
    const string&   target_instance)
{
    UNUSED(target_class);
    UNUSED(target_instance);

    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
}

/**
 *  Enable/disable/start/stop RLProtocols.
 *
 *  @param enable if true, then enable RLProtocols, otherwise disable it.
 */
XrlCmdError
XrlRLProtocolsNode::rl_protocols_0_1_enable_rl_protocols(
    // Input values,
    const bool&	enable)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    UNUSED(enable);

    // XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

XrlCmdError
XrlRLProtocolsNode::rl_protocols_0_1_start_rl_protocols()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    // XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

XrlCmdError
XrlRLProtocolsNode::rl_protocols_0_1_stop_rl_protocols()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

/**
 *  Add an interface
 *
 *  @param add an interface
 */
XrlCmdError
XrlRLProtocolsNode::rl_protocols_0_1_add_interface(
    // Input values,
    const IPv4&	address)
{
    UNUSED(address);

    //printf("%s\n",__PRETTY_FUNCTION__);
    string error_msg;

    //ADD INTERFACE EXEC HERE!
    //printf("XrlRLProtocolsNode::rl_protocols_0_1_add_interface()--called!\n");

    return XrlCmdError::OKAY();
}

/**
 *  Enable/disable the RLProtocols trace log for all operations.
 *
 *  @param enable if true, then enable the trace log, otherwise disable it.
 */
XrlCmdError
XrlRLProtocolsNode::rl_protocols_0_1_enable_log_trace_all(
    // Input values,
    const bool&	enable)
{
    UNUSED(enable);

    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
}




XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_start_protocols_snmp_community_transaction(
									   //Input values,
)
{
  return _rl_protocols_node.start_snmp_community();
}
 
XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_commit_protocols_snmp_community_transaction(
										 //Input values,
										 )
{
  return _rl_protocols_node.commit_snmp_community();
}

XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_delete_protocols_snmp_community(
									    //Input values,
								     const string &community)
{
  return _rl_protocols_node.delete_snmp_community(community);
}

XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_set_protocols_snmp_community_authorization(
										//Input values,
										const string &authorization,
										const string &community)
{
  return _rl_protocols_node.set_snmp_authorization(authorization, community);
}

XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_delete_protocols_snmp_community_authorization(
									   //Input values,
									   const string &authorization,
									   const string &community)
{
  return _rl_protocols_node.delete_snmp_authorization(authorization, community);
}
  
XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_set_protocols_snmp_community_client(
									   //Input values,
									   const IPv4 &client,
									   const string &community)
{
    return _rl_protocols_node.set_snmp_community_client(client, community);
}
 
XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_delete_protocols_snmp_community_client(
									   //Input values,
									   const IPv4 &client,
									   const string &community)
{
    return _rl_protocols_node.delete_snmp_community_client(client, community);
}

XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_set_protocols_snmp_community_network(
									   //Input values,
									   const IPv4Net &network,
									   const string &community)
{
    return _rl_protocols_node.set_snmp_community_network(network, community);
}
 
XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_delete_protocols_snmp_community_network(
									   //Input values,
									   const IPv4Net &network,
									   const string &community)
{
    return _rl_protocols_node.delete_snmp_community_network(network, community);
}

XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_set_protocols_snmp_trap_target(
							 //Input values,
							 const IPv4 &trap_target)
{
    return _rl_protocols_node.set_snmp_trap_target(trap_target);
}

XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_delete_protocols_snmp_trap_target(
							    //Input values,
							    const IPv4 &trap_target)
{
    return _rl_protocols_node.delete_snmp_trap_target(trap_target);
}

XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_set_protocols_snmp_contact(
						     //Input values,
						     const string &contact)
{
    return _rl_protocols_node.set_snmp_contact(contact);
}

XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_delete_protocols_snmp_contact(
							//Input values,
							const string &contact)
{
    return _rl_protocols_node.delete_snmp_contact(contact);
}

XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_set_protocols_snmp_description(
							 //Input values,
							 const string &description)
{
    return _rl_protocols_node.set_snmp_description(description);
}

XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_delete_protocols_snmp_description(
							    //Input values,
							    const string &description)
{
    return _rl_protocols_node.delete_snmp_description(description);
}

XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_set_protocols_snmp_location(
						      //Input values,
						      const string &location)
{
    return _rl_protocols_node.set_snmp_location(location);

}

XrlCmdError 
XrlRLProtocolsNode::rl_protocols_0_1_delete_protocols_snmp_location(
							 //Input values,
							 const string &location)
{
    return _rl_protocols_node.delete_snmp_location(location);

}

XrlCmdError
XrlRLProtocolsNode::rl_protocols_0_1_delete_protocols_snmp( 
							   //Input values,
							   )
{
    return _rl_protocols_node.delete_snmp();
}
