/*
 * Module: rl_protocols_node.cc
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Michael Larson
 * Date: 2005
 * Description:
 *
 * **** End License ****
 *
 */

#include <map>
#include <set>
#include <string>

#include "config.h"
#include "rl_protocols_module.h"
#include "libxorp/xlog.h"
#include "librl_common/rl_fileaccess.hh"
#include "librl_common/rl_command.hh"
#include "rl_protocols_node.hh"

using namespace std;

/**
 *
 **/
RLProtocolsNode::RLProtocolsNode() :
  _snmpd_config_filename("/etc/snmp/snmpd.conf")
{
  //  reset();

  //now add the rlsnmpcounters
  string cmd = "iptables -A INPUT -m rlsnmpstats";
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLProtocolsNode: error on execution: %s", cmd.c_str());
  }

  cmd = "iptables -A OUTPUT -m rlsnmpstats";
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLProtocolsNode: error on execution: %s", cmd.c_str());
  }
}


/**
 *
 **/
RLProtocolsNode::~RLProtocolsNode()
{
}

/**
 *
 **/
XrlCmdError
RLProtocolsNode::reset()
{
  //copy over all default files
  /* snmpd.conf file */

  //remove any files

  //remove all filelocks, if any
  rl_fileaccess::init(_snmpd_config_filename);

  //stop any processes

  //hup any processes
  /* snmpd process ? */

  string cmd(SBINDIR "/snmpd.init stop");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLProtocolsNode: error on execution: %s", cmd.c_str());
  }


  return XrlCmdError::OKAY();
}
  
/**
 *
 **/
XrlCmdError 
RLProtocolsNode::start_snmp_community(
				      )
{
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLProtocolsNode::commit_snmp_community(
				       )
{
  /*
   * Rewrite this file on each commit--each line will be of the form:
   *
   */

  {

    //let's write to the snmpd config file
    rl_fileaccess fa(_snmpd_config_filename);
    
    //write to file only if this entry is not found
    FILE *fd_rd, *fd_wr;
    bool success = fa.get(fd_rd, fd_wr);
    if (success == false) {
      XLOG_ERROR("RLProtocolsNode: failed to open file: %s", _snmpd_config_filename.c_str());
      return XrlCmdError::COMMAND_FAILED();
    }
    
    string line = "dlmod bgp4_mib_1657 " MIBSDIR "/libbgp4_mib_1657.so\n";
    fputs(line.c_str(), fd_wr);
    line = "dlmod xorp_if_mib_module " MIBSDIR "/libxorp_if_mib_module.so\n";
    fputs(line.c_str(), fd_wr);
    line = "dlmod ospf_mib_1850 " MIBSDIR "/libospf_mib_1850.so\n";
    fputs(line.c_str(), fd_wr);
    line = "trap2sink localhost xorp 51510\n";
    fputs(line.c_str(), fd_wr);

    
    SNMPIter iter = _snmp_coll.begin();
    while (iter != _snmp_coll.end()) {
      
      if (strcasecmp(iter->second._authorization.c_str(), "ro") == 0) {
	line = "rocommunity\t" + iter->first;
      } else {
	line = "rwcommunity\t" + iter->first;
      }
      
      //now write out our community strings here
      bool found = false;

      ClientData::ClientIter client_iter = iter->second._client_coll.begin();
      while (client_iter != iter->second._client_coll.end()) {
	found = true;
	string access_line = line + "\t" + client_iter->str() + "\n";
	fputs(access_line.c_str(), fd_wr);
	++client_iter;
      }

      ClientData::NetworkIter network_iter = iter->second._network_coll.begin();
      while (network_iter != iter->second._network_coll.end()) {
	found = true;
	string access_line = line + "\t" + network_iter->str() + "\n";
	fputs(access_line.c_str(), fd_wr);
	++network_iter;
      }

      if (found == false) {
	string access_line = line + "\n";
	fputs(access_line.c_str(), fd_wr);
      }

      ++iter;
    }
    
    if (!_contact.empty()) {
      string contact = "syscontact \t\"" + _contact + "\"\n";
      fputs(contact.c_str(), fd_wr);
    }
    
    if (!_description.empty()) {
      string description = "sysdescr \t\"" + _description + "\"\n";
      fputs(description.c_str(), fd_wr);
    }

    if (!_location.empty()) {
      string location = "syslocation \t\"" + _location + "\"\n";
      fputs(location.c_str(), fd_wr);
    }

    if (!_trap_target.empty()) {
      list<IPv4>::iterator i;
      
      for(i=_trap_target.begin(); i!=_trap_target.end(); i++){
	string trap_target = "trapsink \t" + i->str() + "\n";
	fputs(trap_target.c_str(), fd_wr);
      }
    }
  }
  
  string cmd(SBINDIR "/snmpd.init restart");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLProtocolsNode: error on execution: %s", cmd.c_str());
  }

  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLProtocolsNode::delete_snmp_community(
				       const string &community)
{
  _snmp_coll.erase(community);
  return commit_snmp_community();
}

XrlCmdError 
RLProtocolsNode::set_snmp_authorization(
					//Input values,
					const string &authorization,
					const string &community)
{
  SNMPIter iter = _snmp_coll.find(community);
  if (iter == _snmp_coll.end()) {
    _snmp_coll.insert(pair<string, ClientData>(community, ClientData()));
    iter = _snmp_coll.find(community);
  }
  iter->second._authorization = authorization;
  return XrlCmdError::OKAY();
}

XrlCmdError 
RLProtocolsNode::delete_snmp_authorization(
					   //Input values,
					   const string &authorization,
					   const string &community)
{
  SNMPIter iter = _snmp_coll.find(community);
  if (iter == _snmp_coll.end()) {
    _snmp_coll.insert(pair<string, ClientData>(community, ClientData()));
    iter = _snmp_coll.find(community);
  }
  iter->second._authorization = string("");
  return XrlCmdError::OKAY();
  UNUSED(authorization);
}
  



/**
 *
 **/
XrlCmdError 
RLProtocolsNode::set_snmp_community_client(
					const IPv4 &client,
					const string &community)
{

  SNMPIter iter = _snmp_coll.find(community);
  if (iter == _snmp_coll.end()) {
    _snmp_coll.insert(pair<string, ClientData>(community, ClientData()));
    iter = _snmp_coll.find(community);
  }
  iter->second._client_coll.insert(client);
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLProtocolsNode::delete_snmp_community_client(
					      const IPv4 &client,
					      const string &community)
{
  SNMPIter iter = _snmp_coll.find(community);
  if (iter == _snmp_coll.end()) {
    _snmp_coll.insert(pair<string, ClientData>(community, ClientData()));
    iter = _snmp_coll.find(community);
  }
  iter->second._client_coll.erase(client);
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLProtocolsNode::set_snmp_community_network(
					const IPv4Net &network,
					const string &community)
{

  SNMPIter iter = _snmp_coll.find(community);
  if (iter == _snmp_coll.end()) {
    _snmp_coll.insert(pair<string, ClientData>(community, ClientData()));
    iter = _snmp_coll.find(community);
  }
  iter->second._network_coll.insert(network);
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLProtocolsNode::delete_snmp_community_network(
					      const IPv4Net &network,
					      const string &community)
{
  SNMPIter iter = _snmp_coll.find(community);
  if (iter == _snmp_coll.end()) {
    _snmp_coll.insert(pair<string, ClientData>(community, ClientData()));
    iter = _snmp_coll.find(community);
  }
  iter->second._network_coll.erase(network);
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLProtocolsNode::set_snmp_trap_target(
					//Input values,
					const IPv4 &trap_target)
{
  _trap_target.push_back(trap_target);
  return XrlCmdError::OKAY();
}


/*
 * Commands below are not part of the transaction.
 *
 */

/**
 *
 **/
XrlCmdError 
RLProtocolsNode::delete_snmp_trap_target(
				    //Input values,
				    const IPv4 &trap_target)
{
  _trap_target.remove(trap_target);
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLProtocolsNode::set_snmp_contact(
				   //Input values,
			       const string &contact)
{
  _contact = contact;
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLProtocolsNode::delete_snmp_contact(
				   //Input values,
				  const string &contact)
{
  if (contact == _contact)
    _contact.clear();
  return XrlCmdError::OKAY();

}
 
/**
 *
 **/
XrlCmdError 
RLProtocolsNode::set_snmp_description(
				   //Input values,
				   const string &description)
{
  _description = description;
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLProtocolsNode::delete_snmp_description(
				    //Input values,
				    const string &description)
{
  if (description == _description)
    _description.clear();
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLProtocolsNode::set_snmp_location(
			      const string &location)
{
  _location = location;
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLProtocolsNode::delete_snmp_location(
				 const string &location)
{
  if (location == _location)
    _location.clear();
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLProtocolsNode::delete_snmp(
			       //Input values,
			    )
{
  return reset();
}
