/**
 * Module: xrl_rl_interfaces_tunnel_node.hh
 *
 * Author: Michael Larson
 * Date: 2006
 */

#ifndef __XRL_RL_INTERFACES_TUNNEL_NODE_HH__
#define __XRL_RL_INTERFACES_TUNNEL_NODE_HH__


//
// RLINTERFACESTUNNEL dummy XRL-aware node definition.
//

#include "libxorp/status_codes.h"
#include "libxipc/xrl_std_router.hh"
#include "xrl/interfaces/finder_event_notifier_xif.hh"
#include "xrl/interfaces/fea_ifmgr_xif.hh"

#include "xrl/targets/rl_interfaces_tunnel_base.hh"

class RLFeaMgr;

//
// The top-level class that wraps-up everything together under one roof
//
class XrlRLInterfacesTunnelNode :     public XrlStdRouter, /* used for dispatch of xrl messages */
			      public XrlRlInterfacesTunnelTargetBase { /* used for defined interface */
public:
  typedef std::map<std::string, RLFeaMgr*> RLFeaColl;
  typedef std::map<std::string, RLFeaMgr*>::iterator RLFeaIter;

public:
  XrlRLInterfacesTunnelNode(EventLoop&	eventloop,
			    const string&	class_name,
			    const string&	finder_hostname,
			    uint16_t	        finder_port,
			    const string&	finder_target,
			    const string&       fea_target);
  ~XrlRLInterfacesTunnelNode(); 
  
  /**
   * Startup the node operation.
   *
   * @return true on success, false on failure.
   */
  bool	startup();

  void fea_register_startup();
  
  void finder_register_interest_fea_cb(const XrlError& xrl_error);
  void fea_register_shutdown();
  void finder_deregister_interest_fea_cb(const XrlError& xrl_error);


  /**
   * Shutdown the node operation.
   *
   * @return true on success, false on failure.
   */
  XrlCmdError rl_interfaces_tunnel_0_1_shutdown();
  XrlCmdError rl_interfaces_tunnel_0_1_end_commit();
  
  /**
   *
   *
   */
  XrlCmdError rl_interfaces_tunnel_0_1_set_tunnel_interface(
							    const string &name, 
							    const IPv4 &address, 
							    const uint32_t &prefix_length);
  XrlCmdError rl_interfaces_tunnel_0_1_delete_tunnel_interface(
							       const string &name, 
							       const IPv4 &address, 
							       const uint32_t &prefix_length);
  
  /**
   * Get a reference to the XrlRouter instance.
   *
   * @return a reference to the XrlRouter (@ref XrlRouter) instance.
   */
  XrlRouter&	xrl_router() { return *this; }

  /**
   * Test if the node processing is done.
   * 
   * @return true if the node processing is done, otherwise false.
   */
  bool	is_done() const { return (_status == PROC_DONE); }
  
  XrlCmdError complete_add(const string &iface);
  XrlCmdError complete_delete(const string &iface);

protected:
  //
  // XRL target methods
  //
  
  /**
   *  Get name of Xrl Target
   */
  XrlCmdError common_0_1_get_target_name(
					 // Output values, 
					 string&	name);
  
  /**
   *  Get version string from Xrl Target
   */
  XrlCmdError common_0_1_get_version(
				     // Output values, 
				     string&	version);
  
  /**
   *  Get status of Xrl Target
   */
  XrlCmdError common_0_1_get_status(
				    // Output values, 
				    uint32_t&	status, 
				    string&	reason);
  
  /**
   *  Request clean shutdown of Xrl Target
   */
  XrlCmdError common_0_1_shutdown();
  
  /**
   *  Announce target birth to observer.
   *
   *  @param target_class the target class name.
   *
   *  @param target_instance the target instance name.
   */
  XrlCmdError finder_event_observer_0_1_xrl_target_birth(
							 // Input values,
							 const string&	target_class,
							 const string&	target_instance);
  
  /**
   *  Announce target death to observer.
   *
   *  @param target_class the target class name.
   *
   *  @param target_instance the target instance name.
   */
  XrlCmdError finder_event_observer_0_1_xrl_target_death(
							 // Input values,
							 const string&	target_class,
							 const string&	target_instance);
  
private:
  /**
   * Called when Finder connection is established.
   *
   * Note that this method overwrites an XrlRouter virtual method.
   */
  virtual void finder_connect_event();
  
  /**
   * Called when Finder disconnect occurs.
   *
   * Note that this method overwrites an XrlRouter virtual method.
   */
  virtual void finder_disconnect_event();
  
  const string& my_xrl_target_name() {
    return XrlRlInterfacesTunnelTargetBase::name();
  }
  
  EventLoop&	_eventloop;
  const string	_class_name;
  const string	_instance_name;
  const string	_finder_target;
  const string  _fea_target;
  
  XrlFinderEventNotifierV0p1Client	_xrl_finder_client;
  
  static const TimeVal RETRY_TIMEVAL;
  
  bool		_is_finder_alive;

  uint32_t         _status;

  bool		_is_fea_registered;
  bool		_is_fea_registering;
  bool		_is_fea_deregistering;
  XorpTimer		_fea_register_startup_timer;
  XorpTimer		_fea_register_shutdown_timer;
  XrlIfmgrV0p1Client	_xrl_fea_client;  
  bool		_is_fea_alive;

  RLFeaColl       _fea_mgr_add_coll;
  RLFeaColl       _fea_mgr_del_coll;
};

#endif // __XRL_RL_INTERFACES_TUNNEL_NODE_HH__
