/**
 * Module: xrl_rl_interfaces_tunnel_node.hh
 *
 * Author: Michael Larson
 * Date: 2006
 */

#include "rl_interfaces_tunnel_module.h"

#include "libxorp/xorp.h"
#include "libxorp/xlog.h"
#include "libxorp/debug.h"
#include "libxorp/ipvx.hh"
#include "libxorp/status_codes.h"
#include "rl_interfaces_tunnel_fea.hh"
#include "xrl_rl_interfaces_tunnel_node.hh"

const TimeVal XrlRLInterfacesTunnelNode::RETRY_TIMEVAL = TimeVal(1, 0);

/**
 *
 */
XrlRLInterfacesTunnelNode::XrlRLInterfacesTunnelNode(EventLoop&	eventloop,
						     const string&	class_name,
						     const string&	finder_hostname,
						     uint16_t	finder_port,
						     const string&	finder_target,
						     const string&      fea_target)
    : XrlStdRouter(eventloop, class_name.c_str(), finder_hostname.c_str(),
		   finder_port),
      XrlRlInterfacesTunnelTargetBase(&xrl_router()),
      _eventloop(eventloop),
      _class_name(xrl_router().class_name()),
      _instance_name(xrl_router().instance_name()),
      _finder_target(finder_target),
      _fea_target(fea_target),
      _xrl_finder_client(&xrl_router()),
      _is_finder_alive(false),
      _status(PROC_STARTUP),
      _is_fea_registered(false),
      _is_fea_registering(false),
      _is_fea_deregistering(false),
      _xrl_fea_client(&xrl_router()),
      _is_fea_alive(false)
{
}

/**
 *
 */
XrlRLInterfacesTunnelNode::~XrlRLInterfacesTunnelNode()
{
  rl_interfaces_tunnel_0_1_shutdown();
}

/**
 *
 */
bool
XrlRLInterfacesTunnelNode::startup()
{
  fea_register_startup();
  _status = PROC_READY;
  return true;
}

//
// XRL target methods
//
//
// Register with the FEA
//
void
XrlRLInterfacesTunnelNode::fea_register_startup()
{
    bool success;

    _fea_register_startup_timer.unschedule();
    _fea_register_shutdown_timer.unschedule();

    if (! _is_finder_alive)
	return;		// The Finder is dead

    if (_is_fea_registered)
	return;		// Already registered

    //removing for now from static routes examples
    if (! _is_fea_registering) {
      //>	StaticRoutesNode::incr_startup_requests_n();	// XXX: for the ifmgr
	_is_fea_registering = true;
    }

    //
    // Register interest in the FEA with the Finder
    //
    success = _xrl_finder_client.send_register_class_event_interest(
	_finder_target.c_str(), _instance_name, _fea_target,
	callback(this, &XrlRLInterfacesTunnelNode::finder_register_interest_fea_cb));

    if (! success) {
	//
	// If an error, then start a timer to try again.
	//
	_fea_register_startup_timer = _eventloop.new_oneoff_after(
	    RETRY_TIMEVAL,
	    callback(this, &XrlRLInterfacesTunnelNode::fea_register_startup));
	return;
    }
}

void
XrlRLInterfacesTunnelNode::finder_register_interest_fea_cb(const XrlError& xrl_error)
{
  XLOG_INFO("XrlRLInterfacesTunnelNode::finder_register_interest_fea_cb: %d", xrl_error.error_code());
  

    switch (xrl_error.error_code()) {
    case OKAY:
	//
	// If success, then the FEA birth event will startup the ifmgr
	//
	_is_fea_registering = false;
	_is_fea_registered = true;
	break;

    case COMMAND_FAILED:
	//
	// If a command failed because the other side rejected it, this is
	// fatal.
	//
	XLOG_FATAL("Cannot register interest in finder events: %s",
		   xrl_error.str().c_str());
	break;

    case NO_FINDER:
    case RESOLVE_FAILED:
    case SEND_FAILED:
	//
	// A communication error that should have been caught elsewhere
	// (e.g., by tracking the status of the finder and the other targets).
	// Probably we caught it here because of event reordering.
	// In some cases we print an error. In other cases our job is done.
	//
	XLOG_ERROR("XRL communication error: %s", xrl_error.str().c_str());
	break;

    case BAD_ARGS:
    case NO_SUCH_METHOD:
    case INTERNAL_ERROR:
	//
	// An error that should happen only if there is something unusual:
	// e.g., there is XRL mismatch, no enough internal resources, etc.
	// We don't try to recover from such errors, hence this is fatal.
	//
	XLOG_FATAL("Fatal XRL error: %s", xrl_error.str().c_str());
	break;

    case REPLY_TIMED_OUT:
    case SEND_FAILED_TRANSIENT:
	//
	// If a transient error, then start a timer to try again
	// (unless the timer is already running).
	//
	if (! _fea_register_startup_timer.scheduled()) {
	    XLOG_ERROR("Failed to register interest in finder events: %s. "
		       "Will try again.",
		       xrl_error.str().c_str());
	    _fea_register_startup_timer = _eventloop.new_oneoff_after(
		RETRY_TIMEVAL,
		callback(this, &XrlRLInterfacesTunnelNode::fea_register_startup));
	}
	break;
    }
}

//
// De-register with the FEA
//
void
XrlRLInterfacesTunnelNode::fea_register_shutdown()
{
    bool success;

    _fea_register_startup_timer.unschedule();
    _fea_register_shutdown_timer.unschedule();

    if (! _is_finder_alive)
	return;		// The Finder is dead

    if (! _is_fea_alive)
	return;		// The FEA is not there anymore

    if (! _is_fea_registered)
	return;		// Not registered

    if (! _is_fea_deregistering) {
      //	StaticRoutesNode::incr_shutdown_requests_n();	// XXX: for the ifmgr

	_is_fea_deregistering = true;
    }

    //
    // De-register interest in the FEA with the Finder
    //
    success = _xrl_finder_client.send_deregister_class_event_interest(
	_finder_target.c_str(), _instance_name, _fea_target,
	callback(this, &XrlRLInterfacesTunnelNode::finder_deregister_interest_fea_cb));

    if (! success) {
	//
	// If an error, then start a timer to try again.
	//
	_fea_register_shutdown_timer = _eventloop.new_oneoff_after(
	    RETRY_TIMEVAL,
	    callback(this, &XrlRLInterfacesTunnelNode::fea_register_shutdown));
	return;
    }
}

void
XrlRLInterfacesTunnelNode::finder_deregister_interest_fea_cb(
    const XrlError& xrl_error)
{
    switch (xrl_error.error_code()) {
    case OKAY:
	//
	// If success, then we are done
	//
	_is_fea_deregistering = false;
	_is_fea_registered = false;
	XLOG_INFO("RLInterfacesTunnelNode:: finder_dergistered_interface_fea_cb");
	break;

    case COMMAND_FAILED:
	//
	// If a command failed because the other side rejected it, this is
	// fatal.
	//
	XLOG_FATAL("Cannot deregister interest in finder events: %s",
		   xrl_error.str().c_str());
	break;

    case NO_FINDER:
    case RESOLVE_FAILED:
    case SEND_FAILED:
	//
	// A communication error that should have been caught elsewhere
	// (e.g., by tracking the status of the finder and the other targets).
	// Probably we caught it here because of event reordering.
	// In some cases we print an error. In other cases our job is done.
	//
	_is_fea_deregistering = false;
	_is_fea_registered = false;
	XLOG_INFO("RLInterfacesTunnelNode:: SEND_FAILED");
	break;

    case BAD_ARGS:
    case NO_SUCH_METHOD:
    case INTERNAL_ERROR:
	//
	// An error that should happen only if there is something unusual:
	// e.g., there is XRL mismatch, no enough internal resources, etc.
	// We don't try to recover from such errors, hence this is fatal.
	//
	XLOG_FATAL("Fatal XRL error: %s", xrl_error.str().c_str());
	break;

    case REPLY_TIMED_OUT:
    case SEND_FAILED_TRANSIENT:
	//
	// If a transient error, then start a timer to try again
	// (unless the timer is already running).
	//
	if (! _fea_register_shutdown_timer.scheduled()) {
	    XLOG_ERROR("Failed to deregister interest in finder events: %s. "
		       "Will try again.",
		       xrl_error.str().c_str());
	    _fea_register_shutdown_timer = _eventloop.new_oneoff_after(
		RETRY_TIMEVAL,
		callback(this, &XrlRLInterfacesTunnelNode::fea_register_shutdown));
	}
	break;
    }
}




/**
 *
 */
XrlCmdError
XrlRLInterfacesTunnelNode::rl_interfaces_tunnel_0_1_shutdown()
{
  //handles shutdown of system and deletion of nodes?
  _status = PROC_SHUTDOWN;
  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
XrlRLInterfacesTunnelNode::rl_interfaces_tunnel_0_1_end_commit()
{
  //handles shutdown of system and deletion of nodes?
  //here is where we'll grab the message return
  return XrlCmdError::OKAY();//XrlCmdError::COMMAND_FAILED("FOO");
}


/**
 *
 */
XrlCmdError 
XrlRLInterfacesTunnelNode::rl_interfaces_tunnel_0_1_set_tunnel_interface(const string &name, const IPv4 &address, const uint32_t &prefix_length)
{
  //here is where we need to notify xorp of the new interface
  RLFeaMgr *fea_mgr = new RLFeaMgr(_xrl_fea_client, _fea_target, name, address, prefix_length, false, this, &XrlRLInterfacesTunnelNode::complete_add);
  //we'll want to wire these into the current settings of the state
  _fea_mgr_add_coll.insert(pair<string,RLFeaMgr*>(name, fea_mgr));
  fea_mgr->set_finder_state(_is_finder_alive);
  fea_mgr->set_fea_interest(_is_fea_registered);
  fea_mgr->process();
  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
XrlRLInterfacesTunnelNode::complete_add(const string &iface)
{
  //continue on with other shutdown work...

  //now mark the internal housekeeping as complete
  //and let the module to be shutdown by the rtrmgr.
  RLFeaIter iter = _fea_mgr_add_coll.find(iface);

  if (iter != _fea_mgr_add_coll.end() && 
      iter->second != NULL) {
    delete iter->second;
    _fea_mgr_add_coll.erase(iter);
  }

  //now proceed with the add...
  //can also proceed with other operations after interface has been added
  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError 
XrlRLInterfacesTunnelNode::rl_interfaces_tunnel_0_1_delete_tunnel_interface(const string &name, const IPv4 &address, const uint32_t &prefix_length)
{
  //here is where we need to remove the interface from xorp
  RLFeaMgr *fea_mgr = new RLFeaMgr(_xrl_fea_client, _fea_target, name, address, prefix_length, true, this, &XrlRLInterfacesTunnelNode::complete_delete);
  //we'll want to wire these into the current settings of the state
  _fea_mgr_del_coll.insert(pair<string,RLFeaMgr*>(name, fea_mgr));
  fea_mgr->set_finder_state(_is_finder_alive);
  fea_mgr->set_fea_interest(_is_fea_registered);
  fea_mgr->process();
  return XrlCmdError::OKAY();
}


/**
 *
 */
XrlCmdError
XrlRLInterfacesTunnelNode::complete_delete(const string &iface)
{
  //continue on with other shutdown work...

  //now mark the internal housekeeping as complete
  //and let the module to be shutdown by the rtrmgr.
  RLFeaIter iter = _fea_mgr_del_coll.find(iface);

  if (iter != _fea_mgr_del_coll.end() && 
      iter->second != NULL) {
    delete iter->second;
    _fea_mgr_del_coll.erase(iter);
  }

  //now proceed with the add...
  //can also proceed with other operations after interface has been added
  return XrlCmdError::OKAY();
}

//
// Finder-related events
//
/**
 * Called when Finder connection is established.
 *
 * Note that this method overwrites an XrlRouter virtual method.
 */
void
XrlRLInterfacesTunnelNode::finder_connect_event()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    _is_finder_alive = true;
}

/**
 * Called when Finder disconnect occurs.
 *
 * Note that this method overwrites an XrlRouter virtual method.
 */
void
XrlRLInterfacesTunnelNode::finder_disconnect_event()
{
    XLOG_ERROR("Finder disconnect event. Exiting immediately...");
    //printf("%s\n",__PRETTY_FUNCTION__);

    _is_finder_alive = false;
}

//
// XRL target methods
//

/**
 *  Get name of Xrl Target
 */
XrlCmdError
XrlRLInterfacesTunnelNode::common_0_1_get_target_name(
    // Output values, 
    string&	name)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    name = my_xrl_target_name();

    return XrlCmdError::OKAY();
}

/**
 *  Get version string from Xrl Target
 */
XrlCmdError
XrlRLInterfacesTunnelNode::common_0_1_get_version(
    // Output values, 
    string&	version)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    version = XORP_MODULE_VERSION;

    return XrlCmdError::OKAY();
}

/**
 *  Get status of Xrl Target
 */
XrlCmdError
XrlRLInterfacesTunnelNode::common_0_1_get_status(
    // Output values, 
    uint32_t&	status, 
    string&	reason)
{
  reason = " ";
  status = _status;

  if (PROC_SHUTDOWN == _status) {
    _status = PROC_DONE;
  }

  return XrlCmdError::OKAY();
}

/**
 *  Request clean shutdown of Xrl Target
 */
XrlCmdError
XrlRLInterfacesTunnelNode::common_0_1_shutdown()
{
    string error_msg;
    //printf("%s\n",__PRETTY_FUNCTION__);
    /*
    if (shutdown() != true) {
	error_msg = c_format("Failed to shutdown RLInterfacesTunnel");
	return XrlCmdError::COMMAND_FAILED(error_msg);
    }
    */
    return XrlCmdError::OKAY();
}

/**
 *  Announce target birth to observer.
 *
 *  @param target_class the target class name.
 *
 *  @param target_instance the target instance name.
 */
XrlCmdError
XrlRLInterfacesTunnelNode::finder_event_observer_0_1_xrl_target_birth(
    // Input values,
    const string&   target_class,
    const string&   target_instance)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
    UNUSED(target_class);
    UNUSED(target_instance);
}

/**
 *  Announce target death to observer.
 *
 *  @param target_class the target class name.
 *
 *  @param target_instance the target instance name.
 */
XrlCmdError
XrlRLInterfacesTunnelNode::finder_event_observer_0_1_xrl_target_death(
    // Input values,
    const string&   target_class,
    const string&   target_instance)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
    UNUSED(target_class);
    UNUSED(target_instance);
}

