/*
 * Module: rl_interfaces_tunnel_fea_mgr.cc
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Michael Larson
 * Date: 2005
 * Description:
 *
 * **** End License ****
 *
 */
#include <string>
#include "libxorp/status_codes.h"
#include "libxipc/xrl_std_router.hh"
#include "libfeaclient/ifmgr_xrl_mirror.hh"
#include "xrl/interfaces/finder_event_notifier_xif.hh"
#include "xrl/interfaces/fea_ifmgr_xif.hh"
#include "libxorp/callback.hh"
#include "rl_interfaces_tunnel_fea.hh"


using namespace std;

/**
 *
 **/
RLFeaMgr::RLFeaMgr(
		   XrlIfmgrV0p1Client &fea_client, 
		   const string &fea_target, 
		   const string &iface, 
		   const IPv4 &address,
		   uint32_t prefix_length,
		   bool remove, 
		   XrlRLInterfacesTunnelNode *parent, 
		   XrlCmdError (XrlRLInterfacesTunnelNode::*cb)(const std::string&)) :
  _xrl_fea_client(fea_client),
  _fea_target(fea_target),
  _iface(iface),
  _address(address),
  _prefix_length(prefix_length),
  _remove(remove),
  _is_finder_alive(false),
  _is_fea_registered(false),
  _parent(parent),
  _cb(cb)
{
  XLOG_INFO("RLFeaMgr::RLFeaMgr");
}

/**
 *
 **/
void
RLFeaMgr::process()
{
  XLOG_INFO("RLFeaMgr::process");

  if (_is_finder_alive == false) {
    XLOG_ERROR("RLFeaMgr::process() finder is not alive");
    return;
  }
  if (_is_fea_registered == false) {
    XLOG_ERROR("RLFeaMgr::process() fea is not registered");
    return;
  }
  
  if (_remove) {
    start_delete_interface(_iface); //delete
  }
  else {
    start_add_interface(_iface); //add
  }
}


/**
 *
 **/
void
RLFeaMgr::start_delete_interface(const string &iface)
{
    XLOG_INFO("RLFeaMgr::start_delete_interface() starting to set interface: %s to %s/%d", iface.c_str(), _address.str().c_str(), _prefix_length);
  if (_is_finder_alive == false) {
    XLOG_ERROR("RLFeaMgr::start_delete_interface() finder is not alive");
    return;		// The Finder is dead
  }
  
  if (_is_fea_registered == true) {
    bool success = _xrl_fea_client.send_start_transaction(_fea_target.c_str(), callback(this,  &RLFeaMgr::delete_start_transaction_cb, _address));
    if (success != true) {
      XLOG_ERROR("Failed to send start transaction.");
    }
  }
  else {
    XLOG_ERROR("RLFeaMgr::start_delete_interface() fea is not registered");
    return;
  }
}

/**
 *
 **/
void
RLFeaMgr::start_add_interface(const string &iface)
{
  if (_is_finder_alive == false) {
    XLOG_ERROR("RLFeaMgr::start_add_interface() finder is not alive");
    return;		// The Finder is dead
  }
  
  if (_is_fea_registered == true) {
    XLOG_INFO("RLFeaMgr::start_add_interface() starting to set interface: %s to %s/%d", iface.c_str(), _address.str().c_str(), _prefix_length);
    
    bool success = _xrl_fea_client.send_start_transaction(_fea_target.c_str(), callback(this,  &RLFeaMgr::add_start_transaction_cb, _address));
    if (success != true) {
      XLOG_ERROR("Failed to start transaction.");
    }
  }
  else {
    XLOG_ERROR("RLFeaMgr::start_add_interface() fea is not registered");
    return;
  }
}




/**
 *
 **/
void 
RLFeaMgr::delete_start_transaction_cb(const XrlError &xrl_error, const uint32_t *tid, IPv4 address)
{
  UNUSED(address);
  XLOG_INFO("RLSerialNode::delete_start_transaction_cb(): %d", *tid);

  delete_interface(*tid, _iface);

  UNUSED(xrl_error);
}

/**
 *
 **/
void 
RLFeaMgr::add_start_transaction_cb(const XrlError &xrl_error, const uint32_t *tid, IPv4 address)
{
  UNUSED(address);
  //retreive tid and set
  XLOG_INFO("RLSerialNode::send_add_start_transactioncb(): %d for address: %s", *tid, address.str().c_str());

  add_interface(*tid, _iface);

  UNUSED(xrl_error);
}


/**
 *
 **/
void
RLFeaMgr::delete_interface(const uint32_t tid, const string &iface)
{
  XLOG_INFO("RLFeaMgr::delete_interface: %s", iface.c_str());
  bool success;

  if (_is_fea_registered == true) {
    XLOG_INFO("RLSerialNode::fea_client_send_delete_interface() now with first call in transaction, setting tid: %d", tid);
    
    success = _xrl_fea_client.send_delete_vif(_fea_target.c_str(), tid, iface, iface, callback(this,  &RLFeaMgr::send_transaction_cb));
    if (success != true) {
      XLOG_ERROR("Failed to delete vif.");
    }
    
    success = _xrl_fea_client.send_delete_interface(_fea_target.c_str(), tid, iface, callback(this,  &RLFeaMgr::send_transaction_cb));
    if (success != true) {
      XLOG_ERROR("Failed to delete interface.");
    }
    
    success = _xrl_fea_client.send_commit_transaction(_fea_target.c_str(), tid, callback(this,  &RLFeaMgr::complete_transaction_cb, iface));
    if (success != true) {
      XLOG_ERROR("Failed to commit transaction.");
    }
  }
  else {
    XLOG_ERROR("RLFeaMgr::delete_interface() fea is not registered");
    return;
  }
}


/**
 *
 **/
void
RLFeaMgr::add_interface(const uint32_t tid, const string &iface)
{
  bool success;

  if (_is_fea_registered == true) {

    XLOG_INFO("RLFeaMgr::fea_client_complete_add_interface() now with first call in transaction, setting tid: %d for %s/%d on %s", tid, _address.str().c_str(), _prefix_length, iface.c_str());
    
    success = _xrl_fea_client.send_create_interface(_fea_target.c_str(), tid, iface, callback(this,  &RLFeaMgr::send_transaction_cb));
    if (success != true) {
      XLOG_ERROR("Failed to create interface.");
    }
    
    success = _xrl_fea_client.send_set_interface_enabled(_fea_target.c_str(), tid, iface, true, callback(this,  &RLFeaMgr::send_transaction_cb));
    if (success != true) {
      XLOG_ERROR("Failed to set interface enabled.");
    }

    success = _xrl_fea_client.send_set_interface_discard(_fea_target.c_str(), tid, iface, false,callback(this,  &RLFeaMgr::send_transaction_cb));
    if (success != true) {
      XLOG_ERROR("Failed to set interface discard.");
    }

    success = _xrl_fea_client.send_create_vif(_fea_target.c_str(), tid, iface, iface, callback(this,  &RLFeaMgr::send_transaction_cb));
    if (success != true) {
      XLOG_ERROR("Failed to send_create vif.");
    }

    success = _xrl_fea_client.send_set_vif_enabled(_fea_target.c_str(), tid, iface, iface, true, callback(this,  &RLFeaMgr::send_transaction_cb));
    if (success != true) {
      XLOG_ERROR("Failed to send_create vif enabled.");
    }

    success = _xrl_fea_client.send_create_address4(_fea_target.c_str(), tid, iface, iface, _address, callback(this,  &RLFeaMgr::send_transaction_cb));
    if (success != true) {
      XLOG_ERROR("Failed to send_create create address.");
    }

    success = _xrl_fea_client.send_set_prefix4(_fea_target.c_str(), tid, iface, iface, _address, _prefix_length, callback(this,  &RLFeaMgr::send_transaction_cb));
    if (success != true) {
      XLOG_ERROR("Failed to set set prefix.");
    }

    success = _xrl_fea_client.send_set_address_enabled4(_fea_target.c_str(), tid, iface, iface, _address,  true, callback(this,  &RLFeaMgr::send_transaction_cb));
    if (success != true) {
      XLOG_ERROR("Failed to set address enabled.");
    }

    success = _xrl_fea_client.send_commit_transaction(_fea_target.c_str(), tid, callback(this,  &RLFeaMgr::complete_transaction_cb, iface));
    if (success != true) {
      XLOG_ERROR("Failed to commit transaction.");
    }

    XLOG_INFO("RLFeaMgr::fea_client_send_add_interface() finished setting interface: %s to %s/%d", iface.c_str(), _address.str().c_str(), _prefix_length);
  }
  else {
    XLOG_ERROR("RLFeaMgr::add_interface() fea is not registered");
    return;
  }
}

/**
 *
 **/
void 
RLFeaMgr::send_transaction_cb(const XrlError &xrl_error)

{
  XLOG_INFO("RLFeaMgr::send_transaction_cb");
  UNUSED(xrl_error);
}

/**
 *
 **/
void 
RLFeaMgr::complete_transaction_cb(const XrlError &xrl_error, string iface)
{
  XLOG_INFO("RLFeaMgr::delete_complete_transaction_cb");
  UNUSED(xrl_error);

  //if this is the last one then continue to main commit statement
  if (_parent != NULL) {
    XLOG_INFO("RLFeaMgr::delete_complete_transaction_cb (final callback now)");
    (*_parent.*_cb)(iface);
  }
}

