/**
 * Module: xrl_rl_interfaces_node.hh
 *
 * Author: Michael Larson
 * Date: 2005
 */

#ifndef __RL_INTERFACES_XRL_RL_INTERFACES_NODE_HH__
#define __RL_INTERFACES_XRL_RL_INTERFACES_NODE_HH__


//
// RLSystem XRL-aware node definition.
//
#include "libxorp/status_codes.h"
#include "libxipc/xrl_std_router.hh"
#include "libfeaclient/ifmgr_xrl_mirror.hh"
#include "xrl/interfaces/finder_event_notifier_xif.hh"
#include "xrl/interfaces/fea_ifmgr_xif.hh"
#include "xrl/targets/rl_interfaces_base.hh"
#include "libxorp/callback.hh"
#include "rl_interfaces_node.hh"

//
// The top-level class that wraps-up everything together under one roof
//
class XrlRLInterfacesNode :     public XrlStdRouter, /* used for dispatch of xrl messages */
			    public XrlRlInterfacesTargetBase { /* used for defined interface */
public:
  XrlRLInterfacesNode(EventLoop&	eventloop,
		      const string&	class_name,
		      const string&	finder_hostname,
		      uint16_t	        finder_port,
		      const string&	finder_target,
		      const string&     fea_target);
  ~XrlRLInterfacesNode();
  
  /**
   * Startup the node operation.
   *
   * @return true on success, false on failure.
   */
  bool	startup();
  
  /**
   * Shutdown the node operation.
   *
   * @return true on success, false on failure.
   */
  XrlCmdError	rl_interfaces_0_1_shutdown_interfaces();
  
  /**
   * Get a reference to the XrlRouter instance.
   *
   * @return a reference to the XrlRouter (@ref XrlRouter) instance.
   */
  XrlRouter&	xrl_router() { return *this; }
  
  /**
   * Test if the node processing is done.
   * 
   * @return true if the node processing is done, otherwise false.
   */
    bool	is_done() const { return (_status == PROC_DONE); }
  
protected:
  //
  // XRL target methods
  //

  /**
   *  Get name of Xrl Target
   */
  XrlCmdError common_0_1_get_target_name(
					 // Output values, 
					 string&	name);
  
  /**
   *  Get version string from Xrl Target
   */
  XrlCmdError common_0_1_get_version(
				     // Output values, 
				     string&	version);
  
  /**
   *  Get status of Xrl Target
   */
  XrlCmdError common_0_1_get_status(
				    // Output values, 
				    uint32_t&	status, 
				    string&	reason);
  
  /**
   *
   */
  XrlCmdError rl_interfaces_0_1_start_interface(
						//Input values,
						const string &interface) {UNUSED(interface);return XrlCmdError::OKAY();}

  XrlCmdError rl_interfaces_0_1_commit_interface(
								  //Input values,
							  const string &interface) {UNUSED(interface);return XrlCmdError::OKAY();}

  /**
   *  Request clean shutdown of Xrl Target
   */
  XrlCmdError common_0_1_shutdown();
  
  /**
   *  Announce target birth to observer.
   *
   *  @param target_class the target class name.
   *
   *  @param target_instance the target instance name.
   */
  XrlCmdError finder_event_observer_0_1_xrl_target_birth(
							 // Input values,
							 const string&	target_class,
							 const string&	target_instance);
  
  /**
   *  Announce target death to observer.
   *
   *  @param target_class the target class name.
   *
   *  @param target_instance the target instance name.
   */
  XrlCmdError finder_event_observer_0_1_xrl_target_death(
							 // Input values,
							 const string&	target_class,
							 const string&	target_instance);
  
  /**
   *  Enable/disable/start/stop RLInterfaces
   *
   *  @param enable if true, then enable RLInterfaces, otherwise disable it.
   */
  XrlCmdError rl_interfaces_0_1_enable_rl_interfaces(
						     // Input values,
						     const bool&	enable);
  
  XrlCmdError rl_interfaces_0_1_start_rl_interfaces();
  
  XrlCmdError rl_interfaces_0_1_stop_rl_interfaces();
  XrlCmdError rl_interfaces_0_1_delete_interfaces(
						  //Input values,
						  const string &interface);
  
  XrlCmdError rl_interfaces_0_1_delete_interfaces_vif(
						      //Input values,
						      const string &interface);
  
  
  /**
   *  Enable/disable the RLInterfaces trace log for all operations.
   *
   *  @param enable if true, then enable the trace log, otherwise disable it.
   */
  XrlCmdError rl_interfaces_0_1_enable_log_trace_all(
						     // Input values,
						     const bool&	enable);
  
  
  
  XrlCmdError rl_interfaces_0_1_start_commit();
  XrlCmdError rl_interfaces_0_1_end_commit();
  
  XrlCmdError rl_interfaces_0_1_set_interfaces_vif_vrrp_group(
							      //Input values,
							      const string &interface,
							      const string &authentication,
							      const uint32_t &group,
							      const IPv4 &virtual_address,
							      const uint32_t &advertise_interval,
							      const bool &preempt,
							      const uint32_t &priority,
							      const bool &disable_vmac);
  
  XrlCmdError rl_interfaces_0_1_delete_interfaces_vif_vrrp_group(
								 //Input values,
								 const string &interface,
								 const string &authentication,
								 const uint32_t &group,
								 const IPv4 &virtual_address,
								 const uint32_t &advertise_interval,
								 const bool &preempt,
								 const uint32_t &priority,
								 const bool &disable_vmac);

  
private:
  /**
   * Called when Finder connection is established.
   *
   * Note that this method overwrites an XrlRouter virtual method.
   */
  virtual void finder_connect_event();
  
  /**
   * Called when Finder disconnect occurs.
   *
   * Note that this method overwrites an XrlRouter virtual method.
   */
  virtual void finder_disconnect_event();
  
  void fea_register_startup();
  void finder_register_interest_fea_cb(const XrlError& xrl_error);
  void fea_register_shutdown();
  void finder_deregister_interest_fea_cb(const XrlError& xrl_error);

  const string& my_xrl_target_name() {
    return XrlRlInterfacesTargetBase::name();
  }
  
  EventLoop&		_eventloop;
  const string	_class_name;
  const string	_instance_name;
  const string	_finder_target;
  const string	_fea_target;

  IfMgrXrlMirror	_ifmgr;

  XrlFinderEventNotifierV0p1Client	_xrl_finder_client;
  
  static const TimeVal RETRY_TIMEVAL;
  
  bool		_is_finder_alive;
  bool		_is_fea_registered;
  bool		_is_fea_registering;
  bool		_is_fea_deregistering;
  XorpTimer		_fea_register_startup_timer;
  XorpTimer		_fea_register_shutdown_timer;

  XrlIfmgrV0p1Client	_xrl_fea_client;  

  bool		_is_fea_alive;

  RLInterfacesNode   _rl_interfaces_node;

  uint32_t      _status;
};

#endif // __RL_INTERFACES_XRL_RL_INTERFACES_NODE_HH__
