/**
 * Module: xrl_rl_interfaces_node.hh
 *
 * Author: Michael Larson
 * Date: 2005
 */

#include "rl_interfaces_module.h"
#include "libxorp/xorp.h"
#include "libxorp/xlog.h"
#include "libxorp/debug.h"
#include "libxorp/ipvx.hh"
#include "libxorp/status_codes.h"
#include "xrl/interfaces/fea_ifmgr_xif.hh"
#include "rl_interfaces_node.hh"
#include "xrl_rl_interfaces_node.hh"

const TimeVal XrlRLInterfacesNode::RETRY_TIMEVAL = TimeVal(1, 0);

XrlRLInterfacesNode::XrlRLInterfacesNode(EventLoop&	eventloop,
					 const string&	class_name,
					 const string&	finder_hostname,
					 uint16_t	finder_port,
					 const string&	finder_target,
					 const string& fea_target)
    : XrlStdRouter(eventloop, class_name.c_str(), finder_hostname.c_str(),
		   finder_port),
      XrlRlInterfacesTargetBase(&xrl_router()),
      _eventloop(eventloop),
      _class_name(xrl_router().class_name()),
      _instance_name(xrl_router().instance_name()),
      _finder_target(finder_target),
      _fea_target(fea_target),
      _ifmgr(eventloop, fea_target.c_str(), xrl_router().finder_address(),
	     xrl_router().finder_port()),
      _xrl_finder_client(&xrl_router()),
      _is_finder_alive(false),
      _is_fea_registered(false),
      _is_fea_registering(false),
      _is_fea_deregistering(false),
      _xrl_fea_client(&xrl_router()),
      _is_fea_alive(false),
      _status(PROC_STARTUP)
{
}

XrlRLInterfacesNode::~XrlRLInterfacesNode()
{
    rl_interfaces_0_1_shutdown_interfaces();
}

bool
XrlRLInterfacesNode::startup()
{
  fea_register_startup();
  _status = PROC_READY;
  return true;
}

XrlCmdError
XrlRLInterfacesNode::rl_interfaces_0_1_shutdown_interfaces()
{
  //handles shutdown of interfaces and deletion of nodes?
  return _rl_interfaces_node.reset();
}

//
// Finder-related events
//
/**
 * Called when Finder connection is established.
 *
 * Note that this method overwrites an XrlRouter virtual method.
 */
void
XrlRLInterfacesNode::finder_connect_event()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    _is_finder_alive = true;
}

/**
 * Called when Finder disconnect occurs.
 *
 * Note that this method overwrites an XrlRouter virtual method.
 */
void
XrlRLInterfacesNode::finder_disconnect_event()
{
    XLOG_ERROR("Finder disconnect event. Exiting immediately...");
    //printf("%s\n",__PRETTY_FUNCTION__);

    _is_finder_alive = false;
}

//
// XRL target methods
//
//
// Register with the FEA
//
void
XrlRLInterfacesNode::fea_register_startup()
{
    bool success;

    XLOG_INFO("XrlrLInterfacesNode::fea_register_start(): A");
    _fea_register_startup_timer.unschedule();
    _fea_register_shutdown_timer.unschedule();

    if (! _is_finder_alive)
	return;		// The Finder is dead

    if (_is_fea_registered)
	return;		// Already registered

    XLOG_INFO("XrlrLInterfacesNode::fea_register_start(): B");
    //removing for now from static routes examples
    if (! _is_fea_registering) {
      //>	StaticRoutesNode::incr_startup_requests_n();	// XXX: for the ifmgr
	_is_fea_registering = true;
    }

    XLOG_INFO("XrlrLInterfacesNode::fea_register_start(): C");
    //
    // Register interest in the FEA with the Finder
    //
    success = _xrl_finder_client.send_register_class_event_interest(
	_finder_target.c_str(), _instance_name, _fea_target,
	callback(this, &XrlRLInterfacesNode::finder_register_interest_fea_cb));

    if (! success) {
	//
	// If an error, then start a timer to try again.
	//
	_fea_register_startup_timer = _eventloop.new_oneoff_after(
	    RETRY_TIMEVAL,
	    callback(this, &XrlRLInterfacesNode::fea_register_startup));
	return;
    }
    XLOG_INFO("XrlrLInterfacesNode::fea_register_start(): D");
}

void
XrlRLInterfacesNode::finder_register_interest_fea_cb(const XrlError& xrl_error)
{
  XLOG_INFO("RLInterfacesNode::finder_register_interest_fea_cb: %d", xrl_error.error_code());
  

    switch (xrl_error.error_code()) {
    case OKAY:
	//
	// If success, then the FEA birth event will startup the ifmgr
	//
	_is_fea_registering = false;
	_is_fea_registered = true;
	XLOG_INFO("RLInterfacesNode:: TESTA: REGISTERED FEA");
	break;

    case COMMAND_FAILED:
	//
	// If a command failed because the other side rejected it, this is
	// fatal.
	//
	XLOG_FATAL("Cannot register interest in finder events: %s",
		   xrl_error.str().c_str());
	break;

    case NO_FINDER:
    case RESOLVE_FAILED:
    case SEND_FAILED:
	//
	// A communication error that should have been caught elsewhere
	// (e.g., by tracking the status of the finder and the other targets).
	// Probably we caught it here because of event reordering.
	// In some cases we print an error. In other cases our job is done.
	//
	XLOG_ERROR("XRL communication error: %s", xrl_error.str().c_str());
	break;

    case BAD_ARGS:
    case NO_SUCH_METHOD:
    case INTERNAL_ERROR:
	//
	// An error that should happen only if there is something unusual:
	// e.g., there is XRL mismatch, no enough internal resources, etc.
	// We don't try to recover from such errors, hence this is fatal.
	//
	XLOG_FATAL("Fatal XRL error: %s", xrl_error.str().c_str());
	break;

    case REPLY_TIMED_OUT:
    case SEND_FAILED_TRANSIENT:
	//
	// If a transient error, then start a timer to try again
	// (unless the timer is already running).
	//
	if (! _fea_register_startup_timer.scheduled()) {
	    XLOG_ERROR("Failed to register interest in finder events: %s. "
		       "Will try again.",
		       xrl_error.str().c_str());
	    _fea_register_startup_timer = _eventloop.new_oneoff_after(
		RETRY_TIMEVAL,
		callback(this, &XrlRLInterfacesNode::fea_register_startup));
	}
	break;
    }
}

//
// De-register with the FEA
//
void
XrlRLInterfacesNode::fea_register_shutdown()
{
    bool success;

    _fea_register_startup_timer.unschedule();
    _fea_register_shutdown_timer.unschedule();

    if (! _is_finder_alive)
	return;		// The Finder is dead

    if (! _is_fea_alive)
	return;		// The FEA is not there anymore

    if (! _is_fea_registered)
	return;		// Not registered

    if (! _is_fea_deregistering) {
      //	StaticRoutesNode::incr_shutdown_requests_n();	// XXX: for the ifmgr

	_is_fea_deregistering = true;
    }

    //
    // De-register interest in the FEA with the Finder
    //
    success = _xrl_finder_client.send_deregister_class_event_interest(
	_finder_target.c_str(), _instance_name, _fea_target,
	callback(this, &XrlRLInterfacesNode::finder_deregister_interest_fea_cb));

    if (! success) {
	//
	// If an error, then start a timer to try again.
	//
	_fea_register_shutdown_timer = _eventloop.new_oneoff_after(
	    RETRY_TIMEVAL,
	    callback(this, &XrlRLInterfacesNode::fea_register_shutdown));
	return;
    }

    //
    // XXX: when the shutdown is completed, StaticRoutesNode::status_change()
    // will be called.
    //
    _ifmgr.shutdown();
}

void
XrlRLInterfacesNode::finder_deregister_interest_fea_cb(
    const XrlError& xrl_error)
{
    switch (xrl_error.error_code()) {
    case OKAY:
	//
	// If success, then we are done
	//
	_is_fea_deregistering = false;
	_is_fea_registered = false;
	XLOG_INFO("RLInterfacesNode:: TESTB: finder_dergistered_interface_fea_cb");
	break;

    case COMMAND_FAILED:
	//
	// If a command failed because the other side rejected it, this is
	// fatal.
	//
	XLOG_FATAL("Cannot deregister interest in finder events: %s",
		   xrl_error.str().c_str());
	break;

    case NO_FINDER:
    case RESOLVE_FAILED:
    case SEND_FAILED:
	//
	// A communication error that should have been caught elsewhere
	// (e.g., by tracking the status of the finder and the other targets).
	// Probably we caught it here because of event reordering.
	// In some cases we print an error. In other cases our job is done.
	//
	_is_fea_deregistering = false;
	_is_fea_registered = false;
	XLOG_INFO("RLInterfacesNode:: TESTC: SEND_FAILED");
	break;

    case BAD_ARGS:
    case NO_SUCH_METHOD:
    case INTERNAL_ERROR:
	//
	// An error that should happen only if there is something unusual:
	// e.g., there is XRL mismatch, no enough internal resources, etc.
	// We don't try to recover from such errors, hence this is fatal.
	//
	XLOG_FATAL("Fatal XRL error: %s", xrl_error.str().c_str());
	break;

    case REPLY_TIMED_OUT:
    case SEND_FAILED_TRANSIENT:
	//
	// If a transient error, then start a timer to try again
	// (unless the timer is already running).
	//
	if (! _fea_register_shutdown_timer.scheduled()) {
	    XLOG_ERROR("Failed to deregister interest in finder events: %s. "
		       "Will try again.",
		       xrl_error.str().c_str());
	    _fea_register_shutdown_timer = _eventloop.new_oneoff_after(
		RETRY_TIMEVAL,
		callback(this, &XrlRLInterfacesNode::fea_register_shutdown));
	}
	break;
    }
}


/**
 *  Get name of Xrl Target
 */
XrlCmdError
XrlRLInterfacesNode::common_0_1_get_target_name(
    // Output values, 
    string&	name)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    name = my_xrl_target_name();

    return XrlCmdError::OKAY();
}

/**
 *  Get version string from Xrl Target
 */
XrlCmdError
XrlRLInterfacesNode::common_0_1_get_version(
    // Output values, 
    string&	version)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    version = XORP_MODULE_VERSION;

    return XrlCmdError::OKAY();
}

/**
 *  Get status of Xrl Target
 */
XrlCmdError
XrlRLInterfacesNode::common_0_1_get_status(
    // Output values, 
    uint32_t&	status, 
    string&	reason)
{

  reason = " ";
  status = _status;

  if (PROC_SHUTDOWN == _status) {
    _status = PROC_DONE;
  }

  return XrlCmdError::OKAY();
}

/**
 *  Request clean shutdown of Xrl Target
 */
XrlCmdError
XrlRLInterfacesNode::common_0_1_shutdown()
{
    string error_msg;
    //printf("%s\n",__PRETTY_FUNCTION__);
    /*
    if (shutdown() != true) {
	error_msg = c_format("Failed to shutdown RLInterfaces");
	return XrlCmdError::COMMAND_FAILED(error_msg);
    }
    */
    return XrlCmdError::OKAY();
}

/**
 *  Announce target birth to observer.
 *
 *  @param target_class the target class name.
 *
 *  @param target_instance the target instance name.
 */
XrlCmdError
XrlRLInterfacesNode::finder_event_observer_0_1_xrl_target_birth(
    // Input values,
    const string&   target_class,
    const string&   target_instance)
{
    UNUSED(target_class);
    UNUSED(target_instance);

    if (target_class == _fea_target) {
	//
	// XXX: when the startup is completed,
	// IfMgrHintObserver::tree_complete() will be called.
	//
	_is_fea_alive = true;
	if (_ifmgr.startup() != true) {
	  /*	    StaticRoutesNode::ServiceBase::set_status(SERVICE_FAILED);
		    StaticRoutesNode::update_status();*/
	}
    }

    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
}

/**
 *  Announce target death to observer.
 *
 *  @param target_class the target class name.
 *
 *  @param target_instance the target instance name.
 */
XrlCmdError
XrlRLInterfacesNode::finder_event_observer_0_1_xrl_target_death(
    // Input values,
    const string&   target_class,
    const string&   target_instance)
{
    UNUSED(target_class);
    UNUSED(target_instance);

    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
}

/**
 *  Enable/disable/start/stop RLInterfaces.
 *
 *  @param enable if true, then enable RLInterfaces, otherwise disable it.
 */
XrlCmdError
XrlRLInterfacesNode::rl_interfaces_0_1_enable_rl_interfaces(
    // Input values,
    const bool&	enable)
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    UNUSED(enable);

    // XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

XrlCmdError
XrlRLInterfacesNode::rl_interfaces_0_1_start_rl_interfaces()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    // XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

XrlCmdError
XrlRLInterfacesNode::rl_interfaces_0_1_stop_rl_interfaces()
{
    //printf("%s\n",__PRETTY_FUNCTION__);
    XLOG_UNFINISHED();

    return XrlCmdError::OKAY();
}

XrlCmdError 
XrlRLInterfacesNode::rl_interfaces_0_1_delete_interfaces(
							 //Input values,
							 const string &interface)
{
    /*
     * This method is called with the vif interface is deleted. In 
     * this case we need to remove all vrrp groups and vlan-id.
     */
    printf("rl_interfaces_0_1_delete_interfaces: also needs to clean up other interface elements\n");
    return _rl_interfaces_node.delete_vrrp(interface);
}


XrlCmdError 
XrlRLInterfacesNode::rl_interfaces_0_1_start_commit()
{
  return XrlCmdError::OKAY();
}
XrlCmdError 
XrlRLInterfacesNode::rl_interfaces_0_1_end_commit()
{
  return XrlCmdError::OKAY();
}

XrlCmdError 
XrlRLInterfacesNode::rl_interfaces_0_1_delete_interfaces_vif(
							 //Input values,
							     const string &interface)
{
    /*
     * This method is called with the vif interface is deleted. In 
     * this case we need to remove all vrrp groups and vlan-id.
     */
    return _rl_interfaces_node.delete_vrrp(interface);
}


/**
 *  Enable/disable the RLInterfaces trace log for all operations.
 *
 *  @param enable if true, then enable the trace log, otherwise disable it.
 */
XrlCmdError
XrlRLInterfacesNode::rl_interfaces_0_1_enable_log_trace_all(
    // Input values,
    const bool&	enable)
{
    UNUSED(enable);

    //printf("%s\n",__PRETTY_FUNCTION__);
    return XrlCmdError::OKAY();
}


XrlCmdError 
XrlRLInterfacesNode::rl_interfaces_0_1_set_interfaces_vif_vrrp_group(
							     //Input values,
							      const string &interface,
							      const string &authentication,
							      const uint32_t &group,
							      const IPv4 &virtual_address,
							      const uint32_t &advertise_interval,
							      const bool &preempt,
							      const uint32_t &priority,
							      const bool &disable_vmac)
{
    return _rl_interfaces_node.set_vrrp_group(interface, authentication, group, virtual_address, advertise_interval, preempt, priority, disable_vmac);
}

XrlCmdError 
XrlRLInterfacesNode::rl_interfaces_0_1_delete_interfaces_vif_vrrp_group(
								//Input values,
								 const string &interface,
								 const string &authentication,
								 const uint32_t &group,
								 const IPv4 &virtual_address,
								 const uint32_t &advertise_interval,
								 const bool &preempt,
								 const uint32_t &priority,
								 const bool &disable_vmac)
{
    return _rl_interfaces_node.delete_vrrp_group(interface, authentication, group, virtual_address, advertise_interval, preempt, priority, disable_vmac);
}
