/*
 * Module: rl_interfaces_node.cc
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Michael Larson
 * Date: 2005
 * Description:
 *
 * **** End License ****
 *
 */

#include <string>
#include <dirent.h>
#include "rl_interfaces_module.h"
#include "libxorp/xlog.h"
#include "librl_common/rl_fileaccess.hh"
#include "librl_common/rl_command.hh"
#include "librl_common/rl_interface.hh"
#include "rl_interfaces_node.hh"
/**
 *
 */
RLInterfacesNode::RLInterfacesNode() :
  _vrrpd_filename("/var/vrrpd_state")
{
  //set up 8021q support in the kernel
  string cmd = "modprobe 8021q";
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("rl_interface failed to execute command: %s", cmd.c_str());
  }
  reset();
}

/**
 *
 */
RLInterfacesNode::~RLInterfacesNode()
{
}

/**
 *
 */
XrlCmdError
RLInterfacesNode::reset()
{
  XLOG_INFO("RLInterfacesNode::reset(), resetting module");
  
  //copy over all default files
  rl_fileaccess::init(_vrrpd_filename);

  //remove any files

  // /var/vrrpd_state
  string cmd = "/bin/echo > " + _vrrpd_filename;
  if (rl_command::execute(cmd) == false) {
      XLOG_ERROR("RLInterfaceNode: error on execution: %s", cmd.c_str());
  }

  // /var/log/vrrpd/*
  cmd = "rm -f /var/log/vrrpd/*.log";
  if (rl_command::execute(cmd) == false) {
      XLOG_ERROR("RLInterfaceNode: error on execution: %s", cmd.c_str());
  }

  //remove all filelocks, if any

  //stop any processes
  /* kill all vrrpd processes */
  cmd = string(SBINDIR "/vrrpd.init stop");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("rl_interface failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  
  //hup any processes

  return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
RLInterfacesNode::set_vrrp_group(
				   //Input values,
				   const string &interface,
				   const string &authentication,
				   const uint32_t &group,
				   const IPv4 &virtual_address,
				   const uint32_t &advertise_interval,
				   const bool &preempt,
				   const uint32_t &priority,
				   const bool &disable_vmac)
{
  /*
   * First need to check if the vrrpd daemon is running. Need to identify
   * by group number.
   */
    std::string cmd;
    cmd = interface;
    
    //first check the state of the interface
    string address;
    if (rl_interface::is_configured(interface, address) == false) {
	XLOG_ERROR("rl_interface interface is not configured: %s", interface.c_str());
	return XrlCmdError::COMMAND_FAILED("Interface is not configured with an address.");
    }

    if (address == virtual_address.str()) {
	XLOG_ERROR("rl_interface address is identical to vrrp virtual address: %s", address.c_str());
	return XrlCmdError::COMMAND_FAILED("Interface address is the same as the vrrp virtual address.");
    }

    //let's limit the sessions to one per physical device
    string device_root = interface.substr(0, 4); //truncate to ethX
    VRRPIter iter = _vrrp_coll.begin();
    while (iter != _vrrp_coll.end()) {
      if (iter->first.substr(0, 4) == device_root &&
	  iter->first != interface) {
	XLOG_ERROR("rl_interface only one vrrp session allowed per physical device: %s", address.c_str());
	return XrlCmdError::COMMAND_FAILED("Only one vrrp session is allowed per physical interface.");
      }
      ++iter;
    }

    iter = _vrrp_coll.find(interface);
    if (iter != _vrrp_coll.end()) {
      //will stop the current instance and remove the instance
      string start_cmd = string(SBINDIR "/vrrpd.init stop ") + iter->first;
      printf("vrrpd command: %s\n", start_cmd.c_str());
      if (rl_command::execute(start_cmd) == false) {
	XLOG_ERROR("rl_interface failed to execute command: %s", start_cmd.c_str());
	return XrlCmdError::COMMAND_FAILED();
      }
      _vrrp_coll.erase(iter);
    }

    char buf[20];
    sprintf(buf, " %d ", group);
    cmd += string(buf); 
    
    sprintf(buf, "%d", priority);
    cmd += string(" ") + buf;

    cmd += string(" ") + virtual_address.str();

    sprintf(buf, "%d", advertise_interval);
    cmd += string(" ") + string(buf);

    if (preempt == false) {
	cmd += string(" no");
    }
    else {
	cmd += string(" yes");
    }

    if (authentication.empty() == true) {
	cmd += " -1";
    }
    else {
	cmd += " " + authentication;
    }

    if (disable_vmac == false) {
	cmd += string(" no");
    }
    else {
	cmd += string(" yes");
    }

    string start_cmd = string(SBINDIR "/vrrpd.init restart ") + cmd;
    printf("vrrpd command: %s\n", start_cmd.c_str());
    if (rl_command::execute(start_cmd) == false) {
	XLOG_ERROR("rl_interface failed to execute command: %s", start_cmd.c_str());
	return XrlCmdError::COMMAND_FAILED();
    }

    //add the new configuration
    _vrrp_coll.insert(pair<string, string>(interface, cmd));

    //dump out the vrrp session
    dump_vrrp_state();

    return XrlCmdError::OKAY();
}


/**
 *
 */
XrlCmdError
RLInterfacesNode::delete_vrrp_group(
				    //Input values,
				    const string &interface,
				    const string &authentication,
				    const uint32_t group,
				    const IPv4 &virtual_address,
				    const uint32_t advertise_interval,
				    const bool preempt,
				    const uint32_t priority,
				    const bool disable_vmac)
{
  UNUSED(group);
  UNUSED(virtual_address);
  UNUSED(authentication);
  UNUSED(advertise_interval);
  UNUSED(preempt);
  UNUSED(priority);
  UNUSED(disable_vmac);
  
  std::string cmd;
  cmd = interface;


    VRRPIter iter = _vrrp_coll.find(interface);
    if (iter != _vrrp_coll.end()) {
      //will stop the current instance and remove the instance
      string start_cmd = string(SBINDIR "/vrrpd.init stop ") + iter->first;
      printf("vrrpd command: %s\n", start_cmd.c_str());
      if (rl_command::execute(start_cmd) == false) {
	XLOG_ERROR("rl_interface failed to execute command: %s", start_cmd.c_str());
	return XrlCmdError::COMMAND_FAILED();
      }
      _vrrp_coll.erase(iter);
    }

    dump_vrrp_state();

    return XrlCmdError::OKAY();
}

/**
 *
 */
XrlCmdError
RLInterfacesNode::delete_vrrp(
			      //Input values,
			      const string &interface)
{
  string empty_str("");
  IPv4 empty_ip;
  //just delegate the call
  return delete_vrrp_group(interface, empty_str, uint32_t(1), empty_ip, uint32_t(1), true, uint32_t(1),true);
}







/**
 *
 */
XrlCmdError
RLInterfacesNode::dump_vrrp_state()
{
  //dump out file here
  {
    rl_fileaccess fa(_vrrpd_filename);
    FILE *fd_rd, *fd_wr;
    bool success = fa.get(fd_rd, fd_wr);
    if (success == false) {
      XLOG_ERROR("rl_interfaces_node: failed to open file: %s", _vrrpd_filename.c_str());
      return XrlCmdError::COMMAND_FAILED();
    }
    
    VRRPIter iter = _vrrp_coll.begin();
    while (iter != _vrrp_coll.end()) {
      fputs(iter->second.c_str(), fd_wr);
      ++iter;
    }
  }  
  return XrlCmdError::OKAY();
}

