/*
 * Module: rl_interface_firewall_node.cc
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Michael Larson
 * Date: 2005
 * Description:
 *
 * **** End License ****
 *
 */

#include <string>
#include <map>
#include "rl_interface_firewall_module.h"
#include "libxorp/xlog.h"
#include "libxipc/xrl_atom.hh"
#include "librl_common/rl_command.hh"
#include "rl_interface_firewall_node.hh"

using namespace std;

/**
 *
 **/
string
InterfaceFW::dump() const
{
  return string("["+string(_in_interface?"true":"false")+","+_name+"]");
}

/**
 *
 **/
IFWKey::IFWKey(const string &interface, const string &firewall) :
  _interface(interface),
  _firewall(firewall)
{
}

/**
 *
 **/
bool IFWKey::operator<(const IFWKey &key) const
{
  if (_interface < key._interface) {
    return false;
  }
  else if (_interface == key._interface) {
    if (_firewall <= key._firewall) {
      return false;
    }
  }
  return true;
}

/**
 *
 **/
bool IFWKey::operator==(const IFWKey &key) const 
{
  return (_interface == key._interface &&
      _firewall == key._firewall);
}

/**
 *
 **/
RLInterfaceFirewallNode::RLInterfaceFirewallNode() 
{
  reset();
}

/**
 *
 **/
RLInterfaceFirewallNode::~RLInterfaceFirewallNode()
{
}

/**
 *
 **/
XrlCmdError
RLInterfaceFirewallNode::reset()
{
  //clear out all commands
  //  delete_all();

  /*
    PER the latest prd, these need to be applied on startup
    iptables --flush FORWARD
    iptables --policy FORWARD ACCEPT
    iptables --flush INPUT
    iptables --policy INPUT ACCEPT
    iptables --append INPUT --source 127.0.0.1 .destination 127.0.0.1 .jump RETURN
    iptables .N LOGDROP
    iptables --append LOGDROP --jump LOG
    iptables --append LOGDROP --jump DROP
    iptables .N LOGREJECT
    iptables --append LOGDROP --jump LOG
    iptables --append LOGDROP --jump REJECT
    iptables .N LOGACCEPT
    iptables --append LOGDROP --jump LOG
    iptables --append LOGDROP --jump ACCEPT
  */
  string cmd;
  
  cmd = "iptables --flush FORWARD";
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLInterfaceFirewallNode failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }

  cmd = "iptables --policy FORWARD ACCEPT";
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLInterfaceFirewallNode failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }

  cmd = "iptables --flush INPUT";
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLInterfaceFirewallNode failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }

  cmd = "iptables --policy INPUT ACCEPT";
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLInterfaceFirewallNode failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }

  cmd = "iptables --append INPUT --source 127.0.0.1 --destination 127.0.0.1 --jump ACCEPT";
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLInterfaceFirewallNode failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }

  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLInterfaceFirewallNode::set_firewall(const string &name, const string &firewall, const string &interface)
{
  if (name.empty() == true || firewall.empty() == true || interface.empty() == true) {
    XLOG_ERROR("RLInterfaceFirewallNode: commands are empty");
    return XrlCmdError::COMMAND_FAILED("Name or filewall or interface not specified");
  }
  
  //ignore error
  string rule_spec;
  string cmd = "iptables --new " + name;
  rl_command::execute(cmd); //ignore error
  
  cmd = "iptables --append ";
  if (strcasecmp(firewall.c_str(), "in") == 0) {
    rule_spec = "FORWARD --in-interface " + interface + " --jump " + name;
    _forward_firewall.insert(pair<IFWKey, InterfaceFW>(IFWKey(interface, firewall), InterfaceFW(true, name, rule_spec)));
  }
  else if (strcasecmp(firewall.c_str(), "out") == 0) {
    rule_spec = "FORWARD --out-interface " + interface + " --jump " + name;
    _forward_firewall.insert(pair<IFWKey, InterfaceFW>(IFWKey(interface, firewall), InterfaceFW(false, name, rule_spec)));
  }
  else { //"local"
    rule_spec = "INPUT --in-interface " + interface + " --jump " + name;
    _input_firewall.insert(pair<IFWKey, InterfaceFW>(IFWKey(interface, firewall), InterfaceFW(true, name, rule_spec)));
  }
  cmd += rule_spec;
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLInterfaceFirewallNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }

  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLInterfaceFirewallNode::delete_firewall(const string &name, const string &firewall, const string &interface)
{
  FirewallNameColl *coll;
  
  if (name.empty() == true) {
    //this would happen if the root node is defined (firewall), but not the name.
    return XrlCmdError::OKAY();
  }

  bool in_flag;
  if (strcasecmp(firewall.c_str(), "in") == 0) {
    in_flag = true;
    coll = &_forward_firewall;
  }
  else if (strcasecmp(firewall.c_str(), "out") == 0) {
    in_flag = false;
    coll = &_forward_firewall;
  }
  else { //"local"
    in_flag = true;
    coll = &_input_firewall;
  }


  FirewallNameIter iter = coll->find(IFWKey(interface, firewall));
  if (iter == coll->end()) {
    XLOG_ERROR("RLInterfaceFirewallNode::delete_firewall failed to find firewall %s for %s on interface %s", name.c_str(), firewall.c_str(), interface.c_str());
    /*
      test dump of collection
     */
    FirewallNameIter ii = coll->begin();
    while (ii != coll->end()) {
      XLOG_ERROR("collection dump: %s, %s", ii->first._interface.c_str(), ii->first._firewall.c_str());
      ++ii;
    }
    return XrlCmdError::OKAY();
  }

  string cmd = "iptables --delete " + iter->second._rule_spec;
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLInterfaceFirewallNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }

  coll->erase(iter);

  //check to see if there are any references to this chain.
  //we'll make this more elegant by rolling this into the comparitor
  bool found = false;
  iter = _input_firewall.begin();
  while (iter != _input_firewall.end() && found == false) {
    if (iter->second._name == name) {
      found = true;
    }
    ++iter;
  }

  iter = _forward_firewall.begin();
  while (iter != _forward_firewall.end() && found == false) {
    if (iter->second._name == name) {
      found = true;
    }
    ++iter;
  }

  if (found == false) {
    cmd = "iptables --delete-chain " + name;
    //ignore error from this command, execute anyway
    rl_command::execute(cmd);
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLInterfaceFirewallNode::get_rule_list(const string &name, XrlAtomList &rules)
{
  string line;
  FirewallNameIter iter = _forward_firewall.begin();
  while (iter != _forward_firewall.end()) {
    if (iter->second._name == name) {
      line = iter->second._name + ";forward;;";
      break;
    }
    ++iter;
  }

  iter = _input_firewall.begin();
  while (iter != _input_firewall.end()) {
    if (iter->second._name == name) {
      line = iter->second._name + ";;input;";
      break;
    }
    ++iter;
  }

  rules.append(XrlAtom(line));
  return XrlCmdError::OKAY();
}
