// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
// vim:set sts=4 ts=8:

// Copyright (c) 2001-2005 International Computer Science Institute
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software")
// to deal in the Software without restriction, subject to the conditions
// listed in the XORP LICENSE file. These conditions include: you must
// preserve this copyright notice, and you cannot mention the copyright
// holders in advertising related to the Software without their permission.
// The Software is provided WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED. This
// notice is a summary of the XORP LICENSE file; the license in that file is
// legally binding.

// #define DEBUG_LOGGING
// #define DEBUG_PRINT_FUNCTION_NAME

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string>
#include <set>
#include <list>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "rl_firewall/rl_firewall_module.h"

#include "librl_common/rl_str_proc.hh"

#include "libxorp/debug.h"
#include "libxorp/xlog.h"

#include "libxorp/status_codes.h"
#include "libxorp/eventloop_factory.hh"
#include "libxipc/xrl_std_router.hh"
#include "libxorp/tlv.hh"

#include "xrl/interfaces/rl_interface_firewall_xif.hh"
#include "xrl/interfaces/rl_firewall_xif.hh"

using namespace std;

/**
 * 
 */
class GetFirewallList {
public:
    GetFirewallList(XrlStdRouter& xrl_router, const string &name, const string &interface, vector<string> &stat_coll)
	: _xrl_router(xrl_router),
	  _done(false), _fail(false), _name(name), _interface(interface), _stat_coll(stat_coll)
    {
    }

    void start() {
	//	printf("GetFirewallList::start():\n");
	XrlRlFirewallV0p1Client rl_firewall(&_xrl_router);
	rl_firewall.send_get_firewall_rule_list("rl_firewall",
						_name, 
						callback(this, &GetFirewallList::response));
    }

    bool busy() {
	return !_done;
    }

    bool fail() {
	return _fail;
    }

    list<string>& get() {
	return _nlist;
    }

private:
    void response(const XrlError& error, const XrlAtomList *atomlist) {
	//	printf("GetFirewallList::response\n");
	_done = true;
	if (XrlError::OKAY() != error) {
	    XLOG_WARNING("Attempt to get firewall list failed");
	    _fail = true;
	    return;
	}
	const size_t size = atomlist->size();
	for (size_t i = 0; i < size; i++) {
	    string foo = atomlist->get(i).str().c_str();
	    string token("%3B");
	    //need to process the stat_coll
	    //	    printf("'%s'\n", _stat_coll[i].c_str());

	    printf("%s%s%s%s%s\n", foo.c_str(), _interface.c_str(), token.c_str(), _stat_coll[i].c_str(), token.c_str());
	    _nlist.push_back(atomlist->get(i).str());
	}

	//	printf("GetFirewallList::response.... Leaving....\n");
    }

private:
    XrlStdRouter &_xrl_router;
    bool _done;
    bool _fail;
    string _name;

    list<string> _nlist;

    string _interface;
    vector<string> _stat_coll;
};


/**
 * 
 */
class GetFirewallInterfaceList {
public:
    GetFirewallInterfaceList(XrlStdRouter& xrl_router, const string &name)
	: _xrl_router(xrl_router),
	  _done(false), _fail(false), _name(name)
    {
    }

    void start() {
	//	printf("GetFirewallList::start():\n");
	XrlRlInterfaceFirewallV0p1Client rl_firewall_interface(&_xrl_router);
	rl_firewall_interface.send_get_interfaces_interface_firewall_rule_list("rl_interface_firewall",
						_name, 
						callback(this, &GetFirewallInterfaceList::response));
    }

    bool busy() {
	return !_done;
    }

    bool fail() {
	return _fail;
    }

    string get() {return _interface;}

private:
    void response(const XrlError& error, const XrlAtomList *atomlist) {
	//	printf("GetFirewallList::response\n");
	_done = true;
	if (XrlError::OKAY() != error) {
	    XLOG_WARNING("Attempt to get interface firewall list failed");
	    _fail = true;
	    return;
	}
	const size_t size = atomlist->size();
	for (size_t i = 0; i < size; i++) {
	    string foo = atomlist->get(i).str().c_str();
	    _interface = foo;
	    //	    _nlist.push_back(atomlist->get(i).str());
	}
	//	printf("GetFirewallList::response.... Leaving....\n");
    }
private:
    XrlStdRouter &_xrl_router;
    bool _done;
    bool _fail;
    string _name;


    string _interface;
};


/**
 * The base class that needs to be implemented by all print routines.
 */ 
class Output {
public:
    virtual ~Output()
    {}

    virtual bool begin() {
	//	printf("Rule\n");
	return true;
    }

    void print_first_line(const uint32_t& number) {
	printf("%d", number);
	printf("\n");
    }

    virtual bool print(const uint32_t& number) {
	print_first_line(number);
	return true;
    }
};

int
usage(const char *myname)
{
    fprintf(stderr, "usage: %s\n", myname);

    return -1;
}

int 
main(int argc, char **argv)
{
    if (argc < 2) {
	printf("Usage: print_firewall_rules name\n");
	exit(1);
    }

    bool no_resolve = false;
    if (argc == 3) {
	no_resolve = true;
    }

    XorpUnexpectedHandler x(xorp_unexpected_handler);
    //
    // Initialize and start xlog
    //
    xlog_init(argv[0], NULL);
    xlog_set_verbose(XLOG_VERBOSE_LOW);		// Least verbose messages
    // XXX: verbosity of the error messages temporary increased
    xlog_level_set_verbose(XLOG_LEVEL_ERROR, XLOG_VERBOSE_HIGH);
    xlog_add_default_output();
    xlog_start();

    Output *output = new Output();

    string name(argv[1]);

    try {
	EventLoop& eventloop = *EventLoopFactory::instance().create(eventloop_st);
	XrlStdRouter xrl_router(eventloop, "print_firewall_rules");

	debug_msg("Waiting for router");
	xrl_router.finalize();
	wait_until_xrl_router_is_ready(eventloop, xrl_router);
	debug_msg("\n");

	//first pre-process the iptables output
	vector<string> stat_coll;
	string cmd("iptables -n -L " + name + " -v");
	if (no_resolve) {
	    cmd += " -n";
	}
	FILE *f = popen(cmd.c_str(), "r");
	if (f) {
	    char buf[2049];
	    int ct = 0;
	    while(fgets(buf, 2048, f) != NULL) { 
		string line(buf), tmp;
		if (ct > 1) {
		    StrProc proc_str(line, " ");
		    vector<string> coll = proc_str.get();
		    vector<string>::iterator iter = coll.begin();
		    tmp += string("pkts%3D") + string(coll[0]) + "%3B";
		    tmp += string("bytes%3D") + string(coll[1]) + "%3B";
		    stat_coll.push_back(tmp);
		}
		++ct;
	    }
	    pclose(f);
	}
	
	//now get the interface information;
	GetFirewallInterfaceList get_firewall_interface_list(xrl_router, name);
	get_firewall_interface_list.start();
	while(get_firewall_interface_list.busy())
	    eventloop.run();


	GetFirewallList get_firewall_list(xrl_router, name, get_firewall_interface_list.get(), stat_coll);
	get_firewall_list.start();
	while(get_firewall_list.busy())
	    eventloop.run();



	if (get_firewall_list.fail()) {
	    XLOG_ERROR("Failed to get firewall list");
	    return -1;
	}

	if (!output->begin())
	    return -1;
	delete output;

    } catch (...) {
	xorp_catch_standard_exceptions();
    }

    xlog_stop();
    xlog_exit();

    return 0;
}
