/*
 * Module: rl_firewall_node.cc
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Michael Larson
 * Date: 2005
 * Description:
 *
 * **** End License ****
 *
 */

#include <string>
#include <map>
#include "rl_firewall_module.h"
#include "libxorp/xlog.h"
#include "libxipc/xrl_atom.hh"
#include "librl_common/rl_fw.hh"
#include "librl_common/rl_command.hh"
#include "rl_firewall_node.hh"

using namespace std;

/**
 *
 **/
RLFirewallNode::RLFirewallNode() :
  _log_martians(true),
  _send_redirects(false),
  _receive_redirects(false),
  _ip_src_route(false),
  _broadcast_ping(false),
  _syn_cookies(true)

{
  reset();
}

/**
 *
 **/
RLFirewallNode::~RLFirewallNode()
{
  //  reset();
}

/**
 *
 **/
XrlCmdError
RLFirewallNode::reset()
{
  // migrated to new CLI. do nothing here.
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::start_transaction(const string &filter, const uint32_t &rule_number)
{
  UNUSED(filter);
  UNUSED(rule_number);

  //don't do anything for now...
  return XrlCmdError::OKAY();
}
  
/**
 *
 **/
XrlCmdError 
RLFirewallNode::commit_transaction(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    vector<string> coll;
    XrlCmdError err = chain_iter->second.get_iptables_cmds(name, rule_number, coll);
    if (err != XrlCmdError::OKAY()) {
      XLOG_INFO("RLFirewallNode::commit_transaction(): Failure on commit transaction: %s, %d", name.c_str(), rule_number);
      //need to clean up rule entry...
      chain_iter->second.remove(name, rule_number, coll);
      return err;
    }

    vector<string>::iterator ip_iter = coll.begin();
    while (ip_iter != coll.end()) {
      XLOG_INFO("Executing this firewall statement: %s", ip_iter->c_str());

       if (rl_command::execute(ip_iter->c_str()) == false) {
	 //	chain_iter->second.rollback(); //need to figure htos one out...
	return XrlCmdError::COMMAND_FAILED();
      }
      ++ip_iter;
    }

    //clean up empty rules
    if (chain_iter->second.is_empty() == true) {
      _firewall_coll.erase(chain_iter);
    } 
  }
  /*
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  */


  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
RLFirewallNode::delete_all()
{
  FirewallIter chain_iter = _firewall_coll.begin();
  while (chain_iter != _firewall_coll.end()) {
    ///////////////need thi...    chain_iter->second.remove();
    delete_name(chain_iter->first);
    ++chain_iter;
  }

  _firewall_coll.erase(_firewall_coll.begin(), _firewall_coll.end());

  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_name(const string &name)
{
  //flush old rule chain from iptables
  string cmd = "iptables --flush " + name;
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLFirewallNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }

  cmd = "iptables --delete-chain " + name;
  if (rl_command::execute(cmd) == false) {
    //ignore error as this is dependent on link being removed from main chain.
  }

  //remove entries from map
  FirewallIter iter = _firewall_coll.find(name);
  if (iter != _firewall_coll.end()) {
    _firewall_coll.erase(iter);
  }
  return XrlCmdError::OKAY();
}


/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_rule_number(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  vector<string> coll;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.remove(name, rule_number, coll);
    vector<string>::iterator iter = coll.begin();
    while (iter != coll.end()) {
      XLOG_ERROR("RLFirewallNode::delete_rule_number() executing this command: %s", iter->c_str());
      rl_command::execute(*iter);
      ++iter;
    }    
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }

  //clean up empty rules
  if (chain_iter->second.is_empty() == true) {
    _firewall_coll.erase(chain_iter);
  } 

  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_log_martians(const string &enable)
{
  if (enable == "enable" || enable == "ENABLE") {
    _log_martians = true;
  } 
  else {
    _log_martians = false;
  }

  string cmd;
  cmd = string("echo ") + string(_log_martians ? "1" : "0") + string(" > /proc/sys/net/ipv4/conf/all/log_martians");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLFirewallNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_send_redirects(const string &enable)
{
  if (enable == "enable" || enable == "ENABLE") {
    _send_redirects = true;
  } 
  else {
    _send_redirects = false;
  }

  string cmd;
  cmd = string("echo ") + string(_send_redirects ? "1" : "0") + string(" > /proc/sys/net/ipv4/conf/all/send_redirects");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLFirewallNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_receive_redirects(const string &enable)
{
  if (enable == "enable" || enable == "ENABLE") {
    _receive_redirects = true;
  } 
  else {
    _receive_redirects = false;
  }

  string cmd;
  cmd = string("echo ") + string(_receive_redirects ? "1" : "0") + string(" > /proc/sys/net/ipv4/conf/all/accept_redirects");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLFirewallNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}


/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_ip_src_route(const string &enable)
{
  if (enable == "enable" || enable == "ENABLE") {
    _ip_src_route = true;
  } 
  else {
    _ip_src_route = false;
  }

  string cmd;
  cmd = string("echo ") + string(_ip_src_route ? "1" : "0") + string(" > /proc/sys/net/ipv4/conf/all/accept_source_route");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLFirewallNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_broadcast_ping(const string &enable)
{
  if (enable == "enable" || enable == "ENABLE") {
    _broadcast_ping = true;
  } 
  else {
    _broadcast_ping = false;
  }

  string cmd;
  cmd = string("echo ") + string(_broadcast_ping ? "1" : "0") + string(" > /proc/sys/net/ipv4/icmp_echo_ignore_broadcasts");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLFirewallNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_syn_cookies(const string &enable)
{
  if (enable == "enable" || enable == "ENABLE") {
    _syn_cookies = true;
  } 
  else {
    _syn_cookies = false;
  }

  string cmd;
  cmd = string("echo ") + string(_syn_cookies ? "1" : "0") + string(" > /proc/sys/net/ipv4/tcp_syncookies");
  if (rl_command::execute(cmd) == false) {
    XLOG_ERROR("RLFirewallNode: failed to execute command: %s", cmd.c_str());
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_protocol(const string &protocol, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_protocol(protocol, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_protocol(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_protocol("", rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_icmp_type(const string &type, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_icmp_type(type, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_icmp_type(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_icmp_type("", rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_icmp_code(const string &code, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_icmp_code(code, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_icmp_code(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_icmp_code(0, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}


/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_rule_state_established(const string &enable, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    if (enable == "enable" || enable == "ENABLE") {
      chain_iter->second.set_state_established(true, rule_number);
    } 
    else {
      chain_iter->second.set_state_established(false, rule_number);
    }
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_rule_state_new(const string &enable, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    if (enable == "enable" || enable == "ENABLE") {
      chain_iter->second.set_state_new(true, rule_number);
    } 
    else {
      chain_iter->second.set_state_new(false, rule_number);
    }
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_rule_state_related(const string &enable, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    if (enable == "enable" || enable == "ENABLE") {
      chain_iter->second.set_state_related(true, rule_number);
    } 
    else {
      chain_iter->second.set_state_related(false, rule_number);
    }
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_rule_state_invalid(const string &enable, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    if (enable == "enable" || enable == "ENABLE") {
      chain_iter->second.set_state_invalid(true, rule_number);
    } 
    else {
      chain_iter->second.set_state_invalid(false, rule_number);
    }
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_rule_action(const string &action, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    //    needs to convert to uppercase!!!!!!!!!!!!!!!!!!!!
    XLOG_ERROR("RLFirewallNode::set_rule_action--settting rule action");
    chain_iter->second.set_action(action, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_rule_log(const string &log, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    //    this needs to convert rule to uppercase inside of chain_iter!!!!!!!!!!!!!!!!!!!!!!!!!!
    chain_iter->second.set_rule_log(log, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_source_address(const IPv4 &address, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_source_address(address, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_source_address(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_source_address(IPv4(), rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_source_network(const IPv4Net &network, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_source_network(network, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_source_network(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_source_network(IPv4Net(), rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_source_range(const IPv4 &start, const IPv4 &stop, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_source_address(start, stop, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_source_range(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_source_address(IPv4(), IPv4(), rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_source_port_number(const uint32_t &port_number, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_source_port_number(port_number, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_source_port_number(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_source_port_number(0, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_source_port_name(const string &port_name, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_source_port_name(port_name, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_source_port_name(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_source_port_name("", rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_source_port_range(const uint32_t &start, const uint32_t &stop, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_source_port(start, stop, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_source_port_range(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_source_port(0, 0, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}


/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_destination_address(const IPv4 &address, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_dest_address(address, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_destination_address(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_dest_address(IPv4(), rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_destination_network(const IPv4Net &network, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_dest_network(network, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_destination_network(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_dest_network(IPv4Net(), rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_destination_range(const IPv4 &start, const IPv4 &stop, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_dest_address(start, stop, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_destination_range(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_dest_address(IPv4(), IPv4(), rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_destination_port_number(const uint32_t &port_number, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_dest_port_number(port_number, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_destination_port_number(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_dest_port_number(0, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_destination_port_name(const string &port_name, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_dest_port_name(port_name, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::delete_destination_port_name(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_dest_port_name("", rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError 
RLFirewallNode::set_destination_port_range(const uint32_t &start, const uint32_t &stop, const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_dest_port(start, stop, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}
 
XrlCmdError 
RLFirewallNode::delete_destination_port_range(const string &name, const uint32_t &rule_number)
{
  FirewallIter chain_iter;
  if (get_filter(name, chain_iter)) {
    chain_iter->second.set_dest_port(0, 0, rule_number);
  }
  else {
    XLOG_ERROR("RLFirewallNode: failed to find rule");
    return XrlCmdError::COMMAND_FAILED();
  }
  return XrlCmdError::OKAY();
}

/**
 *
 **/
/*
bool
RLFirewallNode::get_rule(const string &filter, const uint32_t &rule_number, RLFirewallNode::FirewallRuleIter &rule_iter)
{
  FirewallIter iter = _firewall_coll.find(filter);
  if (iter != _firewall_coll.end()) {
    iter->get(rule_number);
    
    rule_iter = iter->second.find(rule_number);
    if (rule_iter == iter->second.end()) {
      pair<FirewallRuleIter, bool> rtn_value =  iter->second.insert(pair<uint32_t, FirewallRule> (rule_number, FirewallRule()));
      rule_iter = rtn_value.first;
    }
  }
  else {
    //this is a new rule, and ignore error
    string cmd = "iptables --new " + filter;
    rl_command::execute(cmd);

    FirewallRuleColl rule_coll;
    pair<FirewallIter, bool> rtn_firewall_rule_coll =_firewall_coll.insert(pair<string, FirewallRuleColl>(filter, rule_coll));

    FirewallRule new_rule;
    pair<FirewallRuleIter, bool> rtn_value = rtn_firewall_rule_coll.first->second.insert(pair<uint32_t, FirewallRule>(rule_number, new_rule));
    rule_iter = rtn_value.first;

    //now add the implicit drop all at the end of the rule set at rule number 1024
    new_rule._mod_rule._action = string("DROP");
    new_rule.commit_mods();
    rtn_firewall_rule_coll.first->second.insert(pair<uint32_t, FirewallRule>(1024, new_rule));    
    cmd = "iptables --insert " + filter + " 1 --jump DROP";
    if (rl_command::execute(cmd) == false) {
      return false;
    }
  }
  return true;
}
*/

/**
 *
 **/
XrlCmdError
RLFirewallNode::retrieve_rules(const string &name, XrlAtomList &rules)
{
  UNUSED(rules);


  if (name.empty() == false) {
    FirewallIter iter = _firewall_coll.find(name);
    if (iter == _firewall_coll.end()) {
      return XrlCmdError::OKAY();
    }
    return retrieve_rule_set(name, iter->second.expose(), rules);
  }
  else {
    FirewallIter iter = _firewall_coll.begin();
    if (iter != _firewall_coll.end()) {
      retrieve_rule_set(iter->first, iter->second.expose(), rules);
      ++iter;
    }
  }
  return XrlCmdError::OKAY();
}

XrlCmdError
RLFirewallNode::retrieve_rule_set(const string &name, FWChain::FirewallColl coll, XrlAtomList &rules)
{
  UNUSED(name);

  FWChain::FirewallIter c_iter = coll.begin();
  while (c_iter != coll.end()) {
    char buf[80];

    int rule_number = c_iter->first;
    FWData data = c_iter->second;

    sprintf(buf, "%d", rule_number);
    string line = string("rule_number=") + string(buf) + ";";
    line += string("protocol=") + data._protocol + ";";
    line += string("icmp_type=") + data._icmp_type + ";";
    line += string("icmp_code=") + data._icmp_code + ";";
    
    string tmp;
    if (data._established_state == true) {
      tmp += "established,";
    }
    if (data._new_state == true) {
      tmp += "new,";
    }
    if (data._related_state == true) {
      tmp += "related,";
    }
    if (data._invalid_state == true) {
      tmp += "invalid,";
    }
    if (tmp.empty() == false) {
      //truncate the last ','
      tmp = tmp.substr(0, tmp.length()-1) + " ";
    }
    line += string("state=") + tmp + ";";

    line += string("action=") + data._action + ";";

    line += string("log=") + data._log + ";";

    sprintf(buf, "%d", data._rate_limit);
    line += string("rate_lim=") + string(buf) + ";";

    line += string("rate_int=") + data._rate_interval + ";";

    sprintf(buf, "%d", data._rate_limit_burst);
    line += string("rate_lim_brst") + string(buf) + ";";

    line += string("src_addr=") + data._source_address.str() + ";";

    line += string("src_net=") + data._source_network.str() + ";";

    line += string("src_addr_start=") + data._source_address_start.str() +  ";";
    line += string("src_addr_stop=") + data._source_address_stop.str() +  ";";

    sprintf(buf, "%d", data._source_port_number);
    line +=  string("src_port_num=") + string(buf) + ";";
    line += string("src_port_name=") + data._source_port_name + ";";
    sprintf(buf, "%d", data._source_port_start);
    line += string("src_port_start=") + string(buf) + ";";

    sprintf(buf, "%d", data._source_port_stop);
    line += string("src_port_stop=") + string(buf) + ";";

    line += string("dst_addr=") + data._dest_address.str() + ";";

    line += string("dst_net=") + data._dest_network.str() + ";";

    line += string("dst_addr_start=") + data._dest_address_start.str() +  ";";
    line += string("dst_addr_stop=") + data._dest_address_stop.str() +  ";";

    sprintf(buf, "%d", data._dest_port_number);
    line += string("dst_port_num=") + string(buf) + ";";
    line += string("dst_port_name=") + data._dest_port_name + ";";

    sprintf(buf, "%d", data._dest_port_start);
    line += string("dst_port_start=") + string(buf) + ";";
    sprintf(buf, "%d", data._dest_port_stop);
    line += string("dst_port_stop=") + string(buf) + ";";

    //now push it on the list
    rules.append(XrlAtom(line));
    ++c_iter;
  }

  return XrlCmdError::OKAY();
}

/**
 *
 **/
bool
RLFirewallNode::get_filter(const string &name, FirewallIter &iter)
{
  iter = _firewall_coll.find(name);
  if (iter == _firewall_coll.end()) {
    //create entry and reset new iter
    string cmd = "iptables --new " + name;
    rl_command::execute(cmd);

    pair<FirewallIter, bool> val = _firewall_coll.insert(pair<string, FWChain>(name, FWChain()));
    iter = val.first;

  }
  return true;
}

