/**
 *    Module: command_proc_show_serial.cc
 *
 *    Author: Michael Larson
 *    Date: 2005
 *    Description:
 *
 *    This program is free software; you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published 
 *    by the Free Software Foundation; either version 2 of the License, 
 *    or (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be  useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 *    02110-1301 USA
 *
 *    Copyright 2006, Vyatta, Inc.
 */
#include <iostream>
#include <list>
#include <map>
#include <string>
#include <time.h>
#include <stdlib.h>

#include "config.h"
#include "rl_str_proc.hh"
#include <command_proc_show_serial.hh>

using namespace std;

/**
 *
 **/
CommandProcShowSerial::CommandProcShowSerial() : CommandProcBase()
{

}

/**
 *
 **/
CommandProcShowSerial::~CommandProcShowSerial()
{


}

/**
 *
./rl_cmd_proc showserialinterfaces wan0 ppp show_serial_ppp_serial.xsl

so, the command will be structured as such:

./rl_cmd_proc showserialinterfaces [serial] [ppp|hdlc|frame|none] [xsl]
[vif]


Note that the last value [vif] is optional and only makes sense when 
used in combination with frame.

 **/
std::string
CommandProcShowSerial::process(const string &cmd, bool debug, string &reason)
{
  UNUSED(reason);

  //first thing, execute this command;
  if (cmd.empty() == false) {
    char buf[2048];
    string wan_cmd;
    string iface;
    string cl_arg_vif, query_vif;

    vector<string> in_coll;
    StrProc proc_str(cmd, " ");
    in_coll = proc_str.get();
    if (in_coll.size() < 4) {
      return _xml_out;
    }
    iface = in_coll[1];
    string mode = in_coll[2];

    if (in_coll.size() == 5) {
      _xsl = XSLDIR "/" + in_coll[4];
      cl_arg_vif = in_coll[3];
    }
    else {
      _xsl = XSLDIR "/" + in_coll[3];
    }

    _xml_out = "<opcommand name='serial'><format type='row'><row name = 'proto'>";

    //let's obtain the serial vifs via the component
    //the first value in the coll is the serial interface
    vector<string> vif_coll;
    string cmd = PKGLIBEXECDIR "/print_serial_data " + iface;
    FILE *f = popen(cmd.c_str(), "r");
    if (f) {
      while(fgets(buf, 2047, f) != NULL) { 
	string line(buf);
	convert_to_xml_vif(line, debug, query_vif);
      } 
      if (pclose(f) != 0) {
	return string("");
      }
    }

    string vif, vif_iface;
    if (cl_arg_vif.empty() == false) {
      vif_iface = iface + "." + cl_arg_vif;
      vif = cl_arg_vif;
    }
    else if (query_vif.empty() == false) {
      vif_iface = iface + "." + query_vif;
      vif = query_vif;
    }
    else {
      vif_iface = iface + ".1";
      vif = "1";
    }


    wan_cmd = "cat /proc/net/wanrouter/interfaces";
    f = popen(wan_cmd.c_str(), "r");
    if (f) {
      while(fgets(buf, 2047, f) != NULL) { 
	string line(buf);
	convert_to_xml_proc_state(line, debug, vif_iface);
      } 
      if (pclose(f) != 0) {
	return string("");
      }
    }

#ifdef RL_NOT_TARGET
    wan_cmd = "cat wan_xm";
#else
    wan_cmd = "wanpipemon -c xm -i " + iface;
#endif
    f = popen(wan_cmd.c_str(), "r");
    if (f) {
      while(fgets(buf, 2047, f) != NULL) { 
	string line(buf);
	convert_to_xml_xm(line, debug);
      } 
      if (pclose(f) != 0) {
	return string("");
      }
    }

#ifdef RL_NOT_TARGET
    wan_cmd = "cat wan_xru";
#else 
    wan_cmd = "wanpipemon -c xru -i " + iface;
#endif
    f = popen(wan_cmd.c_str(), "r");
    if (f) {
      while(fgets(buf, 2047, f) != NULL) { 
	string line(buf);
	convert_to_xml_xru(line, debug);
      } 
      if (pclose(f) != 0) {
	return string("");
      }
    }

#ifdef RL_NOT_TARGET
    wan_cmd = "cat wan_Tread";
#else
    wan_cmd = "wanpipemon -c Tread -i " + vif_iface;
#endif
    f = popen(wan_cmd.c_str(), "r");
    if (f) {
      while(fgets(buf, 2047, f) != NULL) { 
	string line(buf);
	convert_to_xml_Tread(line, debug);
      } 
      if (pclose(f) != 0) {
	return string("");
      }
    }

#ifdef RL_NOT_TARGET
    wan_cmd = "cat wan_Ta";
#else
    wan_cmd = "wanpipemon -c Ta -i " + iface;
#endif    
    f = popen(wan_cmd.c_str(), "r");
    if (f) {
      while(fgets(buf, 2047, f) != NULL) { 
	string line(buf);
	convert_to_xml_Ta(line, debug);
      } 
      if (pclose(f) != 0) {
	return string("");
      }
    }

      //wanpipemon -i wan0 -c so
#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_so";
#else
      wan_cmd = "wanpipemon -c so -i " + iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_so(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }
    /*
     * PPP
     */
    if (mode == "ppp") {
#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_sicp";
#else
      wan_cmd = "wanpipemon -c sicp -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_sicp(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }

#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_xu";
#else
      wan_cmd = "wanpipemon -c xu -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_xu(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }

#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_ca";
#else
      wan_cmd = "wanpipemon -c ca -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_ca(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }

#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_xn";
#else
      wan_cmd = "wanpipemon -c xn -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_xn(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }

#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_ci";
#else
      wan_cmd = "wanpipemon -c ci -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_ci(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }

#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_cg";
#else
      wan_cmd = "wanpipemon -c cg -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_cg(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }

#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_slcp";
#else
      wan_cmd = "wanpipemon -c slcp -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_slcp(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }

#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_sp";
#else
      wan_cmd = "wanpipemon -c sp -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_sp(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }
    }
    /*
     * HDLC
     */
    else if (mode == "hdlc") {
      
      //wanpipemon -i wan0.1 -c crc
#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_crc";
#else
      wan_cmd = "wanpipemon -c crc -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_crc(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }
      //wanpipemon -i wan0 -c sc
#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_sc";
#else
      wan_cmd = "wanpipemon -c sc -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_sc(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }
      //wanpipemon -i wan0 -c ss
#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_ss";
#else
      wan_cmd = "wanpipemon -c ss -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_ss(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }
      //wanpipemon -i wan0 -c xl
#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_xl";
#else
      wan_cmd = "wanpipemon -c xl -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_xl(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }
      //wanpipemon -i wan0.1 -c xcv
#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_xcv";
#else
      wan_cmd = "wanpipemon -c xcv -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_xcv(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }
#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_sp";
#else
      wan_cmd = "wanpipemon -c sp -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_sp(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }
    }
    /*
     * FRAME
     */
    else if (mode == "frame") {
      //wanpipemon -i wan0.16 -c se
#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_se";
#else
      wan_cmd = "wanpipemon -c se -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_se(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }

      //wanpipemon -i wan0 -c sc
#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_sc";
#else
      wan_cmd = "wanpipemon -c sc -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_sc(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }

      //wanpipemon -i wan0.16 -c cl
#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_cl";
#else
      wan_cmd = "wanpipemon -c cl -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_cl(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }

#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_sg";
#else
      wan_cmd = "wanpipemon -c sg -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_sg(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }
    }      
    /*
     * Physical fall through
     */
    else {
      //physical or any other non-protocol case
#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_sp";
#else
      wan_cmd = "wanpipemon -c sp -i " + vif_iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_sp(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }
#ifdef RL_NOT_TARGET
      wan_cmd = "cat wan_sc";
#else
      wan_cmd = "wanpipemon -c sc -i " + iface;
#endif    
      f = popen(wan_cmd.c_str(), "r");
      if (f) {
	while(fgets(buf, 2047, f) != NULL) { 
	  string line(buf);
	  convert_to_xml_sc(line, debug);
	} 
	if (pclose(f) != 0) {
	  return string("");
	}
      }
    }
    _xml_out += "</row>";
      
    //now any additional rows go here
    if (mode == "frame") {
      process_dlcis(iface, vif_iface, cl_arg_vif, debug);
    }
    
    _xml_out += "</format></opcommand>";
    
  }
  return _xml_out;
  }

void
CommandProcShowSerial::convert_to_xml_vif(const string &line, bool debug, string &extracted_vif)
{
  if (debug) {
    cout << "processing: convert_to_xml_vif" << endl;
  }
  StrProc proc_str(line, "%3B");
  vector<string> coll = proc_str.get();

  vector<string>::iterator iter = coll.begin();
  while (iter != coll.end()) {
    uint32_t pos = iter->find("%3D");
    if (pos != string::npos) {
      _xml_out += "<" + iter->substr(0, pos) + ">" + iter->substr(pos+3, iter->length() - pos-3) + "</" + iter->substr(0, pos) + ">";
      if (iter->substr(0,pos) == "vif" && extracted_vif.empty() == true) {
	extracted_vif = iter->substr(pos+3, iter->length() - pos-3);
      }
    }
    ++iter;
  }
  return;
}


/**
 *
Interface name | Device name | Media | Operational State |
wan0           | wanpipe1    | 0     | Connected         |
wan0.1         | wan0        | 0     | Disconnected      |

 **/
void
CommandProcShowSerial::convert_to_xml_proc_state(const std::string &line, bool debug, std::string &iface_vif)
{
  if (debug) {
    cout << "processing: convert_to_xml_proc_state" << endl;
  }
  if (line.find(iface_vif) != string::npos) {
    StrProc proc_str(line, " ");
    _xml_out += "<iface_vif_status>" + proc_str.get(6) + "</iface_vif_status>";
  }
  return;
}

/**
 *
 --------------------------------
 wan0: MODEM STATUS
 --------------------------------

 DCD: High
 CTS: High
 **/
void 
CommandProcShowSerial::convert_to_xml_xm(const string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_xm" << endl;
  }
  list<string> segments;
  string remainder = line;

  StrProc proc_str(line, " ");

  if (line.find("DCD: ") != string::npos) {
      _xml_out += "<status_DCD>" + proc_str.get(1) + "</status_DCD>";
  }

  if (line.find("CTS: ") != string::npos) {
      _xml_out += "<status_CTS>" + proc_str.get(1) + "</status_CTS>";
  }

  return;
}

/**
 *
 ----------------------------------
 wan0: ROUTER UP TIME
 ----------------------------------
 
 Router UP Time:  9 minute(s),  22 seconds
 
 **/
void 
CommandProcShowSerial::convert_to_xml_xru(const string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_xru" << endl;
  }
  StrProc proc_str(line, " ");
  
  if (line.find("Router UP Time:") != string::npos) {
    _xml_out += "<uptime>" + proc_str.get(3)  + " " + proc_str.get(4) + " " + proc_str.get(5) + " " + proc_str.get(6) + "</uptime>";
  }
  return;
}

/**
 *
 ------------------------------------------------------
 wan0: AFT COMMUNICATION ERROR STATISTICS
 ------------------------------------------------------
 
 RX Stats:
 Number of receiver overrun errors:  0
 Number of receiver CRC errors:  0
 Number of receiver Abort errors:  0
 Number of receiver corruption errors:  0
 Number of receiver PCI errors:  0
 Number of receiver DMA descriptor errors:  0
 TX Stats:
 Number of transmitter PCI errors:  0
 Number of transmitter PCI latency warnings:  0
 Number of transmitter DMA descriptor errors:  0
 Number of transmitter DMA descriptor length errors:  0

 **/
void 
CommandProcShowSerial::convert_to_xml_sc(const string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_sc" << endl;
  }
  StrProc proc_str(line, " ");
  

  if (line.find("receiver overrun") != string::npos) {
    _xml_out += "<rx_overrun>" + proc_str.get(5) + "</rx_overrun>";
  }

  if (line.find("receiver CRC") != string::npos) {
      _xml_out += "<rx_CRC>" + proc_str.get(5) + "</rx_CRC>";
  }

  if (line.find("receiver Abort") != string::npos) {
      _xml_out += "<rx_abort>" + proc_str.get(5) + "</rx_abort>";
  }

  if (line.find("receiver corruption") != string::npos) {
      _xml_out += "<rx_corrupt>" + proc_str.get(5) + "</rx_corrupt>";
  }

  if (line.find("receiver PCI") != string::npos) {
      _xml_out += "<rx_PCI>" + proc_str.get(5) + "</rx_PCI>";
  }

  if (line.find("receiver DMA") != string::npos) {
      _xml_out += "<rx_DMA>" + proc_str.get(6) + "</rx_DMA>";
  }

  if (line.find("transmitter PCI errors") != string::npos) {
      _xml_out += "<tx_PCI_errors>" + proc_str.get(5) + "</tx_PCI_errors>";
  }

  if (line.find("transmitter PCI latency") != string::npos) {
      _xml_out += "<tx_PCI_latency>" + proc_str.get(6) + "</tx_PCI_latency>";
  }

  if (line.find("transmitter DMA descriptor errors") != string::npos) {
      _xml_out += "<tx_DMA_descript_err>" + proc_str.get(6) + "</tx_DMA_descript_err>";
  }

  if (line.find("transmitter DMA descriptor length") != string::npos) {
      _xml_out += "<tx_DMA_descript_len_err>" + proc_str.get(7) + "</tx_DMA_descript_len_err>";
  }
  return;
}

/**
 *
Adapter type 1
CSU/DSU wan0 Conriguration:
        Media type      T1
        Framing         ESF
        Encoding        B8ZS
        Line Build      0db
        Channel Base    1-24
        Clock Mode      Normal

 **/
void 
CommandProcShowSerial::convert_to_xml_Tread(const string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_Tread" << endl;
  }
  StrProc proc_str(line, " ");

  if (line.find("Media type") != string::npos) {
      _xml_out += "<csudsu_type>" + proc_str.get(2) + "</csudsu_type>";
  }
  if (line.find("Framing") != string::npos) {
      _xml_out += "<csudsu_framing>" + proc_str.get(1) + "</csudsu_framing>";
  }
  if (line.find("Encoding") != string::npos) {
      _xml_out += "<csudsu_encoding>" + proc_str.get(1) + "</csudsu_encoding>";
  }
  if (line.find("Line Build") != string::npos) {
      _xml_out += "<csudsu_linebuild>" + proc_str.get(2) + "</csudsu_linebuild>";
  }
  if (line.find("Channel Base") != string::npos) {
      _xml_out += "<csudsu_channel>" + proc_str.get(2) + "</csudsu_channel>";
  }
  if (line.find("Clock Mode") != string::npos) {
      _xml_out += "<csudsu_clock>" + proc_str.get(2) + "</csudsu_clock>";
  }
  return;
}

/**
 *
 ***** wan0: T1 Alarms (Framer) *****

ALOS:   OFF     | LOS:  OFF
RED:    OFF     | AIS:  OFF
YEL:    OFF     | OOF:  OFF

***** wan0: T1 Performance Monitoring Counters *****

Framing Bit Error:      2       Line Code Violation:    4186
Out of Frame Errors:    1       Bit Errors:             4

OR IF MODE IS Frame:

~ # wanpipemon -i wan0.21 -c Ta

***** wan0: T1 Alarms (Framer) *****

ALOS:   ON      | LOS:  ON
RED:    ON      | AIS:  OFF
YEL:    OFF     | OOF:  ON


***** wan0: T1 Performance Monitoring Counters *****

Line Code Violation     : 1
Bit Errors (CRC6/Ft/Fs) : 0
Out of Frame Errors     : 0
Framing Bit Errors      : 0



 **/
void 
CommandProcShowSerial::convert_to_xml_Ta(const string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_Ta" << endl;
  }
  StrProc proc_str(line, " ");

  if (line.find("ALOS") != string::npos) {
      _xml_out += "<alarm_alos>" + proc_str.get(1) + "</alarm_alos>";
      _xml_out += "<alarm_los>" + proc_str.get(4) + "</alarm_los>";
  }

  if (line.find("RED") != string::npos) {
      _xml_out += "<alarm_red>" + proc_str.get(1) + "</alarm_red>";
      _xml_out += "<alarm_ais>" + proc_str.get(4) + "</alarm_ais>";
  }

  if (line.find("YEL") != string::npos) {
      _xml_out += "<alarm_yel>" + proc_str.get(1) + "</alarm_yel>";
      _xml_out += "<alarm_oof>" + proc_str.get(4) + "</alarm_oof>";
  }

  if (line.find("Framing Bit Errors") != string::npos) {
    _xml_out += "<t1_perf_frame_bit_err>" + proc_str.get(4) + "</t1_perf_frame_bit_err>";
  }
  if (line.find("Line Code Violation") != string::npos) {
    _xml_out += "<t1_perf_line_code_vio>" + proc_str.get(4) + "</t1_perf_line_code_vio>";    
  }
  if (line.find("Out of Frame") != string::npos) {
    _xml_out += "<t1_perf_out_frame_err>" + proc_str.get(5) + "</t1_perf_out_frame_err>";
  }
  if (line.find("Framing Bit Errors") != string::npos) {
    _xml_out += "<t1_perf_bit_err>" + proc_str.get(4) + "</t1_perf_bit_err>";
  }
  return;
}

/**
 *

       ---------------------------------------------
                wan0.20: GLOBAL STATISTICS: CPE
        ---------------------------------------------

                       Full Status Enquiry messages sent: 0
Link Integrity Verification Status Enquiry messages sent: 669
                           Full Status messages received: 0
    Link Integrity Verification Status messages received: 0
                                     CPE initializations: 0
                            Current Send Sequence Number: 1
                         Current Receive Sequence Number: 0
                                      Current N392 count: 0
                                      Current N393 count: 0

 **/
void
CommandProcShowSerial::convert_to_xml_sg(const string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_sg" << endl;
  }
  StrProc proc_str(line, " ");
  
  if (line.find("Full Status Enquiry") != string::npos) {
      _xml_out += "<full_stat_enq>" + proc_str.get(5) + "</full_stat_enq>";
  }

  if (line.find("Verification Status Enquiry") != string::npos) {
      _xml_out += "<ver_stat_enq>" + proc_str.get(7) + "</ver_stat_enq>";
  }

  if (line.find("Full Status messages") != string::npos) {
      _xml_out += "<full_stat_msg>" + proc_str.get(4) + "</full_stat_msg>";
  }

  if (line.find("Verification Status messages") != string::npos) {
      _xml_out += "<ver_stat_msg>" + proc_str.get(6) + "</ver_stat_msg>";
  }

  if (line.find("CPE initializations") != string::npos) {
      _xml_out += "<cpe_init>" + proc_str.get(2) + "</cpe_init>";
  }

  if (line.find("Send Sequence Number") != string::npos) {
      _xml_out += "<cur_send_seq_num>" + proc_str.get(4) + "</cur_send_seq_num>";
  }

  if (line.find("Receive Sequence Number") != string::npos) {
      _xml_out += "<cur_rec_seq_num>" + proc_str.get(4) + "</cur_rec_seq_num>";
  }

  if (line.find("N392 count") != string::npos) {
       _xml_out += "<n392_ct>" + proc_str.get(3) + "</n392_ct>";
   }

  if (line.find("N393 count") != string::npos) {
      _xml_out += "<n393_ct>" + proc_str.get(3) + "</n393_ct>";
  }
  return;
}

/**
 *
        ---------------------------------------
                wan0.1: PACKET STATISTICS
        ---------------------------------------

Number discards( bad header ): 0
Number discards( unknown/unsupported protocol ): 0
Number discards(unknown/unsupported protocol+too large for Protocol-Reject): 0

                        Received        Transmitted
Number of LCP packets: 0                0
Number of IPCP packets: 0               0
Number of IPXCP packets: 0              0
Number of PAP packets: 0                0
Number of CHAP packets: 0               0
Number of LQR packets: 0                0
Number of IP packets:  0                0
Number of IPX packets: 0                0



 **/
void
CommandProcShowSerial::convert_to_xml_sp(const string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_sp" << endl;
  }
  StrProc proc_str(line, " ");
 
  if (line.find("bad header") != string::npos) {
      _xml_out += "<num_bad_hdr>" + proc_str.get(5) + "</num_bad_hdr>";
  }

  if (line.find("unknown/unsupported") != string::npos) {
      _xml_out += "<num_un_proto>" + proc_str.get(3) + "</num_un_proto>";
  }

  if (line.find("too large for Protocol") != string::npos) {
      _xml_out += "<num_un_too_lrg>" + proc_str.get(7) + "</num_un_too_lrg>";
  }

  if (line.find("LCP") != string::npos) {
      _xml_out += "<rx_LCP>" + proc_str.get(4) + "</rx_LCP>";
      _xml_out += "<tx_LCP>" + proc_str.get(5) + "</tx_LCP>";
  }

  if (line.find("IPCP") != string::npos) {
      _xml_out += "<rx_IPCP>" + proc_str.get(4) + "</rx_IPCP>";
      _xml_out += "<tx_IPCP>" + proc_str.get(5) + "</tx_IPCP>";
  }

  if (line.find("IPXCP") != string::npos) {
      _xml_out += "<rx_IPXCP>" + proc_str.get(4) + "</rx_IPXCP>";
      _xml_out += "<tx_IPXCP>" + proc_str.get(5) + "</tx_IPXCP>";
  }

  if (line.find("PAP") != string::npos) {
      _xml_out += "<rx_PAP>" + proc_str.get(4) + "</rx_PAP>";
      _xml_out += "<tx_PAP>" + proc_str.get(5) + "</tx_PAP>";
  }

  if (line.find("CHAP") != string::npos) {
      _xml_out += "<rx_CHAP>" + proc_str.get(4) + "</rx_CHAP>";
      _xml_out += "<tx_CHAP>" + proc_str.get(5) + "</tx_CHAP>";
  }

  if (line.find("LQR") != string::npos) {
      _xml_out += "<rx_LQR>" + proc_str.get(4) + "</rx_LQR>";
      _xml_out += "<tx_LQR>" + proc_str.get(5) + "</tx_LQR>";
  }

  if (line.find("IP ") != string::npos) {
      _xml_out += "<rx_IP>" + proc_str.get(4) + "</rx_IP>";
      _xml_out += "<tx_IP>" + proc_str.get(5) + "</tx_IP>";
  }

  if (line.find("IPX") != string::npos) {
      _xml_out += "<rx_IPX>" + proc_str.get(4) + "</rx_IPX>";
      _xml_out += "<tx_IPX>" + proc_str.get(5) + "</tx_IPX>";
  }
}

/**
 *


        -------------------------------------
                wan0.1: IPCP STATISTICS
        -------------------------------------

Packets discarded (unknown IPCP code): 0
Received IPCP packets too large: 0
Received packets invalid or out-of-sequence Configure-Acks: 0
Received packets invalid Configure-Naks or Configure-Rejects: 0
Configure-Naks or Configure-Rejects with bad Identifier: 0

                        Received        Transmitted
Number of Config-Request pkts: 0        0
Number of Config-Ack packets : 0        0
Number of Config-Nack packets: 0        0
Number of Config-Reject packets: 0      0
Number of Term-Reqst packets : 0        0
Number of Terminate-Ack packets: 0      0
Number of Code-Reject packets: 0        0
 **/
void
CommandProcShowSerial::convert_to_xml_sicp(const string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_sicp" << endl;
  }
  StrProc proc_str(line, " ");
 
  if (line.find("unknown IPCP code") != string::npos) {
      _xml_out += "<num_ipcp_code>" + proc_str.get(5) + "</num_ipcp_code>";
  }

  if (line.find("IPCP packets too large") != string::npos) {
      _xml_out += "<rx_ipcp_pckts_too_lr>" + proc_str.get(5) + "</rx_ipcp_pckts_too_lr>";
  }

  if (line.find("out-of-sequence Configure") != string::npos) {
      _xml_out += "<rx_inv_acks>" + proc_str.get(6) + "</rx_inv_acks>";
  }
  if (line.find("invalid Configure-Naks or") != string::npos) {
      _xml_out += "<rx_cfg_naks>" + proc_str.get(6) + "</rx_cfg_naks>";
  }

  if (line.find("bad Identifier") != string::npos) {
      _xml_out += "<cfg_naks_bad_ident>" + proc_str.get(6) + "</cfg_naks_bad_ident>";
  }

  if (line.find("Config-Requests pkts") != string::npos) {
      _xml_out += "<rx_cfg_req_pkts>" + proc_str.get(4) + "</rx_cfg_req_pkts>";
      _xml_out += "<tx_cfg_req_pkts>" + proc_str.get(5) + "</tx_cfg_req_pkts>";
  }

  if (line.find("Config-Ack packets") != string::npos) {
      _xml_out += "<rx_cfg_ack_pkts>" + proc_str.get(4) + "</rx_cfg_ack_pkts>";
      _xml_out += "<tx_cfg_ack_pkts>" + proc_str.get(5) + "</tx_cfg_ack_pkts>";
  }

  if (line.find("Config-Nack packets") != string::npos) {
      _xml_out += "<rx_cfg_nack_pkts>" + proc_str.get(4) + "</rx_cfg_nack_pkts>";
      _xml_out += "<tx_cfg_nack_pkts>" + proc_str.get(5) + "</tx_cfg_nack_pkts>";
  }

  if (line.find("Config-Reject packets") != string::npos) {
      _xml_out += "<rx_cfg_rej_pkts>" + proc_str.get(4) + "</rx_cfg_rej_pkts>";
      _xml_out += "<tx_cfg_rej_pkts>" + proc_str.get(5) + "</tx_cfg_rej_pkts>";
  }

  if (line.find("Term-Reqst packets") != string::npos) {
      _xml_out += "<rx_term_reqst_pkts>" + proc_str.get(4) + "</rx_term_reqst_pkts>";
      _xml_out += "<tx_term_reqst_pkts>" + proc_str.get(5) + "</tx_term_reqst_pkts>";
  }

  if (line.find("Termiante-Ack packets") != string::npos) {
      _xml_out += "<rx_term_ack_pkts>" + proc_str.get(4) + "</rx_term_ack_pkts>";
      _xml_out += "<tx_term_ack_pkts>" + proc_str.get(5) + "</tx_term_ack_pkts>";
  }

  if (line.find("Code-Reject packets") != string::npos) {
      _xml_out += "<rx_code_rej_pkts>" + proc_str.get(4) + "</rx_code_rej_pkts>";
      _xml_out += "<tx_code_rej_pkts>" + proc_str.get(5) + "</tx_code_rej_pkts>";
  }
  return;
}

/**
 *
        --------------------------------
                wan0.1: PPP TIMERS
        --------------------------------

Restart timer: 0
Authentication restart timer: 0
Authentication wait timer: 0
DCD/CTS failure detection timer: 0
Drop DTR duration timer: 0
Connection attempt timeout: 0
Max-Configure counter: 0
Max-Terminate counter: 0
Max-Failure counter: 0
Max-Authenticate counter: 0

 **/
void 
CommandProcShowSerial::convert_to_xml_xu(const std::string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_xu" << endl;
  }
  StrProc proc_str(line, " ");
 
  if (line.find("Restart timer") != string::npos) {
      _xml_out += "<timer_restart>" + proc_str.get(2) + "</timer_restart>";
  }

  if (line.find("timer_auth_restart") != string::npos) {
      _xml_out += "<timer_auth_restart>" + proc_str.get(3) + "</timer_auth_restart>";
  }

  if (line.find("DCD/CTS") != string::npos) {
      _xml_out += "<timer_dcd_cts>" + proc_str.get(4) + "</timer_dcd_cts>";
  }

  if (line.find("Drop DTR") != string::npos) {
      _xml_out += "<timer_drop_dtr>" + proc_str.get(4) + "</timer_drop_dtr>";
  }

  if (line.find("connection attempt") != string::npos) {
      _xml_out += "<timer_conn_attemp>" + proc_str.get(3) + "</timer_conn_attemp>";
  }

  if (line.find("Max-Configure") != string::npos) {
      _xml_out += "<timer_max_conf>" + proc_str.get(2) + "</timer_max_conf>";
  }

  if (line.find("Max-Terminate") != string::npos) {
      _xml_out += "<timer_max_term>" + proc_str.get(2) + "</timer_max_term>";
  }

  if (line.find("Max-Failure") != string::npos) {
      _xml_out += "<timer_max_fail>" + proc_str.get(2) + "</timer_max_fail>";
  }

  if (line.find("Max-Authenticate") != string::npos) {
      _xml_out += "<timer_max_auth>" + proc_str.get(2) + "</timer_max_auth>";
  }
  return;
}

/**
 *
        ----------------------------------------
                wan0.1: PPP AUTHENTICATION
        ----------------------------------------

Allow the use of PAP for inbound/outbound: No
Allow the use of CHAP for inbound/outbound: No

 **/
void 
CommandProcShowSerial::convert_to_xml_ca(const std::string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_ca" << endl;
  }
  StrProc proc_str(line, " ");
 
  if (line.find("PAP") != string::npos) {
      _xml_out += "<ppp_auth_pap>" + proc_str.get(7) + "</ppp_auth_pap>";
  }

  if (line.find("CHAP") != string::npos) {
      _xml_out += "<ppp_auth_chap>" + proc_str.get(7) + "</ppp_auth_chap>";
  }

}

/**
 *
        --------------------------------------
                wan0.1: PPP NEGOTIATIONS
        --------------------------------------

Remote Maximum Receive Unit: 103
Negotiated IP options: 00( IP Disabled )
Local IP address: 0.0.0.0
Remote IP address: 0.0.0.0
Negotiated IPX options: 00( IPX Disabled )
IPX network number: 00 00 00 00
Local IPX node number: 00 00 00 00 00 00
Remote IPX node number: 00 00 00 00 00 00
Remote IPX router name:
Authentication status: No inbound authentication negotiated
Inbound PeerID:

 **/
void 
CommandProcShowSerial::convert_to_xml_xn(const std::string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_xn" << endl;
  }
  StrProc proc_str(line, " ");
 
  if (line.find("Remote Maximum") != string::npos) {
      _xml_out += "<ppp_neg_remote_max>" + proc_str.get(4) + "</ppp_neg_remote_max>";
  }

  if (line.find("Negotiated IP ") != string::npos) {
      _xml_out += "<ppp_neg_remote_ip_opt>" + proc_str.get(3) + " " + proc_str.get(4) + " " + proc_str.get(5) + "</ppp_neg_remote_ip_opt>";
  }

  if (line.find("Local IP ") != string::npos) {
      _xml_out += "<ppp_neg_local_ip>" + proc_str.get(3) + "</ppp_neg_local_ip>";
  }

  if (line.find("Remote IP ") != string::npos) {
      _xml_out += "<ppp_neg_remote_ip>" + proc_str.get(3) + "</ppp_neg_remote_ip>";
  }

  if (line.find("Negotiated IPX") != string::npos) {
      _xml_out += "<ppp_neg_ipx_opt>" + proc_str.get(3) + " " + proc_str.get(4) + " " + proc_str.get(5) + " " + proc_str.get(6) + "</ppp_neg_ipx_opt>";
  }

  if (line.find("Local IPX node") != string::npos) {
      _xml_out += "<ppp_neg_local_ipx_node>" + proc_str.get(4) + " " + proc_str.get(5) + " " + proc_str.get(6) + " " + proc_str.get(7) + " " + proc_str.get(8) + " " + proc_str.get(9) + "</ppp_neg_local_ipx_node>";
  }

  if (line.find("Remote IPX node") != string::npos) {
      _xml_out += "<ppp_neg_remote_ipx_node>" + proc_str.get(4) + " " + proc_str.get(5) + " " + proc_str.get(6) + " " + proc_str.get(7) + " " + proc_str.get(8) + " " + proc_str.get(9) + "</ppp_neg_remote_ipx_node>";
  }

  if (line.find("Remote IPX router") != string::npos) {
      _xml_out += "<ppp_neg_remote_ipx_router>" + proc_str.get(4) + "</ppp_neg_remote_ipx_router>";
  }

  if (line.find("Authentication status") != string::npos) {
      _xml_out += "<ppp_neg_auth_status>" + proc_str.get(2) + " " + proc_str.get(3) + " " + proc_str.get(4) + " " + proc_str.get(5) + "</ppp_neg_auth_status>";
  }

  if (line.find("Inbound PeerID") != string::npos) {
      _xml_out += "<ppp_neg_peer_id>" + proc_str.get(2) + "</ppp_neg_peer_id>";
  }
  return;
}

/**
 *
        ------------------------------------------
                wan0.1: PPP IP CONFIGURATION
        ------------------------------------------

Enable the use of IP: No
Notify remote of locally-configure address: No
Local IP address( 0.0.0.0 = request ): 0.0.0.0
Request remote to provide local address: No
Provide remote with pre-configured address: No
Remote IP address: 0.0.0.0
Require that remote provide an address: No
 **/
void 
CommandProcShowSerial::convert_to_xml_ci(const std::string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_ci" << endl;
  }
  StrProc proc_str(line, " ");
 
  if (line.find("Enable the use") != string::npos) {
      _xml_out += "<ppp_ip_conf_enable_use>" + proc_str.get(5) + "</ppp_ip_conf_enable_use>";
  }

  if (line.find("Notify remote of locally") != string::npos) {
      _xml_out += "<ppp_ip_conf_remote_local>" + proc_str.get(5) + "</ppp_ip_conf_remote_local>";
  }

  if (line.find("Local IP address") != string::npos) {
      _xml_out += "<ppp_ip_conf_local_ip_addr>" + proc_str.get(7) + "</ppp_ip_conf_local_ip_addr>";
  }

  if (line.find("Request remote to provide") != string::npos) {
      _xml_out += "<ppp_ip_conf_provide_local_addr>" + proc_str.get(6) + "</ppp_ip_conf_provide_local_addr>";
  }

  if (line.find("pre-configured address") != string::npos) {
      _xml_out += "<ppp_ip_conf_pre_addr>" + proc_str.get(5) + "</ppp_ip_conf_pre_addr>";
  }

  if (line.find("Remote IP address") != string::npos) {
      _xml_out += "<ppp_ip_conf_remote_ip_addr>" + proc_str.get(3) + "</ppp_ip_conf_remote_ip_addr>";
  }

  if (line.find("remote provide an address") != string::npos) {
      _xml_out += "<ppp_ip_conf_remote_provide_addr>" + proc_str.get(6) + "</ppp_ip_conf_remote_provide_addr>";
  }
  return;
}

/**
 *
        -----------------------------------------------------
                wan0.1: GENERAL CONFIGURATION 502 Board
        -----------------------------------------------------

Discard transmit-aborted frames: No
Baud rate in bps: 0
Update transmit statistics( user data ): No
Update receive statistics( user data ): No
Timestamp received packets: No
Maximum Receive/Transmit Unit(MRU/MTU): 0
Minimum remote MRU required on connection: 0

 **/
void 
CommandProcShowSerial::convert_to_xml_cg(const std::string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_cg" << endl;
  }
  StrProc proc_str(line, " ");
 
  if (line.find("Discard transmit-aborted") != string::npos) {
      _xml_out += "<gen_502_xmit_abort_frm>" + proc_str.get(3) + "</gen_502_xmit_abort_frm>";
  }


  if (line.find("Baud rate in bps") != string::npos) {
      _xml_out += "<gen_502_baud_bps>" + proc_str.get(4) + "</gen_502_baud_bps>";
  }

  if (line.find("Update transmit stat") != string::npos) {
      _xml_out += "<gen_502_xmit_stat>" + proc_str.get(6) + "</gen_502_xmit_stat>";
  }

  if (line.find("Update receive stat") != string::npos) {
      _xml_out += "<gen_502_rec_stat>" + proc_str.get(6) + "</gen_502_rec_stat>";
  }

  if (line.find("Timestamp receive stat") != string::npos) {
      _xml_out += "<gen_502_timestamp_rec_pkts>" + proc_str.get(4) + "</gen_502_timestamp_rec_pkts>";
  }

  if (line.find("Maximum Receive/Transmit") != string::npos) {
      _xml_out += "<gen_502_max_mru>" + proc_str.get(3) + "</gen_502_max_mru>";
  }

  if (line.find("Minimum remote MRU") != string::npos) {
      _xml_out += "<gen_502_min_remote_mru>" + proc_str.get(6) + "</gen_502_min_remote_mru>";
  }
  return;
}

/**
 *
Packets discarded (unknown LCP code): 0
Received LCP packets too large: 0
Received packets invalid or out-of-sequence Configure-Acks: 0
Received packets invalid Configure-Naks or Configure-Rejects: 0
Configure-Naks or Configure-Rejects with bad Identifier: 0

                        Received        Transmitted
Number of Config-Request pkts: 0        0
Number of Config-Ack packets : 0        0
Number of Config-Nack packets: 0        0
Number of Config-Reject packets: 0      0
Number of Term-Reqst packets : 0        0
Number of Terminate-Ack packets: 0      0
Number of Code-Reject packets: 0        0
Number of Protocol-Rej packets: 0       0
Number of Echo-Request packets: 0       0
Number of Echo-Reply packets : 0        0
Number of Discard-Request packets: 0    0

 **/
void 
CommandProcShowSerial::convert_to_xml_slcp(const std::string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_slcp" << endl;
  }
  StrProc proc_str(line, " ");
 
  if (line.find("unknown LCP code") != string::npos) {
      _xml_out += "<ppp_lcp_stat_unknown_lcp_code>" + proc_str.get(5) + "</ppp_lcp_stat_unknown_lcp_code>";
  }

  if (line.find("Received LCP packets") != string::npos) {
      _xml_out += "<ppp_lcp_stat_rc_lcp_pkts>" + proc_str.get(5) + "</ppp_lcp_stat_rc_lcp_pkts>";
  }

  if (line.find("Configure-Acks:") != string::npos) {
      _xml_out += "<ppp_lcp_stat_rc_seq_conf_acks>" + proc_str.get(6) + "</ppp_lcp_stat_rc_seq_conf_acks>";
  }

  if (line.find("Configure-Rejects:") != string::npos) {
      _xml_out += "<ppp_lcp_stat_rc_conf_rej>" + proc_str.get(6) + "</ppp_lcp_stat_rc_conf_rej>";
  }

  if (line.find("bad Identifier:") != string::npos) {
      _xml_out += "<ppp_lcp_stat_bad_ident>" + proc_str.get(6) + "</ppp_lcp_stat_bad_ident>";
  }

  if (line.find("Config-Request") != string::npos) {
      _xml_out += "<ppp_lcp_stat_num_rx_conf_req>" + proc_str.get(4) + "</ppp_lcp_stat_num_rx_conf_req>";
      _xml_out += "<ppp_lcp_stat_num_tx_conf_req>" + proc_str.get(5) + "</ppp_lcp_stat_num_tx_conf_req>";
  }

  if (line.find("Config-Ack") != string::npos) {
      _xml_out += "<ppp_lcp_stat_num_rx_conf_ack>" + proc_str.get(5) + "</ppp_lcp_stat_num_rx_conf_ack>";
      _xml_out += "<ppp_lcp_stat_num_tx_conf_ack>" + proc_str.get(6) + "</ppp_lcp_stat_num_tx_conf_ack>";
  }

  if (line.find("Config-Nack") != string::npos) {
      _xml_out += "<ppp_lcp_stat_num_rx_conf_nack>" + proc_str.get(4) + "</ppp_lcp_stat_num_rx_conf_nack>";
      _xml_out += "<ppp_lcp_stat_num_tx_conf_nack>" + proc_str.get(5) + "</ppp_lcp_stat_num_tx_conf_nack>";
  }

  if (line.find("Config-Reject") != string::npos) {
      _xml_out += "<ppp_lcp_stat_num_rx_conf_rej>" + proc_str.get(4) + "</ppp_lcp_stat_num_rx_conf_rej>";
      _xml_out += "<ppp_lcp_stat_num_tx_conf_rej>" + proc_str.get(5) + "</ppp_lcp_stat_num_tx_conf_rej>";
  }

  if (line.find("Term-Reqst") != string::npos) {
      _xml_out += "<ppp_lcp_stat_num_rx_term_reqst>" + proc_str.get(5) + "</ppp_lcp_stat_num_rx_term_reqst>";
      _xml_out += "<ppp_lcp_stat_num_tx_term_reqst>" + proc_str.get(6) + "</ppp_lcp_stat_num_tx_term_reqst>";
  }

  if (line.find("Terminate-Ack") != string::npos) {
      _xml_out += "<ppp_lcp_stat_num_rx_term_ack>" + proc_str.get(4) + "</ppp_lcp_stat_num_rx_term_ack>";
      _xml_out += "<ppp_lcp_stat_num_tx_term_ack>" + proc_str.get(5) + "</ppp_lcp_stat_num_tx_term_ack>";
  }

  if (line.find("Code-Reject") != string::npos) {
      _xml_out += "<ppp_lcp_stat_num_rx_code_rej>" + proc_str.get(4) + "</ppp_lcp_stat_num_rx_code_rej>";
      _xml_out += "<ppp_lcp_stat_num_tx_code_rej>" + proc_str.get(5) + "</ppp_lcp_stat_num_tx_code_rej>";
  }

  if (line.find("Protocol-Rej") != string::npos) {
      _xml_out += "<ppp_lcp_stat_num_rx_proto_rej>" + proc_str.get(4) + "</ppp_lcp_stat_num_rx_proto_rej>";
      _xml_out += "<ppp_lcp_stat_num_tx_proto_rej>" + proc_str.get(5) + "</ppp_lcp_stat_num_tx_proto_rej>";
  }

  if (line.find("Echo-Request") != string::npos) {
      _xml_out += "<ppp_lcp_stat_num_rx_echo_req>" + proc_str.get(4) + "</ppp_lcp_stat_num_rx_echo_req>";
      _xml_out += "<ppp_lcp_stat_num_tx_echo_req>" + proc_str.get(5) + "</ppp_lcp_stat_num_tx_echo_req>";
  }

  if (line.find("Echo-Reply") != string::npos) {
      _xml_out += "<ppp_lcp_stat_num_rx_echo_rep>" + proc_str.get(5) + "</ppp_lcp_stat_num_rx_echo_rep>";
      _xml_out += "<ppp_lcp_stat_num_tx_echo_rep>" + proc_str.get(6) + "</ppp_lcp_stat_num_tx_echo_rep>";
  }

  if (line.find("Discard-Request") != string::npos) {
      _xml_out += "<ppp_lcp_stat_num_rx_disc_req>" + proc_str.get(4) + "</ppp_lcp_stat_num_rx_disc_req>";
      _xml_out += "<ppp_lcp_stat_num_tx_disc_req>" + proc_str.get(5) + "</ppp_lcp_stat_num_tx_disc_req>";
  }
  return;
}


/**
 *
        -----------------------------------------
                wan0.1: CHDLC CONFIGURATION
        -----------------------------------------

Baud rate: 0
Line configuration: V.35

Link State depends on:
                        DCD:  YES
                        CTS:  YES
        Keepalive Reception:  YES


                 Maximum data field length:  0
          Keepalive transmit interval (ms):  0
Expected keepalive reception interval (ms):  0
       Keepalive reception error tolerance:  0
      SLARP request transmit interval (ms):  0
 **/
void 
CommandProcShowSerial::convert_to_xml_crc(const std::string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_crc" << endl;
  }
  StrProc proc_str(line, " ");
 
  if (line.find("Line configuration") != string::npos) {
      _xml_out += "<chdlc_conf_line>" + proc_str.get(2) + "</chdlc_conf_line>";
  }

  if (line.find("DCD") != string::npos) {
      _xml_out += "<chdlc_conf_dcd>" + proc_str.get(1) + "</chdlc_conf_dcd>";
  }

  if (line.find("CTS") != string::npos) {
      _xml_out += "<chdlc_conf_cts>" + proc_str.get(1) + "</chdlc_conf_cts>";
  }

  if (line.find("Keepalive Reception") != string::npos) {
      _xml_out += "<chdlc_conf_keepalive_rcpt>" + proc_str.get(2) + "</chdlc_conf_keepalive_rcpt>";
  }

  if (line.find("Maximum data field") != string::npos) {
      _xml_out += "<chdlc_conf_max_data_field>" + proc_str.get(4) + "</chdlc_conf_max_data_field>";
  }

  if (line.find("Keepalive transmit") != string::npos) {
      _xml_out += "<chdlc_conf_keepalive_xmit>" + proc_str.get(4) + "</chdlc_conf_keepalive_xmit>";
  }

  if (line.find("Expected keepalive reception") != string::npos) {
      _xml_out += "<chdlc_conf_expct_keepalive_rcpt>" + proc_str.get(5) + "</chdlc_conf_expct_keepalive_rcpt>";
  }

  if (line.find("Keepalive reception error") != string::npos) {
      _xml_out += "<chdlc_conf_keepalive_rcpt_err>" + proc_str.get(4) + "</chdlc_conf_keepalive_rcpt_err>";
  }

  if (line.find("SLARP") != string::npos) {
      _xml_out += "<chdlc_conf_slarp>" + proc_str.get(5) + "</chdlc_conf_slarp>";
  }
  return;
}


/**
 *
        ----------------------------------------------
                wan0: AFT OPERATIONAL STATISTICS
        ----------------------------------------------

             Number of frames transmitted:   0
              Number of bytes transmitted:   0
                      Transmit Throughput:   0
 Transmit frames discarded (length error):   0
                Transmit frames realigned:   0

                Number of frames received:   0
                 Number of bytes received:   0
                       Receive Throughput:   0
    Received frames discarded (too short):   0
     Received frames discarded (too long):   0
Received frames discarded (link inactive):   0

HDLC link active/inactive and loopback statistics
                           Times that the link went active:   0
         Times that the link went inactive (modem failure):   1
     Times that the link went inactive (keepalive failure):   0
                                         link looped count:   0


 **/
void 
CommandProcShowSerial::convert_to_xml_so(const std::string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_so" << endl;
  }
  StrProc proc_str(line, " ");

 
  if (line.find("Number of frames transmitted") != string::npos) {
      _xml_out += "<chdlc_aft_num_frame_xmit>" + proc_str.get(4) + "</chdlc_aft_num_frame_xmit>";
  }

  if (line.find("Number of bytes transmitted") != string::npos) {
      _xml_out += "<chdlc_aft_num_bytes_xmit>" + proc_str.get(4) + "</chdlc_aft_num_bytes_xmit>";
  }

  if (line.find("Transmit Throughput") != string::npos) {
      _xml_out += "<chdlc_aft_xmit_throughput>" + proc_str.get(2) + "</chdlc_aft_xmit_throughput>";
  }

  if (line.find("Transmit frames discarded") != string::npos) {
      _xml_out += "<chdlc_aft_xmit_frames_discard>" + proc_str.get(5) + "</chdlc_aft_xmit_frames_discard>";
  }

  if (line.find("Transmit frames realigned") != string::npos) {
      _xml_out += "<chdlc_aft_xmit_frames_realign>" + proc_str.get(3) + "</chdlc_aft_xmit_frames_realign>";
  }

  if (line.find("Number of frames received") != string::npos) {
      _xml_out += "<chdlc_aft_num_frame_rec>" + proc_str.get(4) + "</chdlc_aft_num_frame_rec>";
  }

  if (line.find("Number of bytes received") != string::npos) {
      _xml_out += "<chdlc_aft_num_bytes_rec>" + proc_str.get(4) + "</chdlc_aft_num_bytes_rec>";
  }

  if (line.find("Receive Throughput") != string::npos) {
      _xml_out += "<chdlc_aft_rec_tput>" + proc_str.get(2) + "</chdlc_aft_rec_tput>";
  }

  if (line.find("discarded (too short") != string::npos) {
      _xml_out += "<chdlc_aft_num_frames_too_shrt>" + proc_str.get(5) + "</chdlc_aft_num_frames_too_shrt>";
  }

  if (line.find("discarded (too long") != string::npos) {
      _xml_out += "<chdlc_aft_frames_too_lng>" + proc_str.get(5) + "</chdlc_aft_frames_too_lng>";
  }

  if (line.find("(link inactive)") != string::npos) {
      _xml_out += "<chdlc_aft_num_frame_link_inactive>" + proc_str.get(5) + "</chdlc_aft_num_frame_link_inactive>";
  }

  if (line.find("link went active") != string::npos) {
      _xml_out += "<chdlc_aft_time_link_active>" + proc_str.get(6) + "</chdlc_aft_time_link_active>";
  }

  if (line.find("(modem failure)") != string::npos) {
      _xml_out += "<chdlc_aft_modem_fail>" + proc_str.get(8) + "</chdlc_aft_modem_fail>";
  }

  if (line.find("(keepalive failure)") != string::npos) {
      _xml_out += "<chdlc_aft_keepalive_fail>" + proc_str.get(8) + "</chdlc_aft_keepalive_fail>";
  }

  if (line.find("link looped count") != string::npos) {
      _xml_out += "<chdlc_aft_link_loop_ct>" + proc_str.get(3) + "</chdlc_aft_link_loop_ct>";
  }
  return;
}


/**
 *
        --------------------------------------
                wan0.1: SLARP STATISTICS
        --------------------------------------


 SLARP frame transmission/reception statistics
       SLARP request packets transmitted:   0
          SLARP request packets received:   0

         SLARP Reply packets transmitted:   0
            SLARP Reply packets received:   0

     SLARP keepalive packets transmitted:   0
        SLARP keepalive packets received:   0

Incoming SLARP Packets with format errors
                      Invalid SLARP Code:   0
                Replies with bad IP addr:   0
                Replies with bad netmask:   0

SLARP timeout/retry statistics
                  SLARP Request timeouts:   0
            keepalive reception timeouts:   0

Cisco Discovery Protocol frames
                             Transmitted:   0
                                Received:   0

 **/
void 
CommandProcShowSerial::convert_to_xml_ss(const std::string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_ss" << endl;
  }
  StrProc proc_str(line, " ");
 
  if (line.find("SLARP request packets transmitted") != string::npos) {
      _xml_out += "<chdlc_slarp_req_xmit>" + proc_str.get(4) + "</chdlc_slarp_req_xmit>";
  }

  if (line.find("SLARP request packets received") != string::npos) {
      _xml_out += "<chdlc_slarp_req_rec>" + proc_str.get(4) + "</chdlc_slarp_req_rec>";
  }

  if (line.find("SLARP Reply packets transmitted") != string::npos) {
      _xml_out += "<chdlc_slarp_rep_xmit>" + proc_str.get(4) + "</chdlc_slarp_rep_xmit>";
  }

  if (line.find("SLARP Reply packets received") != string::npos) {
      _xml_out += "<chdlc_slarp_rep_rec>" + proc_str.get(4) + "</chdlc_slarp_rep_rec>";
  }

  if (line.find("SLARP keepalive packets transmitted") != string::npos) {
      _xml_out += "<chdlc_slarp_keepalive_xmit>" + proc_str.get(4) + "</chdlc_slarp_keepalive_xmit>";
  }

  if (line.find("SLARP keepalive packets received") != string::npos) {
      _xml_out += "<chdlc_slarp_keepalive_rec>" + proc_str.get(4) + "</chdlc_slarp_keepalive_rec>";
  }

  if (line.find("Invalid SLARP Code") != string::npos) {
      _xml_out += "<chdlc_slarp_inv_slarp_code>" + proc_str.get(3) + "</chdlc_slarp_inv_slarp_code>";
  }

  if (line.find("Replies with bad IP addr") != string::npos) {
      _xml_out += "<chdlc_slarp_rep_bad_ip_addr>" + proc_str.get(5) + "</chdlc_slarp_rep_bad_ip_addr>";
  }

  if (line.find("Replies with bad netmask") != string::npos) {
      _xml_out += "<chdlc_slarp_rep_bad_netmask>" + proc_str.get(4) + "</chdlc_slarp_rep_bad_netmask>";
  }

  if (line.find("SLARP Request timetouts") != string::npos) {
      _xml_out += "<chdlc_slarp_req_timeouts>" + proc_str.get(3) + "</chdlc_slarp_req_timeouts>";
  }

  if (line.find("keepalive reception timeouts") != string::npos) {
      _xml_out += "<chdlc_slakeepalive_rec_timeouts>" + proc_str.get(3) + "</chdlc_slakeepalive_rec_timeouts>";
  }

  if (line.find("Transmitted: ") != string::npos) {
      _xml_out += "<chdlc_slarp_cisco_disc_xmit>" + proc_str.get(1) + "</chdlc_slarp_cisco_disc_xmit>";
  }

  if (line.find("Received: ") != string::npos) {
      _xml_out += "<chdlc_slarp_cisco_disc_rec>" + proc_str.get(1) + "</chdlc_slarp_cisco_disc_rec>";
  }
  return;
}


/**
 *
        -----------------------------------
                wan0: AFT LINK STATUS
        -----------------------------------

Device Link Status:     Disconnected

 **/
void 
CommandProcShowSerial::convert_to_xml_xl(const std::string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_xl" << endl;
  }
  StrProc proc_str(line, " ");
 
  if (line.find("Device Link Status") != string::npos) {
      _xml_out += "<chdlc_link_status>" + proc_str.get(3) + "</chdlc_link_status>";
  }
  return;
}


/**
 *
        ------------------------------------
                wan0: AFT CODE VERSION
        ------------------------------------

Code version: HDLC rev.25

 **/
void 
CommandProcShowSerial::convert_to_xml_xcv(const std::string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_xcv" << endl;
  }
  StrProc proc_str(line, " ");
 
  if (line.find("Code version:") != string::npos) {
      _xml_out += "<chdlc_code_ver>" + proc_str.get(2) + " " + proc_str.get(3) + "</chdlc_code_ver>";
  }
}

/**
 *
        ---------------------------------------
                wan0.17: ERROR STATISTICS
        ---------------------------------------

  I-frames not transmitted after a tx. int. due to exessive frame length: 0
   I-frames not transmitted after a tx. int. due to excessive throughput: 0
     Received frames discarded as they were either too short or too long: 0
                               discarded I-frames with unconfigured DLCI: 0
                                discarded I-frames due to a format error: 0
App. didn't respond to the triggered IRQ within the given timeout period: 0
            discarded In-channel Signalling frames due to a format error: 0
   In-channel frames received with an invalid Send Seq. Numbers received: 0
In-channel frames received with an invalid Receive Seq. Numbers received: 0
                    Number of unsolicited responses from the Access Node: 0
                                              timeouts on the T391 timer: 0
                                  consecutive timeouts on the T391 timer: 0
times that N392 error threshold was reached during N393 monitored events: 0
 **/
void 
CommandProcShowSerial::convert_to_xml_se(const std::string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_se" << endl;
  }
  StrProc proc_str(line, " ");
 
  if (line.find("length:") != string::npos) {
      _xml_out += "<fr_err_ex_fr_len>" + proc_str.get(12) + "</fr_err_ex_fr_len>";
  }

  if (line.find("throughput:") != string::npos) {
      _xml_out += "<fr_err_ex_tput>" + proc_str.get(11) + "</fr_err_ex_tput>";
  }

  if (line.find("too long:") != string::npos) {
      _xml_out += "<fr_err_dis_wr_size>" + proc_str.get(12) + "</fr_err_dis_wr_size>";
  }

  if (line.find("length:") != string::npos) {
      _xml_out += "<fr_err_fr_len>" + proc_str.get(12) + "</fr_err_fr_len>";
  }

  if (line.find("DLCI:") != string::npos) {
      _xml_out += "<fr_err_unc_dlci>" + proc_str.get(5) + "</fr_err_unc_dlci>";
  }

  if (line.find("format error:") != string::npos) {
      _xml_out += "<fr_err_for_err>" + proc_str.get(9) + "</fr_err_for_err>";
  }

  if (line.find("Send Seq. Numbers received:") != string::npos) {
      _xml_out += "<fr_err_inv_send_num>" + proc_str.get(10) + "</fr_err_inv_send_num>";
  }

  if (line.find("Receive Seq. Numbers received:") != string::npos) {
      _xml_out += "<fr_err_inv_rec_num>" + proc_str.get(10) + "</fr_err_inv_rec_num>";
  }

  if (line.find("Node:") != string::npos) {
      _xml_out += "<fr_err_access_node>" + proc_str.get(8) + "</fr_err_access_node>";
  }

  if (line.find("  timeouts on the T391") != string::npos) {
      _xml_out += "<fr_err_time_t391>" + proc_str.get(5) + "</fr_err_time_t391>";
  }

  if (line.find("consecutive times on the T391") != string::npos) {
      _xml_out += "<fr_err_con_time_t391>" + proc_str.get(6) + "</fr_err_con_time_t391>";
  }

  if (line.find("monitored events:") != string::npos) {
      _xml_out += "<fr_err_thresh_n393>" + proc_str.get(11) + "</fr_err_thresh_n393>";
  }
  return;
}

/**
 *
~ # wanpipemon -i wan0.17 -c cl


        -------------------------------------------
                wan0.17: LIST OF ACTIVE DLCIs
        -------------------------------------------

All DLCIs are Inactive.
 **/
void 
CommandProcShowSerial::convert_to_xml_cl(const std::string &line, bool debug)
{
  if (debug) {
    cout << "processing: convert_to_xml_cl" << endl;
  }
  StrProc proc_str(line, " ");

  if (line.find("  DLCI:") != string::npos) {
      _xml_out += "<fr_active_dlci_active_vif aux='" + proc_str.get(2) + "'>" + proc_str.get(2) + "</fr_active_dlci_active_vif>";
      _xml_out += "<fr_active_dlci_active_name aux='" + proc_str.get(2) + "'>" + proc_str.get(0) + "</fr_active_dlci_active_name>";
      _xml_out += "<fr_active_dlci_active_status aux='" + proc_str.get(2) + "'>" + proc_str.get(3) + "</fr_active_dlci_active_status>";
  }
  return;
}

/**
 *
 **/
void
CommandProcShowSerial::process_dlcis(const string &iface, const string &vif_iface, const string &cl_vif_arg, bool debug)
{
  if (debug) {
    cout << "processing: process_dlcis" << endl;
  }
  map<string, string> coll;
  string cmd = "wanpipemon -i " + vif_iface + " -c clr";
  FILE *f = popen(cmd.c_str(), "r");
  if (f) {
    char buf[2048];
    while(fgets(buf, 2047, f) != NULL) { 

      string line(buf);
      string fragment;
      StrProc proc_str(line, " ");
     
      if (line.find("active") != string::npos) {
	string vif = proc_str.get(2).substr(0,proc_str.get(2).length()-1);
	//if the vif is not specified then provide all, otherwise only matched vif
	if (cl_vif_arg.empty() == true || cl_vif_arg == vif) {
	  fragment = "<row name='frame_dlci'>";
	  fragment += "<dlci_list_iface>" + proc_str.get(0).substr(0,proc_str.get(0).length()-1) + "</dlci_list_iface>";
	  fragment += "<dlci_list_dlci>" + vif + "</dlci_list_dlci>";
	  fragment += "<dlci_list_status>" + proc_str.get(3) + "</dlci_list_status>";
	  coll.insert(pair<string, string>(vif, fragment));
	}
      }
    }
    if (pclose(f) != 0) {
      return;
    }
  }

  string proc_xml;
  map<string, string>::iterator iter = coll.begin();
  while (iter != coll.end()) {
    proc_xml += iter->second;
    //now process another command
    string stat_cmd = "wanpipemon -i " + iface + "." + iter->first + " -c sd -d " + iter->first;
    f = popen(stat_cmd.c_str(), "r");
    if (f) {
      char buf_stat[2048];
      while (fgets(buf_stat, 2047, f) != NULL) {
	string line_stat(buf_stat);
	StrProc proc_str(line_stat, " ");
	
	if (line_stat.find("frames transmitted") != string::npos) {
	  proc_xml += "<dlci_frms_trans>" + proc_str.get(3) + "</dlci_frms_trans>";
	}
	if (line_stat.find("bytes transmitted") != string::npos) {
	  proc_xml += "<dlci_bytes_trans>" + proc_str.get(3) + "</dlci_bytes_trans>";
	}
	if (line_stat.find("Information frames received") != string::npos) {
	  proc_xml += "<dlci_frms_recv>" + proc_str.get(3) + "</dlci_frms_recv>";
	}
	if (line_stat.find("bytes received") != string::npos) {
	  proc_xml += "<dlci_bytes_recv>" + proc_str.get(3) + "</dlci_bytes_recv>";
	}
	if (line_stat.find("I-frames discarded") != string::npos) {
	  proc_xml += "<dlci_ifrms_disc>" + proc_str.get(7) + "</dlci_ifrms_disc>";
	}
	if (line_stat.find("(DE)") != string::npos) {
	  proc_xml += "<dlci_disc_elg_set>" + proc_str.get(8) + "</dlci_disc_elg_set>";
	}
	if (line_stat.find("FECN") != string::npos) {
	  proc_xml += "<dlci_FECN>" + proc_str.get(7) + "</dlci_FECN>";
	}
	if (line_stat.find("BECN") != string::npos) {
	  proc_xml += "<dlci_BECN>" + proc_str.get(7) + "</dlci_BECN>";
	}
      }
      if (pclose(f) != 0) {
	return;
      }
    }
    ++iter;
    proc_xml += "</row>";
  }
  _xml_out += proc_xml;
}
