/*
 * Module: netlink_rt_entry.cc
 *
 * **** License ****
 * Version: VPL 1.0
 *
 * The contents of this file are subject to the Vyatta Public License
 * Version 1.0 ("License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.vyatta.com/vpl
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * This code was originally developed by Vyatta, Inc.
 * Portions created by Vyatta are Copyright (C) 2005, 2006, 2007 Vyatta, Inc.
 * All Rights Reserved.
 *
 * Author: Stig Thormodsrud
 * Date: 2007
 * Description: 
 *
 * **** End License ****
 *
 */

#include "rib_module.h"
#include "route.hh"

#include "libxorp/xorp.h"
#include "libxorp/xlog.h"

#include "netlink_rt_entry.hh"


NetlinkRouteEntry::NetlinkRouteEntry(EventLoop& eventloop) 
    : NetlinkSocket(eventloop),
      _is_running(false),
      _pid(getpid()),
      _ns_reader(*(NetlinkSocket *)this)
{
}

NetlinkRouteEntry::~NetlinkRouteEntry(void)
{
    _pid = 0;
}

bool NetlinkRouteEntry::start(void)
{
    string error_msg;

    if (_is_running) {
	return (XORP_OK);
    }

    if (NetlinkSocket::start(error_msg) < 0) {
        XLOG_ERROR("%s", error_msg.c_str());
	return (XORP_ERROR);
    }
    
    _is_running = true;

    return (XORP_OK);
}

bool NetlinkRouteEntry::stop(void)
{
    string error_msg;

    if (! _is_running) {
        return (XORP_OK);
    }

    if (NetlinkSocket::stop(error_msg) < 0) {
        XLOG_ERROR("%s", error_msg.c_str());
	return (XORP_ERROR);
    }

    _is_running = false;

    return (XORP_OK);
}

template <typename A>
bool NetlinkRouteEntry::add_entry(const IPRouteEntry<A>& route)
{
    struct nlmsghdr*	nlh = &_buffer.nlh;
    struct sockaddr_nl	snl;
    struct rtmsg*	rtmsg;
    struct rtattr*	rtattr;
    int			rta_len;
    uint8_t*		data;
    NetlinkSocket&	ns = *this;
    uint32_t            bufsiz = sizeof(_buffer);

    if (route.protocol().name() == "connected") {
	return true;
    }

    snl.nl_family = AF_NETLINK;
    snl.nl_pad    = 0;
    snl.nl_pid    = 0;		// nl_pid = 0 if destination is the kernel
    snl.nl_groups = 0;

    nlh->nlmsg_len   = NLMSG_LENGTH(sizeof(*rtmsg));
    nlh->nlmsg_type  = RTM_NEWROUTE;
    nlh->nlmsg_flags = NLM_F_REQUEST| NLM_F_CREATE| NLM_F_REPLACE| NLM_F_ACK;
    nlh->nlmsg_seq   = ns.seqno();
    nlh->nlmsg_pid   = ns.nl_pid();

    int family   = route.net().af();

    rtmsg = static_cast<struct rtmsg*>(NLMSG_DATA(nlh));
    rtmsg->rtm_family   = family;
    rtmsg->rtm_dst_len  = route.net().prefix_len(); 
    rtmsg->rtm_src_len  = 0;
    rtmsg->rtm_tos      = 0;
    rtmsg->rtm_table    = RT_TABLE_MAIN;
    rtmsg->rtm_protocol = RTPROT_XORP;		// Mark this as a XORP route
    rtmsg->rtm_scope    = RT_SCOPE_UNIVERSE;
    rtmsg->rtm_type     = RTN_UNICAST;
    rtmsg->rtm_flags    = RTM_F_NOTIFY;

    // Add the destination address as an attribute
    rta_len = RTA_LENGTH(route.net().masked_addr().addr_bytelen());
    if (NLMSG_ALIGN(nlh->nlmsg_len) + rta_len > bufsiz) {
        XLOG_ERROR("AF_NETLINK buffer size error: %u instead of %u",
		   XORP_UINT_CAST(bufsiz),
		   XORP_UINT_CAST(NLMSG_ALIGN(nlh->nlmsg_len) + rta_len));
	return false;
    }
    rtattr = XORP_RTM_RTA(rtmsg);
    rtattr->rta_type = RTA_DST;
    rtattr->rta_len  = rta_len;
    data = static_cast<uint8_t*>(RTA_DATA(rtattr));
    route.net().masked_addr().copy_out(data);
    nlh->nlmsg_len = NLMSG_ALIGN(nlh->nlmsg_len) + rta_len;

    void*    rta_align_data;

    // Add the nexthop address as an attribute
    // An interface route will not have a next-hop
    if (route.nexthop_addr() != A::ZERO(family)) {
        rta_len = RTA_LENGTH(route.nexthop_addr().addr_bytelen());
	if (NLMSG_ALIGN(nlh->nlmsg_len) + rta_len > bufsiz) {
	    XLOG_ERROR("AF_NETLINK buffer size error: %u instead of %u",
		       XORP_UINT_CAST(bufsiz),
		       XORP_UINT_CAST(NLMSG_ALIGN(nlh->nlmsg_len) + rta_len));
	    return false;
	}
	rta_align_data = reinterpret_cast<char*>(rtattr)+ 
	    RTA_ALIGN(rtattr->rta_len);
	rtattr = static_cast<struct rtattr*>(rta_align_data);
	rtattr->rta_type = RTA_GATEWAY;
	rtattr->rta_len  = rta_len;
	data = static_cast<uint8_t*>(RTA_DATA(rtattr));
	route.nexthop_addr().copy_out(data);
	nlh->nlmsg_len = NLMSG_ALIGN(nlh->nlmsg_len) + rta_len;
    } 

    // Get the interface index, if it exists.
    // If the interface has an index in the host stack, add it
    // as an attribute.
    uint32_t if_index = 0;
    if_index = route.vif()->pif_index();
    if (if_index != 0) {
        int int_if_index = if_index;
	rta_len = RTA_LENGTH(sizeof(int_if_index));
	if (NLMSG_ALIGN(nlh->nlmsg_len) + rta_len > bufsiz) {
	    XLOG_ERROR("AF_NETLINK buffer size error: %u instead of %u",
		       XORP_UINT_CAST(bufsiz),
		       XORP_UINT_CAST(NLMSG_ALIGN(nlh->nlmsg_len) + rta_len));
	}
	rta_align_data = reinterpret_cast<char*>(rtattr) + 
	    RTA_ALIGN(rtattr->rta_len);
	rtattr = static_cast<struct rtattr*>(rta_align_data);
	rtattr->rta_type = RTA_OIF;
	rtattr->rta_len  = rta_len;
	data = static_cast<uint8_t*>(RTA_DATA(rtattr));
	memcpy(data, &int_if_index, sizeof(int_if_index));
	nlh->nlmsg_len = NLMSG_ALIGN(nlh->nlmsg_len) + rta_len;
    } else {
        //
        // Xorp supports blackhole routes by creating a 
        // interface route to a non-existing interface.
        // Vyatta does not support non-existing interfaces,
        // so we should never get here.
        //
        XLOG_ERROR("unsupported interface %d", if_index);
	return false;
    }

    // Add the route priority as an attribute
    int int_priority = route.metric();
    rta_len = RTA_LENGTH(sizeof(int_priority));
    if (NLMSG_ALIGN(nlh->nlmsg_len) + rta_len > bufsiz) {
        XLOG_ERROR("AF_NETLINK buffer size error: %u instead of %u",
		   XORP_UINT_CAST(bufsiz),
		   XORP_UINT_CAST(NLMSG_ALIGN(nlh->nlmsg_len) + rta_len));
    }
    rta_align_data = reinterpret_cast<char*>(rtattr)+ 
        RTA_ALIGN(rtattr->rta_len);
    rtattr = static_cast<struct rtattr*>(rta_align_data);
    rtattr->rta_type = RTA_PRIORITY;
    rtattr->rta_len  = rta_len;
    data = static_cast<uint8_t*>(RTA_DATA(rtattr));
    memcpy(data, &int_priority, sizeof(int_priority));
    nlh->nlmsg_len = NLMSG_ALIGN(nlh->nlmsg_len) + rta_len;

    //
    // XXX: the Linux kernel doesn't keep the admin distance, hence
    // we don't add it.
    //
    
    string error_msg;
    ssize_t rc;
    rc = ns.sendto(&_buffer, nlh->nlmsg_len, 0, 
		   reinterpret_cast<struct sockaddr*>(&snl), sizeof(snl));
    if (rc != (ssize_t)nlh->nlmsg_len) {
        XLOG_ERROR("Error writing to netlink socket: %s", strerror(errno));
	return false;
    }
    
    int last_errno = 0;
    int nl_rc;
    nl_rc = NlmUtils::check_netlink_request(_ns_reader, ns, nlh->nlmsg_seq,
					    last_errno, error_msg);
    if (nl_rc != XORP_OK) {
        XLOG_ERROR("Error checking netlink request: %s", error_msg.c_str());
  	return false;
    }

    return true;
}

template <typename A>
bool NetlinkRouteEntry::delete_entry(const IPRouteEntry<A>& route)
{
    struct nlmsghdr*	nlh = &_buffer.nlh;
    struct sockaddr_nl	snl;
    struct rtmsg*	rtmsg;
    struct rtattr*	rtattr;
    int			rta_len;
    uint8_t*		data;
    NetlinkSocket&	ns = *this;
    uint32_t            bufsiz = sizeof(_buffer);

    if (route.protocol().name() == "connected") {
	return true;	
    }
    
    // Set the socket
    snl.nl_family = AF_NETLINK;
    snl.nl_pid    = 0;		// nl_pid = 0 if destination is the kernel
    snl.nl_groups = 0;

    //
    // Set the request
    //
    nlh->nlmsg_len   = NLMSG_LENGTH(sizeof(*rtmsg));
    nlh->nlmsg_type  = RTM_DELROUTE;
    nlh->nlmsg_flags = NLM_F_REQUEST | NLM_F_CREATE | NLM_F_REPLACE | NLM_F_ACK;
    nlh->nlmsg_seq   = ns.seqno();
    nlh->nlmsg_pid   = ns.nl_pid();

    int	family = route.net().af();

    rtmsg = static_cast<struct rtmsg*>(NLMSG_DATA(nlh));
    rtmsg->rtm_family   = family;
    rtmsg->rtm_dst_len  = route.net().prefix_len(); 
    rtmsg->rtm_src_len  = 0;
    rtmsg->rtm_tos      = 0;
    rtmsg->rtm_table    = RT_TABLE_MAIN;
    rtmsg->rtm_protocol = RTPROT_XORP;		// Mark this as a XORP route
    rtmsg->rtm_scope    = RT_SCOPE_UNIVERSE;
    rtmsg->rtm_type     = RTN_UNICAST;
    rtmsg->rtm_flags    = RTM_F_NOTIFY;

    // Add the destination address as an attribute
    rta_len = RTA_LENGTH(route.net().masked_addr().addr_bytelen());
    if (NLMSG_ALIGN(nlh->nlmsg_len) + rta_len > bufsiz) {
	XLOG_FATAL("AF_NETLINK buffer size error: %u instead of %u",
		   XORP_UINT_CAST(bufsiz),
		   XORP_UINT_CAST(NLMSG_ALIGN(nlh->nlmsg_len) + rta_len));
    }
    rtattr = XORP_RTM_RTA(rtmsg);
    rtattr->rta_type = RTA_DST;
    rtattr->rta_len  = rta_len;
    data = static_cast<uint8_t*>(RTA_DATA(rtattr));
    route.net().masked_addr().copy_out(data);
    nlh->nlmsg_len = NLMSG_ALIGN(nlh->nlmsg_len) + rta_len;

    ssize_t rc;
    rc = ns.sendto(&_buffer, nlh->nlmsg_len, 0,
		   reinterpret_cast<struct sockaddr*>(&snl), sizeof(snl));
    if (rc != (ssize_t)nlh->nlmsg_len) {
        XLOG_ERROR("Error writing to netlink socket: %s", strerror(errno));
	return false;
    }

    int last_errno = 0;
    string error_msg;
    int nl_rc;
    nl_rc = NlmUtils::check_netlink_request(_ns_reader, ns, nlh->nlmsg_seq,
					    last_errno, error_msg);
    if (nl_rc != XORP_OK) {
        //
	// XXX: If the outgoing interface was taken down earlier, then
	// most likely the kernel has removed the matching forwarding
	// entries on its own. Hence, check whether all of the following
	// is true:
	//   - the error code matches
	//   - the outgoing interface is down
	//
	// If all conditions are true, then ignore the error and consider
	// the deletion was success.
	// Note that we could add to the following list the check whether
	// the forwarding entry is not in the kernel, but this is probably
	// an overkill. If such check should be performed, we should
	// use the corresponding FtiConfigTableGetNetlink provider.
	//
        // Check whether the error code matches
        if (last_errno != ESRCH) {
	    //
	    // XXX: The "No such process" error code is used by the
	    // kernel to indicate there is no such forwarding entry
	    // to delete.
	    //
	    return true;
	}
        XLOG_ERROR("Error checking netlink request: %s", error_msg.c_str());
        return false;
    }

    return true;
}


//
// Instantiate IPv4 and IPv6 versions of the methods
//
template bool NetlinkRouteEntry::add_entry(const IPRouteEntry<IPv4>&);
template bool NetlinkRouteEntry::add_entry(const IPRouteEntry<IPv6>&);
template bool NetlinkRouteEntry::delete_entry(const IPRouteEntry<IPv4>&);
template bool NetlinkRouteEntry::delete_entry(const IPRouteEntry<IPv6>&);


