#ifndef __DISPLAY_HH__
#define __DISPLAY_HH__

#include <map>
#include <string>

/**
 *
 *
 **/
class RouteData
{
public:
  RouteData(IPv4 next_hop, string ifname, string vifname, uint32_t metric, uint32_t admin_distance, string cookie, string protocol_origin, bool recursive) : 
    _next_hop(next_hop),
    _ifname(ifname),
    _vifname(vifname),
    _metric(metric),
    _admin_distance(admin_distance),
    _cookie(cookie),
    _protocol_origin(protocol_origin),
    _recursive(recursive) {}

public:
  IPv4 _next_hop;
  string _ifname;
  string _vifname;
  uint32_t _metric;
  uint32_t _admin_distance;
  string _cookie;
  string _protocol_origin;
  bool _recursive;
};

/**
 *
 *
 **/
class RouteFormat
{
public:
  RouteFormat();
  RouteFormat(const string &hints);

  string
  process(const IPv4Net &route, const RouteData &data) const;

  IPv4Net 
  prefix() const {return _prefix;}
  
  int
  mask() const {return _mask;}

  string
  protocol() const {return _protocol;}

  IPv4
  nexthop() const {return _nexthop;}

  int
  exact() const {return _exact;}

  int
  filter() const {return _filter;}

  int
  match() const {return _match;}

private:
  IPv4Net _prefix;
  int     _mask;
  IPv4    _nexthop;
  string  _protocol;
  int     _mode;
  int     _exact;
  int     _match;
  int     _filter;
};

/**
 *
 *
 **/
template <class T>
class RouteKey 
{
public:
  RouteKey(T filt, IPv4Net &route) : _filt(filt), _route(route) {}
  //  RouteKey(const T filt, IPv4Net &route) : _filt(filt), _route(route) {}
  RouteKey(const T filt, const IPv4Net &route) : _filt(filt), _route(route) {}
  RouteKey(const IPv4Net &route) : _route(route) {}
  RouteKey() {}

  bool
  operator<(const RouteKey &key) const;
public:
  T     _filt;
  IPv4Net _route;
};

/**
 *
 *
 **/
template <class T>
class RouteContainer
{
public:
  typedef map<RouteKey<T>, RouteData*> RouteColl;
  typedef typename map<RouteKey<T>, RouteData*>::iterator RouteIter;

public:
  RouteContainer();
  ~RouteContainer() {}
  
  void
  add_route(const IPv4Net &prefix, const T &param, RouteData *data);
  
  void
  remove_route(const IPv4Net &prefix, const T &param, bool release_data = false);

  RouteData*
  get(const IPv4Net &prefix, const T &param);

  void
  send_data(int sock, const RouteKey<T> &start_key, const RouteKey<T> &end_key, const RouteFormat &formatter);

  bool
  send_data();

  int
  slice_size(const RouteKey<T> &start_key, const RouteKey<T> &end_key);

  int
  size() {return _route_coll.size();}

  RouteKey<T>
  match(const IPv4Net &prefix, const T &param);

  bool
  done() {return _done;}

private:
  RouteColl   _route_coll;
  RouteKey<T> _start_key;
  RouteKey<T> _end_key;
  RouteFormat _format;
  int         _max_output;
  bool        _done;
  int         _sock;
};

/**
 *
 *
 **/
class Protocol
{
public:
  Protocol() {;}
  Protocol(const string &proto) : _proto(proto) {}


  bool
  operator==(const Protocol &proto) const {
    return (_proto == proto._proto);
  }

  bool
  operator<(const Protocol &proto) const {
    return (proto._proto < _proto);
  }
  string _proto;
};

/**
 *
 *
 **/
class ProtocolBGP
{
public:
  ProtocolBGP() {;}
  ProtocolBGP(const string &proto)
  {
    _proto = proto;
    if (proto == "ibgp" || proto == "ebgp") {
      _proto = "bgp";
    }
  }


  bool
  operator==(const ProtocolBGP &proto) const {
    return (_proto == proto._proto);
  }

  bool
  operator<(const ProtocolBGP &proto) const {
    return (proto._proto < _proto);
  }
  string _proto;
};

/**
 * Class for Querying RIB routes
 *
 **/
class ShowRoutesProcessor
  : public XrlShowRoutesTargetBase, public ServiceBase {
public:
  ShowRoutesProcessor(EventLoop &e, bool test);
  ~ShowRoutesProcessor();
  
  bool startup();
  bool shutdown();
  
public:
  XrlCmdError common_0_1_get_target_name(string& name);
  XrlCmdError common_0_1_get_version(string& version);
  XrlCmdError common_0_1_get_status(uint32_t& status, string& reason);
  XrlCmdError common_0_1_shutdown();
  
  XrlCmdError finder_event_observer_0_1_xrl_target_birth(const string& cls,
							 const string& ins);
  XrlCmdError finder_event_observer_0_1_xrl_target_death(const string& cls,
							 const string& ins);
  
  XrlCmdError redist4_0_1_starting_route_dump(const string&	cookie);
  XrlCmdError redist4_0_1_finishing_route_dump(const string&	cookie);
  XrlCmdError redist4_0_1_add_route(const IPv4Net&	dst,
				    const IPv4&	nexthop,
				    const string&	ifname,
				    const string&	vifname,
				    const uint32_t&	metric,
				    const uint32_t&	admin_distance,
				    const string&	cookie,
				    const string&	protocol_origin,
				    const bool&         recursive);
  XrlCmdError redist4_0_1_delete_route(
				       const IPv4Net&	dst,
				       const IPv4&	nexthop,
				       const string&	ifname,
				       const string&	vifname,
				       const uint32_t&	metric,
				       const uint32_t&	admin_distance,
				       const string&	cookie,
				       const string&	protocol_origin,
				       const bool&	recursive);
  
  XrlCmdError redist6_0_1_starting_route_dump(const string&	cookie);
  XrlCmdError redist6_0_1_finishing_route_dump(const string&	cookie);
  XrlCmdError redist6_0_1_add_route(const IPv6Net&	dst,
				    const IPv6&	nexthop,
				    const string&	ifname,
				    const string&	vifname,
				    const uint32_t&	metric,
				    const uint32_t&	admin_distance,
				    const string&	cookie,
				    const string&	protocol_origin,
				    const bool&         recursive);
  XrlCmdError redist6_0_1_delete_route(
				       // Input values,
				       const IPv6Net&,
				       const IPv6&,
				       const string&,
				       const string&,
				       const uint32_t&,
				       const uint32_t&,
				       const string&,
				       const string&,
				       const bool&);
  
  
  bool poll_ready_failed();
  
  void init_command_request();
  bool poll_command_request();

  /**
   * @return true if cookie matches instance cookie, false otherwise.
   */
  bool check_cookie(const string& cookie);
  
  /**
   * Register with Finder to watch RIB birth and death events.  If
   * the RIB crashes we don't want to hang waiting for messages from
   * the RIB that will never arrive (start route dump, add route,
   * finish route dump)
   */
  void watch_rib();
  void watch_rib_cb(const XrlError& xe);
  
  /**
   * Request redistribution of user requested protocol.
   */
  void request_redist();
  void request_redist_cb(const XrlError& xe);
  
  /**
   * Request redistribution cease.
   */
  void request_cease();
  void request_cease_cb(const XrlError& xe);
  
  bool
  load_test_rib();
  
private:
  bool
  process_request();

  bool
  process_request(RouteFormat &format, int sock);

protected:
  EventLoop&			_e;
  XrlRouter*			_rtr;
  XorpTimer			_t_service;
  XorpTimer                     _t_pipe;
  XorpTimer                     _t_test;
  string			_cookie;
  int                           _request_sock;
  RouteContainer<int>           _route_coll;
  RouteContainer<Protocol>      _proto_coll;
  RouteContainer<ProtocolBGP>   _proto_bgp_coll;
  RouteContainer<IPv4>          _nexthop_coll;
  RouteContainer<int>           _mask_coll;
  bool                          _test;
};

#endif
