// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-

// Copyright (c) 2001-2005 International Computer Science Institute
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software")
// to deal in the Software without restriction, subject to the conditions
// listed in the XORP LICENSE file. These conditions include: you must
// preserve this copyright notice, and you cannot mention the copyright
// holders in advertising related to the Software without their permission.
// The Software is provided WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED. This
// notice is a summary of the XORP LICENSE file; the license in that file is
// legally binding.

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <string.h>
#include <stdio.h>
#include <sys/types.h>
#include <string>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "rib/rib_module.h"

#include "libxorp/xorp.h"
#include "libxorp/xlog.h"
#include "libxorp/status_codes.h"

#include <iomanip>
#include "libxorp/c_format.hh"
#include "libxorp/eventloop.hh"
#include "libxorp/eventloop_factory.hh"
#include "libxorp/ipv4.hh"
#include "libxorp/ipv6.hh"
#include "libxorp/service.hh"

#include "libxipc/xrl_std_router.hh"

#include "xrl/interfaces/rib_xif.hh"
#include "xrl/interfaces/finder_event_notifier_xif.hh"
#include "xrl/targets/show_routes_base.hh"
#include "common.hh"
#include "route_display_server.hh"

#define SOCK_PATH "/tmp/route_display"


/**
 *
 **/
template <class T>
bool
RouteKey<T>::operator<(const RouteKey<T> &key) const 
{
    if (key._filt < _filt) {
	return false;
    }
    else if (key._filt == _filt) {
	if (key._route.masked_addr() < _route.masked_addr()) {
	    return false;
	}
	else if (key._route.masked_addr() == _route.masked_addr()) {
	    if (key._route.netmask() <= _route.netmask()) {
		return false;
	    }
	}
    }
    return true;
}

/**
 *
 **/
template <>
bool
RouteKey<int>::operator<(const RouteKey<int> &key) const 
{
    IPv4Net a,b;
    //if -1 then make mask equal to incoming

    a = key._route;
    b = _route;

    if (key._filt == -1) {
	a = IPv4Net(key._route.masked_addr(), _route.prefix_len());
    }
    else if (_filt == -1) {
	b = IPv4Net(_route.masked_addr(), key._route.prefix_len());
    }

    if (a.masked_addr() < b.masked_addr()) {
	return false;
    }
    else if (a.masked_addr() == b.masked_addr()) {
	if (a.netmask() <= b.netmask()) {
	    return false;
	}
    }
    return true;
}

/**
 *
 **/
RouteFormat::RouteFormat() :
    _mask(0),
    _mode(BRIEF),
    _exact(0),
    _filter(0)
{
}

/**
 *
 **/
RouteFormat::RouteFormat(const string &hints) : 
    _mask(0), 
    _mode(BRIEF), 
    _exact(0), 
    _filter(0)
{
    //    cout << "incoming hints: " << hints << endl;

    //parser hints here
    unsigned int pos = hints.find(":");
    string mode_str = hints.substr(0, pos);
    unsigned long mode = strtoul(mode_str.c_str(), NULL, 10);
    _mode = mode & DETAIL;
    //skip brief here
    //    cout << "mode: " << _mode << endl;
    
    _exact = mode & EXACT;
    //    cout << "exact: " << _exact << endl;

    _match = mode & MATCH;

    _filter = mode & FILTER_MASK;

    unsigned int end_pos = hints.find(":", pos+1);
    if (end_pos != string::npos) {
	if (pos + 1 != end_pos) {
	    //		cout << "setting prefix (before): " << _prefix.str() << ", " << prefix().str() << endl;
	    string filt = hints.substr(pos+1, end_pos - pos - 1);
	    if (_filter == PREFIX_FILTER) {
		if (_match) {
		    _prefix = IPv4Net(filt.c_str(),32);
		}
		else {
		    _prefix = IPv4Net(filt.c_str());
		}
	    }
	    else if (_filter == MASK_FILTER) {
		_mask = strtoul(filt.c_str(), NULL, 10);
	    }
	    else if (_filter == NEXTHOP_FILTER) {
		_nexthop = IPv4(filt.c_str());
	    }
	    else if (_filter == PROTO_FILTER) {
		_protocol = filt;
	    }
	    //		cout << "setting prefix (after): " << _prefix.str() << ", " << prefix().str() << endl;
	}
    }
}

/**
 *
 **/
string
RouteFormat::process(const IPv4Net &route, const RouteData &data) const 
{
    string line = route.str();// + string("\t\t[");
    if (line.length() < 16) {
	line += "\t";
    }
    line += "\t[";

    if (data._protocol_origin == "static" && data._recursive)
	line += string("st/rec(");
    else
	line += data._protocol_origin + string("(");
    char buf[80];
    sprintf(buf, "%d", data._metric);
    line += string(buf) + string(")]\t> to ");

    string temp = data._next_hop.str();
    if (temp.length() < 11) {
	temp += "\t";
    }
    line += temp + string("\tvia ");


    if (data._ifname != data._vifname) { //means that this is a vif
	line += data._ifname + string(".") + data._vifname + string("\n");
    }
    else {
	line += data._ifname + string("\n");
    }
    return line;
}

/**
 *
 **/
template <class T>
RouteContainer<T>::RouteContainer() : 
    _max_output(500),
    _done(true),
    _sock(0)
{
}

/**
 *
 **/
template <class T>
void
RouteContainer<T>::add_route(const IPv4Net &prefix, const T &param, RouteData *data)
{
    _route_coll.insert(pair<RouteKey<T>, RouteData*>(RouteKey<T>(param, prefix), data));
}

/**
 *
 **/
template <class T>
void
RouteContainer<T>::remove_route(const IPv4Net &prefix, const T &param, bool release_data)
{
    RouteIter iter = _route_coll.find(RouteKey<T>(param, prefix));
    if (iter == _route_coll.end()) {
	return;
    }

    if (release_data == true) {
	if (iter->second != NULL) {
	    delete iter->second;
	}
    }

    _route_coll.erase(iter);
}

/**
 *
 **/
template <class T>
RouteData*
RouteContainer<T>::get(const IPv4Net &prefix, const T &param)
{
    RouteIter iter = _route_coll.find(RouteKey<T>(param, prefix));
    if (iter != _route_coll.end()) {
	return iter->second;
    }
    return NULL;
}

/**
 *
 **/
template <class T>
bool
RouteContainer<T>::send_data()
{
    if (_done) {
	return _done;
    }

    D("RouteContainer<T>::send_data(), processing\n");

    RouteIter start_iter = _route_coll.lower_bound(_start_key);
    RouteIter end_iter = _route_coll.upper_bound(_end_key);


    int ct = 0;


    if (_format.exact()) {
	D("RouteContainer<T>::send_data(), exact\n");
	_end_key = _start_key; //ensure they are equivalent
	if (start_iter != _route_coll.end()) {
	    if (start_iter->first._route == _format.prefix()) {
		string line = _format.process(start_iter->first._route, *(start_iter->second));
		int num_bytes = send(_sock, line.c_str(), line.length(), MSG_NOSIGNAL);
		if (num_bytes < 0) {
		    D("RouteContainer<T>::send_data() failure on send\n");
		    start_iter = _route_coll.end(); //terminate iteration at this point
		}
	    }
	    start_iter = _route_coll.end(); //doesn't find exact entry
	}
    }
    else {
	while ((start_iter != _route_coll.end()) && (start_iter != end_iter) && (++ct % _max_output) != 0) {
	    string line = _format.process(start_iter->first._route, *(start_iter->second));
	    int num_bytes = send(_sock, line.c_str(), line.length(), MSG_NOSIGNAL);
	    if (num_bytes < 0) {
		D("RouteContainer<T>::send_data() failure on send\n");
		start_iter = _route_coll.end(); //terminate iteration at this point
		break;
	    }
	    start_iter++;
	}
    }


    if (start_iter == _route_coll.end() || start_iter == end_iter) {
	D("RouteContainer<T>::send_data(), finished processing\n");
	//funky, but should really be responsible for this in the routecontainer
	close(_sock);
	_sock = 0;
	_done = true;
    }
    else {
	_start_key = start_iter->first;
    }
    return _done;
}

/**
 *
 **/
template <class T>
void
RouteContainer<T>::send_data(int sock, const RouteKey<T> &start_key, const RouteKey<T> &end_key, const RouteFormat &format)
{
    if (!_done) {
	return; //don't accept command if we are already processing...
    }

    D("RouteContainer<T>::send_data(): start\n");

    _sock = sock;
    _done = false;
    _format = format;
    _start_key = start_key;
    _end_key = end_key;

    char total_size_buf[80], buf[80];
    sprintf(total_size_buf, "%d", _route_coll.size());
    if (!_format.exact()) {
	sprintf(buf, "%d", slice_size(start_key, end_key));
    }
    else {
	int ct = 0;
	RouteIter start_iter = _route_coll.lower_bound(_start_key);
	if (start_iter != _route_coll.end()) {
	    if (start_iter->first._route == _format.prefix()) {
		ct = 1;
	    }
	}
	sprintf(buf, "%d", ct);
    }
    string header = "Routes: " + string(buf) + "/" + total_size_buf + ", Paths: " + string(buf) + "/" + total_size_buf + "\n";
    if (send(sock, header.c_str(), header.length(), 0) < 0) {
	perror("send");
    }
    send_data();
    D("RouteContainer<T>::send_data(): end\n");
    return;
}

/**
 *
 **/
template <class T>
int
RouteContainer<T>::slice_size(const RouteKey<T> &start_key, const RouteKey<T> &end_key)
{
    RouteIter iter = _route_coll.lower_bound(start_key);
    RouteIter end = _route_coll.upper_bound(end_key);
    RouteColl coll;
    coll.insert(iter, end);
    return coll.size();
}

/**
 *
 **/
template <class T>
RouteKey<T>
RouteContainer<T>::match(const IPv4Net &prefix, const T &param)
{
    RouteIter iter = _route_coll.lower_bound(RouteKey<T>(param, prefix));
    RouteIter end = _route_coll.upper_bound(RouteKey<T>(param, prefix));

    //    cout << "::match(): lower: " << iter->first._route.str() << ", upper: " << end->first._route.str() << endl;
    
    IPv4Net matched_prefix;
    //walk through range looking for most specific match...
    while (iter != end) {
	IPv4Net p1(prefix.masked_addr(), iter->first._route.prefix_len());
	//	cout << "::match() comparing: " << p1.str() << ", to " << iter->first._route.str() << endl;
	if (p1 == iter->first._route &&
	    iter->first._route.netmask() > matched_prefix.netmask()) {
	    //	    cout << "::match(): matched: " << iter->first._route.str() << endl;
	    matched_prefix = iter->first._route;
	}
	++iter;
    }
    return matched_prefix;
}

/**
 *
 **/
ShowRoutesProcessor::ShowRoutesProcessor(EventLoop &e, bool test)
    : _e(e),
      _rtr(NULL),
      _request_sock(0),
      _test(test)
{
}

/**
 *
 **/
ShowRoutesProcessor::~ShowRoutesProcessor()
{
    if (_rtr != NULL) {
	delete _rtr;
	_rtr = NULL;
    }

    if (_request_sock) {
	close(_request_sock);
    }
}

/**
 *
 **/
bool
ShowRoutesProcessor::startup()
{
    if (status() != SERVICE_READY) {
	return false;
    }

    // Create XrlRouter
    string process = c_format("show_routes<%d>", getpid());
    _rtr = new XrlStdRouter(_e, process.c_str(),
			    FinderConstants::FINDER_DEFAULT_HOST().str().c_str(),
			    FinderConstants::FINDER_DEFAULT_PORT());

    // Glue the router to the Xrl methods class exports
    this->set_command_map(_rtr);

    // Register methods with Finder
    _rtr->finalize();

    // Start timer to poll whether XrlRouter becomes ready or fails so
    // we can continue processing.
    _t_service = _e.new_periodic_ms(250,
				    callback(this,
					     &ShowRoutesProcessor::poll_ready_failed));
    
    init_command_request();
    _t_pipe = _e.new_periodic_ms(250,
				 callback(this,
					  &ShowRoutesProcessor::poll_command_request));
    
    if (_test) {
	_t_test = _e.new_periodic_ms(1000,
				     callback(this,
					      &ShowRoutesProcessor::load_test_rib));
    }

    set_status(SERVICE_STARTING);
    return true;
}

/**
 *
 **/
bool
ShowRoutesProcessor::shutdown()
{
    // Withdraw the Xrl methods
    this->set_command_map(NULL);

    ServiceStatus st = this->status();
    if (st == SERVICE_FAILED
	|| st == SERVICE_SHUTTING_DOWN
	|| st == SERVICE_SHUTDOWN)
	return false;

    set_status(SERVICE_SHUTTING_DOWN);
    request_cease();
    return true;
}


/**
 *
 **/
void
ShowRoutesProcessor::init_command_request()
{
    //    cout << "poll_pipe->entering" << endl;
    int len;
    struct sockaddr_un local;

    if ((_request_sock = socket(AF_UNIX, SOCK_STREAM, 0)) == -1) {
	perror("socket");
	exit(1);
    }
    
    fcntl(_request_sock, F_SETFL, O_NONBLOCK);
    
    local.sun_family = AF_UNIX;
    strcpy(local.sun_path, SOCK_PATH);
    unlink(local.sun_path);
    len = strlen(local.sun_path) + sizeof(local.sun_family);
    if (bind(_request_sock, (struct sockaddr *)&local, len) == -1) {
	perror("bind");
	exit(1);
    }
    
    if (listen(_request_sock, 5) == -1) {
	perror("listen");
	exit(1);
    }

    //now change permissions on sock
    chmod(SOCK_PATH, S_IRWXO | S_IRWXG | S_IRWXU);
    
}

/**
 *
 **/
bool
ShowRoutesProcessor::poll_command_request()
{
    if (process_request()) {
	int s, t;
	struct sockaddr_un remote;
	char str[256];
	
	t = sizeof(remote);
	if ((s = accept(_request_sock, (struct sockaddr *)&remote, (socklen_t*)&t)) > 0) {
	    D("ShowRoutesProcessor::poll_command_request(), received new request\n");
	    int n = recv(s, str, 256, 0);
	    if (n > 0) {
		RouteFormat format(str);
		return process_request(format, s);
	    }
	}
    }
    return true;
}

/**
 *
 **/
bool
ShowRoutesProcessor::process_request()
{
    bool status = false;
    status  = _route_coll.send_data();
    status |= _mask_coll.send_data();
    status |= _nexthop_coll.send_data();
    status |= _proto_bgp_coll.send_data();
    status |= _proto_coll.send_data();
    return status;
}

/**
 *
 **/
bool
ShowRoutesProcessor::process_request(RouteFormat &format, int sock)
{
    if (format.filter() == PREFIX_FILTER) {
	D("ShowRoutesProcessor::process_request(): PREFIX_FILTER\n");
	if (format.match()) {
	    RouteKey<int> match_key = _route_coll.match(format.prefix(), -1); //-1 is ignore mask in match
	    _route_coll.send_data(sock, match_key, match_key, format);
	}
	else {
	    RouteKey<int> start_key(0, format.prefix());
	    IPv4Net temp(format.prefix().top_addr(), 32);
	    RouteKey<int> end_key(0, temp);
	    _route_coll.send_data(sock, start_key, end_key, format);
	}
    }
    else if (format.filter() == MASK_FILTER) {
	D("ShowRoutesProcessor::process_request(): MASK_FILTER\n");
	//		cout << "mask-filter" << endl;
	IPv4Net temp1("0.0.0.0", 0);
	RouteKey<int> start_key(format.mask(), temp1);
	IPv4Net temp2("255.255.255.255", 32);
	RouteKey<int> end_key(format.mask(), temp2);
	
	_mask_coll.send_data(sock, start_key, end_key, format);
    }
    else if (format.filter() == NEXTHOP_FILTER) {
	D("ShowRoutesProcessor::process_request(): NEXTHOP_FILTER\n");
	//		cout << "nexthop filter" << endl;
	IPv4Net temp1("0.0.0.0", 0);
	RouteKey<IPv4> start_key(format.nexthop(), temp1);
	IPv4Net temp2("255.255.255.255", 32);
	RouteKey<IPv4> end_key(format.nexthop(), temp2);
	
	_nexthop_coll.send_data(sock, start_key, end_key, format);
    }
    else if (format.filter() == PROTO_FILTER) {
	D("ShowRoutesProcessor::process_request(): PROTO_FILTER\n");
	if (format.protocol() == "bgp") {
	    //		    cout << "proto filter (bgp)" << endl;
	    IPv4Net temp1("0.0.0.0", 0);
	    RouteKey<ProtocolBGP> start_key(ProtocolBGP(format.protocol()), temp1);
	    IPv4Net temp2("255.255.255.255", 32);
	    RouteKey<ProtocolBGP> end_key(ProtocolBGP(format.protocol()), temp2);
	    
	    _proto_bgp_coll.send_data(sock, start_key, end_key, format);
	}
	else {
	    //		    cout << "proto filter" << endl;
	    IPv4Net temp1("0.0.0.0", 0);
	    RouteKey<Protocol> start_key(Protocol(format.protocol()), temp1);
	    IPv4Net temp2("255.255.255.255", 32);
	    RouteKey<Protocol> end_key(Protocol(format.protocol()), temp2);
	    
	    _proto_coll.send_data(sock, start_key, end_key, format);
	}
    }
    //    close(s2);
    return true;
}

/**
 *
 **/
bool
ShowRoutesProcessor::poll_ready_failed()
{
    if (_rtr == 0) {
	return false;
    } else if (_rtr->ready()) {
	set_status(SERVICE_RUNNING);
	watch_rib();
	return false;
    } else if (_rtr->failed()) {
	set_status(SERVICE_FAILED, "Failed: No Finder?");
	return false;
    }
    return true;
}

/**
 *
 **/
void
ShowRoutesProcessor::watch_rib()
{
    XrlFinderEventNotifierV0p1Client fen(_rtr);
    
    if (fen.send_register_class_event_interest(
					       "finder", _rtr->instance_name(), "RIB",
					       callback(this, &ShowRoutesProcessor::watch_rib_cb)) == false) {
	set_status(SERVICE_FAILED,
		   c_format("Failed to register interest in RIB"));
		   return;
	    }
}
    
/**
 *
 **/
void
ShowRoutesProcessor::watch_rib_cb(const XrlError& xe)
{
    if (xe == XrlError::OKAY()) {
	request_redist();
	return;
    }
    set_status(SERVICE_FAILED,
	       c_format("Failed to register interest in RIB"));
    return;
}

/**
 *
 **/
inline string
rib_table_name(const string& protocol, bool ribin)
{
    if (protocol == "all")
	return protocol;
    // "all" is a special name in the RIB for the rib out table.
    // "all-<proto>" adds  redist with a filter for <proto>
    return c_format("%s%s", (ribin ? "" : "all-"), protocol.c_str());
}

/**
 *
 **/
void
ShowRoutesProcessor::request_redist()
{
    string protocol = rib_table_name("all", false);

    XrlRibV0p1Client::RedistEnable4CB cb;
    cb = callback(this, &ShowRoutesProcessor::request_redist_cb);

    XrlRibV0p1Client rib(_rtr);

    bool sent = false;
    IPv4Net network_prefix4(IPv4::ZERO(), 0);	// XXX: get the whole table
    sent = rib.send_redist_enable4("rib",
				   _rtr->instance_name(),
				   protocol, true, false,
				   network_prefix4,
				   _cookie, cb);
    if (sent == false) {
	set_status(SERVICE_FAILED, "Failed to request redist.");
    }
}

/**
 *
 **/
void
ShowRoutesProcessor::request_redist_cb(const XrlError& xe)
{
    if (xe == XrlError::OKAY()) {
	return;
    }
    set_status(SERVICE_FAILED,
	       c_format("Request for routes to be redistributed from %s "
			"failed.\nThe protocol is probably not active.",
			"all"));
    return;
}

/**
 *
 **/
void
ShowRoutesProcessor::request_cease()
{
    string proto = rib_table_name("all", false);

    XrlRibV0p1Client::RedistDisable4CB cb;
    cb = callback(this, &ShowRoutesProcessor::request_cease_cb);

    XrlRibV0p1Client rib(_rtr);

    bool sent = false;
    sent = rib.send_redist_disable4("rib",
				    _rtr->instance_name(),
				    proto, true, false,
				    _cookie, cb);
    if (sent == false) {
	set_status(SERVICE_FAILED, "Failed to request redistribution end.");
	return;
    }
    set_status(SERVICE_SHUTTING_DOWN);
}

/**
 *
 **/
void
ShowRoutesProcessor::request_cease_cb(const XrlError& /* xe */)
{
    set_status(SERVICE_SHUTDOWN);
    return;
}


/**
 *
 **/
bool
ShowRoutesProcessor::check_cookie(const string& cookie)
{
    if (cookie != _cookie) {
	cerr << "Bug detected cookie mismatch \"" << cookie << "\" != \""
	     << _cookie << "\"" << endl;
    }
    return true;
}

/**
 *
 **/
XrlCmdError
ShowRoutesProcessor::common_0_1_get_target_name(string& name)
{
    name = this->name();
    return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
ShowRoutesProcessor::common_0_1_get_version(string& version)
{
    version = this->version();
    return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
ShowRoutesProcessor::common_0_1_get_status(uint32_t&	status,
					   string&	/* reason */)
{
    switch (this->status()) {
    case SERVICE_READY:		status = PROC_NULL;		break;
    case SERVICE_STARTING:	status = PROC_STARTUP;		break;
    case SERVICE_RUNNING:	status = PROC_READY;		break;
    case SERVICE_PAUSED:					/* FALLTHRU */
    case SERVICE_PAUSING:					/* FALLTHRU */
    case SERVICE_RESUMING:	status = PROC_NOT_READY;	break;
    case SERVICE_SHUTTING_DOWN:	status = PROC_SHUTDOWN;		break;
    case SERVICE_SHUTDOWN:	status = PROC_DONE;		break;
    case SERVICE_FAILED:	status = PROC_FAILED;		break;
    case SERVICE_ALL:						break;
    }
    return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
ShowRoutesProcessor::common_0_1_shutdown()
{
    this->shutdown();
    return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
ShowRoutesProcessor::finder_event_observer_0_1_xrl_target_birth(
						const string& /* cls */,
						const string& /* ins */
						)
{
    return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
ShowRoutesProcessor::finder_event_observer_0_1_xrl_target_death(
						const string& cls,
						const string& /* ins */)
{
    if (cls == "rib")
	this->shutdown();
    return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
ShowRoutesProcessor::redist4_0_1_starting_route_dump(const string& cookie)
{
    _cookie = cookie;
    return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
ShowRoutesProcessor::redist4_0_1_finishing_route_dump(const string& cookie)
{
    check_cookie(cookie);

    //    this->shutdown();
    return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
ShowRoutesProcessor::redist4_0_1_add_route(const IPv4Net&	dst,
					   const IPv4&		nexthop,
					   const string&	ifname,
					   const string&	vifname,
					   const uint32_t&	metric,
					   const uint32_t&	admin_distance,
					   const string&	cookie,
					   const string&	protocol_origin,
					   const bool&          recursive)
{
    //    cout << "adding route: " << dst.str() << endl;

    if (this->status() != SERVICE_RUNNING || check_cookie(cookie) == false) {
	return XrlCmdError::OKAY();
    }

    try {
	RouteData *rd = new RouteData(nexthop, ifname, vifname, metric, admin_distance, cookie, protocol_origin, recursive);
	
	_route_coll.add_route(dst, 0, rd);
	_proto_coll.add_route(dst, Protocol(protocol_origin), rd);
	_proto_bgp_coll.add_route(dst, ProtocolBGP(protocol_origin), rd);
	
	_nexthop_coll.add_route(dst, nexthop, rd);
	_mask_coll.add_route(dst, dst.prefix_len(), rd);
    }
    catch (...) {
	printf("ShowRoutesProcess::redist4_0_1_add_route() exception thrown in display rib insertion\n");
    }

    return XrlCmdError::OKAY();
}

/**
 *
 **/

//XrlCmdError
//ShowRoutesProcessor::redist4_0_1_delete_route(const IPv4Net&	dst,
//					      const string&	/* cookie  */,
//					      const string&	/* protocol_origin */)
XrlCmdError 
ShowRoutesProcessor::redist4_0_1_delete_route(
						 // Input values,
						 const IPv4Net&	dst,
						 const IPv4&	nexthop,
						 const string&	ifname,
						 const string&	vifname,
						 const uint32_t&	metric,
						 const uint32_t&	admin_distance,
						 const string&	cookie,
						 const string&	protocol_origin,
						 const bool&    recursive)
{
    UNUSED(ifname);
    UNUSED(vifname);
    UNUSED(metric);
    UNUSED(cookie);
    UNUSED(protocol_origin);
    UNUSED(nexthop);
    UNUSED(admin_distance);
    UNUSED(recursive);
    //    cout << "removing route: " << dst.str() << endl;
    try {
	//search in the _route_coll--as this will be fast, then use the other data to remove the other guys
	RouteData *data = _route_coll.get(dst, 0);
	if (data != NULL) {
	    //	    cout << "delete_route--route found! " << data->_protocol_origin << endl;
	    _proto_coll.remove_route(dst, data->_protocol_origin);
	    _proto_bgp_coll.remove_route(dst, data->_protocol_origin);
	    _nexthop_coll.remove_route(dst, data->_next_hop);
	    _mask_coll.remove_route(dst, dst.prefix_len());
	    _route_coll.remove_route(dst, 0, true); //also releases memory
	}

    }
    catch (...) {
	printf("ShowRoutesProcess::redist4_0_1_delete_route() exception thrown in display rib deletion\n");
    } 
   return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
ShowRoutesProcessor::redist6_0_1_starting_route_dump(const string& cookie)
{
    check_cookie(cookie);
    return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
ShowRoutesProcessor::redist6_0_1_finishing_route_dump(const string& cookie)
{
    check_cookie(cookie);
    //    this->shutdown();
    return XrlCmdError::OKAY();
}

/**
 *
 **/
XrlCmdError
ShowRoutesProcessor::redist6_0_1_add_route(const IPv6Net&	dst,
					   const IPv6&		nexthop,
					   const string&	ifname,
					   const string&	vifname,
					   const uint32_t&	metric,
					   const uint32_t&	admin_distance,
					   const string&	cookie,
					   const string&	protocol_origin,
					   const bool&	        recursive)
{
    return XrlCmdError::OKAY();
    UNUSED(dst);
    UNUSED(nexthop);
    UNUSED(ifname);
    UNUSED(vifname);
    UNUSED(metric);
    UNUSED(admin_distance);
    UNUSED(protocol_origin);
    UNUSED(cookie);
    UNUSED(recursive);
}

/**
 *
 **/
XrlCmdError
ShowRoutesProcessor::redist6_0_1_delete_route(
					      // Input values,
					      const IPv6Net&,
					      const IPv6&,
					      const string&,
					      const string&,
					      const uint32_t&,
					      const uint32_t&,
					      const string&,
					      const string&,
					      const bool&)

{
    // XXX For now we ignore deletions that occur during route dump
    return XrlCmdError::OKAY();
}

/**
 *
 **/
bool
ShowRoutesProcessor::load_test_rib()
{
    D("Loading test rib...\n");
    int val = int(255. * (float(rand()) / float(RAND_MAX)));

    for (int i = 0; i < 254; ++i) {
	for (int j = 0; j < 32; ++j) {
	    char ip[80];
	    sprintf(ip, "%d.1.%d.1", val, i);
	    IPv4Net prefix(ip, j);

	    IPv4 nexthop("192.168.0.1");
	    string ifname("eth0");
	    string vifname("eth0");
	    uint32_t metric(1);
	    uint32_t admin_distance(1);
	    string cookie("cookie");
	    string protocol_origin("ebgp");
	    bool recursive(false);
	    RouteData *rd = new RouteData(nexthop, ifname, vifname, metric, admin_distance, cookie, protocol_origin, recursive);

	    _route_coll.add_route(prefix, 0, rd);
	    _proto_coll.add_route(prefix, Protocol(protocol_origin), rd);
	    _proto_bgp_coll.add_route(prefix, ProtocolBGP(protocol_origin), rd);
	    
	    _nexthop_coll.add_route(prefix, nexthop, rd);
	    _mask_coll.add_route(prefix, prefix.prefix_len(), rd);
	}
    }
    return true;
}

/**
 *
 **/
int
main(int argc, char* argv[])
 {
    //
    // Initialize and start xlog
    //
    xlog_init(argv[0], NULL);
    xlog_set_verbose(XLOG_VERBOSE_LOW);         // Least verbose messages
    xlog_level_set_verbose(XLOG_LEVEL_ERROR, XLOG_VERBOSE_HIGH);
    xlog_add_default_output();
    xlog_start();
    bool test = false;

    D("Starting route_display_server\n");

    //    cout << "Starting up route_display_server\n" << endl;
    int ch;
    while ((ch = getopt(argc, argv, "t")) != -1) {
	switch (ch) {
	case 't':
	    test = true;
	    break;
	}
    }

    try {
	EventLoop& e = *EventLoopFactory::instance().create(eventloop_st);
	ShowRoutesProcessor srp(e, test);
	
	srp.startup();
	
	//	cout << "service is now running..." << endl;
	while ((srp.status() != SERVICE_FAILED)
	       && (srp.status() != SERVICE_SHUTDOWN)) {
	    e.run();
	}

	D("Shutting down: route_display_server\n");

	if (srp.status() == SERVICE_FAILED) {
	    srp.shutdown();
	    if (srp.status_note().empty() == false) {
		cout << srp.status_note() << endl;
	    } else {
		cout << "Failed" << endl;
	    }
	    cout << endl;
	}

    } catch (...) {
	D("Main exception caught\n");
	xorp_print_standard_exceptions();
    }


    //    cout << "exiting!" << endl;

    //
    // Gracefully stop and exit xlog
    //
    xlog_stop();
    xlog_exit();

    return 0;
    UNUSED(argc);
}


