// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
// vim:set sts=4 ts=8:

// Copyright (c) 2001-2007 International Computer Science Institute
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software")
// to deal in the Software without restriction, subject to the conditions
// listed in the XORP LICENSE file. These conditions include: you must
// preserve this copyright notice, and you cannot mention the copyright
// holders in advertising related to the Software without their permission.
// The Software is provided WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED. This
// notice is a summary of the XORP LICENSE file; the license in that file is
// legally binding.

#ident "$XORP$"

#include <sys/sysinfo.h>

#include "libxorp_module.h"

#include "libxorp/xorp.h"
#include "libxorp/xlog.h"
#include "libxorp/debug.h"

#include "eventloop_st.hh"


//
// Number of EventLoopST instances.
//
int libxorp_simple_eventloop_instances;

EventLoopST::EventLoopST()
    : _clock(new SystemClock), _timer_list(_clock),
      _selector_list(_clock)
{
    //XLOG_ASSERT(!libxorp_simple_eventloop_instances);
    libxorp_simple_eventloop_instances++;
    last_ev_run = last_warned = 0;
}

EventLoopST::~EventLoopST()
{
    --libxorp_simple_eventloop_instances;
    //XLOG_ASSERT(!libxorp_simple_eventloop_instances);
    delete _clock;
    _clock = NULL;
}

void
EventLoopST::run()
{
    const time_t MAX_ALLOWED = 2;

    if (last_ev_run) {
	struct sysinfo si;
	sysinfo(&si);
	time_t now = si.uptime;
	/*
	time_t now = time(0);
	*/
	time_t diff = now - last_ev_run;

	if (now - last_warned > 0 && (diff > MAX_ALLOWED)) {
	    XLOG_WARNING("%d seconds between calls to EventLoopST::run", (int)diff);
	    last_warned = now;
	}
    }

    TimeVal t;

    _timer_list.advance_time();
    _timer_list.get_next_delay(t);

    int timer_priority = XorpTask::PRIORITY_INFINITY;
    int selector_priority = XorpTask::PRIORITY_INFINITY;
    int task_priority = XorpTask::PRIORITY_INFINITY;

    if (t == TimeVal::ZERO()) {
	timer_priority = _timer_list.get_expired_priority();
    }

    if (_selector_list.ready()) {
	selector_priority = _selector_list.get_ready_priority();
    }

    if (!_task_list.empty()) {
	task_priority = _task_list.get_runnable_priority();
    }

    debug_msg("Priorities: timer = %d selector = %d task = %d\n",
	      timer_priority, selector_priority, task_priority);

    if ( (timer_priority != XorpTask::PRIORITY_INFINITY)
	 && (timer_priority <= selector_priority)
	 && (timer_priority <= task_priority)) {

	// the most important thing to run next is a timer
	_timer_list.run();

    } else if ( (selector_priority != XorpTask::PRIORITY_INFINITY)
		&& (selector_priority <= task_priority) ) {

	// the most important thing to run next is a selector
	_selector_list.wait_and_dispatch(&t);

    } else if (task_priority != XorpTask::PRIORITY_INFINITY) {

	// the most important thing to run next is a task
	_task_list.run();

    } else {
	// there's nothing immediate to run, so go to sleep until the
	// next selector or timer goes off
	_selector_list.wait_and_dispatch(&t);
    }

    struct sysinfo si;
    sysinfo(&si);
    /*
    last_ev_run = time(0);
    */
    last_ev_run = si.uptime;
}

XorpTask
EventLoopST::new_oneoff_task(const OneoffTaskCallback& cb, int priority,
			   int weight)
{
    return _task_list.new_oneoff_task(cb, priority, weight);
}

XorpTask
EventLoopST::new_task(const RepeatedTaskCallback& cb, int priority, int weight)
{
    return _task_list.new_task(cb, priority, weight);
}
