/*
 *  Copyright 2006, Vyatta, Inc.
 *
 *  GNU General Public License
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 *  02110-1301 USA
 *
 *  Module:       vrrp_set.cc
 *
 *  Author(s):    Michael Larson
 *  Date:         2006
 *  Description:  Class used to interface with rtrmgr and fea
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xrl_module.h"
#include "libxorp/xorp.h"
#include "libxorp/xlog.h"
#include "libxorp/eventloop_factory.hh"
#include "routelogics.hh"

#include "xrl_std_router.hh"
#include "xrl_args.hh"
#include "xrl_parser_input.hh"

#include "librl_common/rl_command.hh"

static const char* ROUTER_NAME = "call_xrl";

static int wait_time = 6000;	// Time to wait for the callback in ms.
static int retry_count = 0;	// Number of times to resend xrl on error.

static string g_tid;

EventLoop& g_e = *EventLoopFactory::instance().create(eventloop_st);
XrlStdRouter *g_router = NULL;


enum {
    // Return values from call_xrl
    OK = 0, BADXRL = -1, NOCALLBACK = -2
};


/**
 *
 **/
static void
response_handler(const XrlError& e,
		 XrlArgs*	 response,
		 bool*		 done_flag,
		 bool* 		 resolve_failed,
		 Xrl*		 xrl)
{
    if (e == XrlError::RESOLVE_FAILED()) {
	XLOG_ERROR("Failed.  Reason: %s (\"%s\")",
		   e.str().c_str(), xrl->str().c_str());
	*resolve_failed = true;
	return;
    }
    if (e != XrlError::OKAY()) {
	XLOG_ERROR("Failed.  Reason: %s (\"%s\")",
		   e.str().c_str(), xrl->str().c_str());
	// The code use to exit at this point, but rather
	// than siliently die it's better that the daemon
	// trys to continue on.  Bug 2325
    }
    if (!response->str().empty()) {
	//	printf("%s\n", response->str().c_str());
	/*
	  now grab the transaction if we have a match....
	*/
	string match("tid:u32=");
	if (response->str().find(match) != string::npos) {
	    g_tid = response->str().substr(match.length(), response->str().length()-match.length());
	}

	fflush(stdout);
    }
    *done_flag = true;
}

/**
 *
 **/
int
call_xrl(EventLoop& e, XrlRouter& router, const char* request, bool resp = false)
{
    resp = resp;

    XLOG_INFO("vrrp::call_xrl(): %s", request);

    try {
	Xrl x(request);

	int tries;
	bool done, resolve_failed;

	tries = 0;
	done = false;
	resolve_failed = true;

	while (done == false && tries <= retry_count) {
	    resolve_failed = false;
	    router.send(x, callback(&response_handler,
				    &done,
				    &resolve_failed,
				    &x));
	    bool timed_out = false;
	    XorpTimer timeout = e.set_flag_after_ms(wait_time, &timed_out);
	    while (timed_out == false && done == false) {
		// NB we don't test for resolve failed here because if
		// resolved failed we want to wait before retrying.
		e.run();
	    }
	    tries++;

	    if (resolve_failed) {
		continue;
	    }
	    
	    if (timed_out) {
		XLOG_WARNING("request: %s no response waited %d ms", request,
			     wait_time);
		continue;
	    }

	    if (router.connected() == false) {
		XLOG_FATAL("Lost connection to finder\n");
	    }
	}

	if (resolve_failed) {
	    XLOG_WARNING("request: %s resolve failed", request);
	}
	
 	if (false == done && true == resolve_failed)
	    XLOG_WARNING("request: %s failed after %d retries",
			 request, retry_count);
	return done == true ? OK : NOCALLBACK;
    } catch(const InvalidString& s) {
	cerr << s.str() << endl;
	return BADXRL;
    }
}

/**
 *
 **/
int
set_mac(char *interface,
	char *mac)
{
    char sbuf[80];
    print_mac_address((unsigned char*)mac, sbuf);

    RL_TRACE
    string cmd = "finder://fea/ifmgr/0.1/start_transaction";
    int err = call_xrl(g_e, *g_router, cmd.c_str(), true);
    if (err) {
	XLOG_ERROR("set_mac failed on start_transaction");
	return 1;
    }
    cmd = "finder://fea/ifmgr/0.1/set_mac?tid:u32=" + g_tid + "&ifname:txt=" + interface + "&mac:mac=" + string(sbuf);
    err = call_xrl(g_e, *g_router, cmd.c_str());
    if (err) {
	XLOG_ERROR("set_mac failed on set_mac: %s", sbuf);
	return 1;
    }
    cmd = "finder://fea/ifmgr/0.1/commit_transaction?tid:u32=" + g_tid;
    err = call_xrl(g_e, *g_router, cmd.c_str());
    if (err) {
	XLOG_ERROR("set_mac failed on commit_transaction");
	return 1;
    }
    return 0;
}


/**
 *
 **/
int
delete_mac(char *interface)
{
    RL_TRACE
    string cmd = "finder://fea/ifmgr/0.1/start_transaction";
    int err = call_xrl(g_e, *g_router, cmd.c_str(), true);
    if (err) {
	XLOG_ERROR("delete_mac failed on start_transaction");
	return 1;
    }
    cmd = "finder://fea/ifmgr/0.1/delete_mac?tid:u32=" + g_tid + "&ifname:txt=" + interface;
    err = call_xrl(g_e, *g_router, cmd.c_str());
    if (err) {
	XLOG_ERROR("delete_mac failed on delete_mac");
	return 1;
    }
    cmd = "finder://fea/ifmgr/0.1/commit_transaction?tid:u32=" + g_tid;
    err = call_xrl(g_e, *g_router, cmd.c_str());
    if (err) {
	XLOG_ERROR("delete_mac failed on commit_transaction");
	return 1;
    }
    return 0;
}


/**
 *
 **/
int
set_address(char *interface,
	    char *address,
	    int mask_length)
{
    RL_TRACE
    string cmd = "finder://fea/ifmgr/0.1/start_transaction";
    int err = call_xrl(g_e, *g_router, cmd.c_str(), true);
    if (err) {
	XLOG_ERROR("set_address failed on start_transaction");
	return 1;
    }

    cmd = "finder://fea/ifmgr/0.1/create_address4?tid:u32=" + g_tid + "&ifname:txt=" + interface + "&vif:txt=" + interface + "&address:ipv4=" + address;
    err = call_xrl(g_e, *g_router, cmd.c_str());
    if (err) {
	XLOG_ERROR("set_address failed on create_address");
	return 1;
    }
    
    char buf[40];
    sprintf(buf, "%d", mask_length);
    cmd = "finder://fea/ifmgr/0.1/set_prefix4?tid:u32=" + g_tid + "&ifname:txt=" + interface + "&vif:txt=" + interface + "&address:ipv4=" + address + "&prefix_len:u32=" + string(buf);
    err = call_xrl(g_e, *g_router, cmd.c_str());
    if (err) {
	XLOG_ERROR("set_address failed on set_prefix");
	return 1;
    }

    cmd = "finder://fea/ifmgr/0.1/set_vif_enabled?tid:u32=" + g_tid + "&ifname:txt=" + interface + "&vif:txt=" + interface + "&enabled:bool=true";
    err = call_xrl(g_e, *g_router, cmd.c_str());
    if (err) {
	XLOG_ERROR("set_address failed on set_vif_enabled");
	return 1;
    }
    cmd = "finder://fea/ifmgr/0.1/set_address_enabled4?tid:u32=" + g_tid + "&ifname:txt=" + interface + "&vif:txt=" + interface + "&address:ipv4=" + address + "&enabled:bool=true";
    err = call_xrl(g_e, *g_router, cmd.c_str());
    if (err) {
	XLOG_ERROR("set_address failed on set_address_enabled");
	return 1;
    }
    cmd = "finder://fea/ifmgr/0.1/commit_transaction?tid:u32=" + g_tid;
    err = call_xrl(g_e, *g_router, cmd.c_str());
    if (err) {
	XLOG_ERROR("set_address failed on commit_transaction");
	return 1;
    }
    return 0;
}


/**
 *
 **/
int
delete_address(char *interface,
	       char *address)
{
    RL_TRACE
    string cmd = "finder://fea/ifmgr/0.1/start_transaction";
    int err = call_xrl(g_e, *g_router, cmd.c_str(), true);
    if (err) {
	XLOG_ERROR("delete_address failed on start_transaction");
	return 1;
    }

    cmd = "finder://fea/ifmgr/0.1/delete_address4?tid:u32=" + g_tid + "&ifname:txt=" + interface + "&vif:txt=" + interface + "&address:ipv4=" + address;
    err = call_xrl(g_e, *g_router, cmd.c_str());
    if (err) {
	XLOG_ERROR("delete_address failed on delete_address");
	return 1;
    }

    cmd = "finder://fea/ifmgr/0.1/commit_transaction?tid:u32=" + g_tid;
    err = call_xrl(g_e, *g_router, cmd.c_str());
    if (err) {
	XLOG_ERROR("delete_address failed on commit_transaction");
	return 1;
    }
    return 0;
}


/**
 *
 **/
void vrrp_init()
{
    XorpUnexpectedHandler x(xorp_unexpected_handler);
    //
    // Initialize and start xlog
    //

    xlog_init("vrrpd", NULL);
    xlog_set_verbose(XLOG_VERBOSE_LOW);		// Least verbose messages
    // XXX: verbosity of the error messages temporary increased
    xlog_level_set_verbose(XLOG_LEVEL_ERROR, XLOG_VERBOSE_HIGH);
    xlog_add_default_output();
    xlog_start();

    string finder_host = FinderConstants::FINDER_DEFAULT_HOST().str();
    uint16_t port = FinderConstants::FINDER_DEFAULT_PORT();
    try {
	g_router = new XrlStdRouter(g_e, ROUTER_NAME, finder_host.c_str(), port);

	g_router->finalize();

	while (false == g_router->failed() && false == g_router->ready()) {
	    g_e.run();
	}

	if (true == g_router->failed()) {
	    XLOG_ERROR("Router failed to communicate with finder.\n");
	    exit(-1);
	}

	if (false == g_router->ready()) {
	    XLOG_ERROR("Connected to finder, but did not become ready.\n");
	    exit(-1);
	}

    } catch(...) {
	xorp_catch_standard_exceptions();
    }
}

/**
 *
 **/
void vrrp_shutdown()
{
    xlog_stop();
    xlog_exit();

    if (g_router) {
	delete g_router;
    }
}


/**
 *
 **/
void vrrp_poll()
{
    g_e.run();
}

