#include <sys/time.h>
#include <string.h>
#include <err.h>
#include <sys/ioctl.h>
#include <sys/param.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <time.h>

#include <sys/sysinfo.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <syslog.h>
#include <unistd.h>
#include <string.h>

#include "config.h"
#include "xrl_module.h"
#include "libxorp/xorp.h"
#include "libxorp/xlog.h"
#include "vrrp_set.hh"


#ifndef SIOCETHTOOL
#define SIOCETHTOOL 0x8946
#endif

#include <linux/types.h>
#include <linux/ethtool.h>
#include <termios.h>
#include <fcntl.h>
#include <netinet/if_ether.h>



#include "routelogics.hh"



struct interface {
  struct interface *next, *prev;
  char name[IFNAMSIZ];
  struct sockaddr addr;
  struct sockaddr dstaddr;
  struct sockaddr netmask;
  struct sockaddr bcastaddr;
  char hwaddr[32];

  int has_ip;
  int has_dst;
};

int fetch_interface(struct interface *ife);

int    g_link_status = 0;
unsigned long g_link_holddown_time = 0;

typedef struct {	/* parameters per interface -- rfc2338.6.1.1 */
	int		auth_type;	/* authentification type. VRRP_AUTH_* */
	unsigned char   auth_data[8];	/* authentification data */

	unsigned long	ipaddr;		/* the address of the interface */
	char		hwaddr[6];	/* WORK: lame hardcoded for ethernet !!!! */
	char		*ifname;	/* the device name for this ipaddr */
} vrrp_if;

typedef struct {
	unsigned long	addr;		/* the ip address */
	int		deletable;	/* TRUE if one of my primary addr */
} vip_addr;

typedef struct {	/* parameters per virtual router -- rfc2338.6.1.2 */
	int	vrid;		/* virtual id. from 1(!) to 255 */
	int	priority;	/* priority value */
	int	naddr;		/* number of ip addresses */
	vip_addr *vaddr;	/* point on the ip address array */
	int	adver_int;	/* delay between advertisements(in sec) */	

	int	preempt;	/* true if a higher prio preempt a lower one */

	int	state;		/* internal state (init/backup/master) */
	int	wantstate;	/* user explicitly wants a state (back/mast) */

	int	sockfd;		/* the socket descriptor */
	
	int	initF;		/* true if the struct is init */
	
	int	no_vmac;	/* dont handle the virtual MAC --rfc2338.7.3 */

	/* rfc2336.6.2 */
	unsigned long	ms_down_timer;
	unsigned long	adver_timer;

	vrrp_if	vif;
} vrrp_rt;

void
construct_status(vrrp_rt *vrrp_data, int master_ipaddr, int master_pri, char *state_buf, char *vrrp_hwaddr, int *id, char **output_line);



//location for logging statistics file:
const char* g_log_file = "/var/log/vrrpd/vrrpd";
const char* g_xrl_loc = PKGLIBDIR;

int g_fd_write;

int g_net_mask[33];

int g_masklength = -1;

/**
 *
 **/
void rl_syslog(int severity, char* msg)
{
  if (severity == XLOG_LEVEL_FATAL) {
    XLOG_FATAL(msg);
  }
  else if (severity == XLOG_LEVEL_ERROR) {
    XLOG_ERROR(msg);
  }
  else if (severity == XLOG_LEVEL_WARNING) {
    XLOG_WARNING(msg);
  }
  else { // (severity == XLOG_LEVEL_INFO) {
    XLOG_INFO(msg);
  }
  //XLOG_FN(severity, msg);
}


/**
 *
 **/
//void set_xorp_address(char* ifname, unsigned long ip)
int get_interface_masklength(char* ifname, unsigned long ip)
{
  if (g_masklength < 0) {
    g_masklength = find_masklength(ifname, ip);
    if (g_masklength < 0 || g_masklength > 32) {
      g_masklength = 32;
    }
  }
  return g_masklength;
}

/**
 *
 **/
char *print_mac_address(unsigned char mac_addr[6], char *s)
{
  sprintf(s, "%02X:%02X:%02X:%02X:%02X:%02X",
	  mac_addr[0], mac_addr[1], mac_addr[2],
	  mac_addr[3], mac_addr[4], mac_addr[5]
	  );

  return s;
}

/**
 *
 **/
void
rl_log(void *vrrp, int master_ipaddr, int master_pri, char *vrrp_hwaddr)
{
  char file_buf[80];
  FILE *rd_fd = NULL, *wr_fd = NULL;
  char current_state_buf[80], alt_state_buf[80];

  if (vrrp == NULL) {
    return;
  }
  if (vrrp_hwaddr == NULL) {
    return;
  }

  vrrp_rt *vrrp_data = (vrrp_rt*)vrrp;

  if (vrrp_data->state == 3) {
    sprintf(current_state_buf, "master");
    sprintf(alt_state_buf, "backup");
  }
  else if (vrrp_data->state == 2) {
    sprintf(current_state_buf, "backup");
    sprintf(alt_state_buf, "master");
  }

  int id;
  char status_buf[1024], *sbuf = status_buf;

  construct_status(vrrp_data, master_ipaddr, master_pri, current_state_buf, vrrp_hwaddr, &id, &sbuf);
  if (status_buf == NULL) {
    return;
  }

  //log a backup state
  sprintf(file_buf, "%s_%s_%d.log", g_log_file, vrrp_data->vif.ifname, id);

  if (file_get(file_buf, &rd_fd, &wr_fd) == 0) {
    fputs(status_buf, wr_fd);

    fflush(wr_fd);
    file_done(file_buf, rd_fd, wr_fd);
  }
}

/**
 *
 **/
void
rl_log_init()
{
  int i;
  int mask = 0;
  g_net_mask[0] = 0;
  for (i = 0; i < 32; ++i) {
    mask = (mask << 1) + 1;
    g_net_mask[i+1] = mask;
  }

  //cleans up all rl vrrpd log files
  /*
  char buf[80];
  sprintf(buf, "%s*.log*", g_log_file);
  unlink(buf);
  */
}

/**
 *
 **/
int
file_get(const char* file_buf, FILE** rd_fd, FILE** wr_fd)
{
  char lock_file_buf[80];

  if (file_buf == NULL) {
    return 1;
  }

  *rd_fd = *wr_fd = NULL;

  sprintf(lock_file_buf, "%s.lck", file_buf);

  g_fd_write = open(lock_file_buf, O_WRONLY | O_CREAT | O_EXCL, 0644);
  if (g_fd_write < 0 && errno == EEXIST) {
    return 1;
  }
  else if (g_fd_write < 0) {
    return 1;
  }

  *wr_fd = fdopen(g_fd_write, "w");
  if (*wr_fd == NULL) {
    close(g_fd_write);
    unlink(lock_file_buf);
    return 1;
  }

  //we are free to lock this file now
  *rd_fd = fopen(file_buf, "r"); //let's get a file descriptor
  if (*rd_fd == NULL && errno == ENOENT) {
    //Let's create the file here and try again.
    *rd_fd = fopen(file_buf, "w+");
    if (*rd_fd != NULL) {
      fclose(*rd_fd);
    }
    *rd_fd = fopen(file_buf, "r"); //let's get a file descriptor
    if (*rd_fd == NULL) {
      close(g_fd_write);
      unlink(file_buf);
      return 1;
    }
  }
  else if (*rd_fd == NULL) {
    close(g_fd_write);
    unlink(file_buf);
    return 1;
  }
  return 0;
}

/**
 *
 **/
void
file_done(const char *file_buf, FILE *fd_rd, FILE *fd_wr)
{
  char lock_buf[80];

  if (file_buf == NULL ||
      fd_rd == NULL || 
      fd_wr == NULL) {
    return;
  }

  close(g_fd_write);
  fclose(fd_rd);
  fclose(fd_wr);

  sprintf(lock_buf, "%s.lck", file_buf);
  rename(lock_buf, file_buf);
}

/**
 *
 **/
void
rl_log_clear(int vrid)
{
  //actually I don't expect this function to be used at all...
  char buf[80];
  sprintf(buf, "%s%d.log", g_log_file, vrid);
  unlink(buf);
}





/*
 Format is as follows:

 Physical interface: ge-3/1/0, Unit: 0, Address: 192.168.29.254/24
   Interface state: up, Group: 10, State: master
   Priority: 200, Advertisement interval: 3, Authentication type: simple
   Preempt: yes, VIP count: 1, VIP: 192.168.29.55
   Advertisement timer: 2.407s, Master router: 192.168.29.254
   Virtual MAC: 00:00:5e:00:01:0a

   Physical interface: vif
   Unit: ??
   Address: vaddr[0]
   Interface state: state
   Group: vrid
   State: passed in with the call
   Priority: priority
   VIP count: ??
   VIP: ??
   Advertisement timer: adver_timer
   Master router: ?? 
   Virtual MAC ??

*/

void
construct_status(vrrp_rt *vrrp_data, int master_ipaddr, int master_pri, char *state_buf, char *vrrp_hwaddr, int *id, char **output_line)
{
  //  char empty_buf[80] = "[empty]\0";
  char *tmp_buf;
  char addr_buf[80], master_buf[80];

  *id = vrrp_data->vrid;

  /*
   * Now construct the output line.....
   */
  struct in_addr in;
  in.s_addr = htonl(vrrp_data->vif.ipaddr);
  tmp_buf = inet_ntoa(in);
  strcpy(addr_buf, tmp_buf);

  in.s_addr = (vrrp_data->state == 3) ? htonl(vrrp_data->vif.ipaddr) : master_ipaddr;
  tmp_buf = inet_ntoa(in);
  strcpy(master_buf, tmp_buf);

  char internal_state_buf[80];
  if (!get_link_status(vrrp_data->vif.ifname)) {
    //    sprintf(internal_state_buf, "backup");
    sprintf(internal_state_buf, "down");
  }
  else {
    //    sprintf(internal_state_buf, "master");
    sprintf(internal_state_buf, "up");
  }
      /*  else { //  vrrp_data->state == 1
    sprintf(internal_state_buf, "init");
  }
      */
  char auth_buf[80];
  if (vrrp_data->vif.auth_type == 0/*VRRP_AUTH_NONE*/) {
    sprintf(auth_buf, "none");
  }
  else if (vrrp_data->vif.auth_type == 1/*VRRP_AUTH_PASS*/) {
    sprintf(auth_buf, "simple");
  }
  else if (vrrp_data->vif.auth_type == 2/*VRRP_AUTH_AH*/) {
    sprintf(auth_buf, "md5");
  }

  char vip_buf[80];
  int i;
  for (i = 0; i < vrrp_data->naddr; ++i) {
    in.s_addr = htonl(vrrp_data->vaddr[i].addr);
    tmp_buf = inet_ntoa(in);
    char tmp_buf2[80];
    sprintf(tmp_buf2, "%s, ", tmp_buf);
    strcpy(vip_buf, tmp_buf);
  }

  if (vrrp_data->state == 2) {
    /* backup */
    sprintf(*output_line, 
	    "Physical interface: %s, Address: %s\n" \
	    "  Interface state: %s, Group: %d, State: %s\n" \
	    "  Priority: %d, Advertisement interval: %ds, Authentication type: %s\n" \
	    "  Preempt: %s, VIP count: %d, VIP: %s\n" \
	    "  Dead timer: %lus, Master priority: %d, Master router: %s\n", 
	    vrrp_data->vif.ifname, addr_buf,
	    internal_state_buf, vrrp_data->vrid, state_buf,
	    vrrp_data->priority, vrrp_data->adver_int / 1000000, auth_buf,
	    vrrp_data->preempt ? "yes" : "no", vrrp_data->naddr, vip_buf,
	    vrrp_data->ms_down_timer / 1000000, master_pri, master_buf);
  }
  else {
    /* master */
    char sbuf[80];
    sprintf(*output_line, 
	    "Physical interface: %s, Address: %s\n" \
	    "  Interface state: %s, Group: %d, State: %s\n" \
	    "  Priority: %d, Advertisement interval: %ds, Authentication type: %s\n" \
	    "  Preempt: %s, VIP count: %d, VIP: %s\n" \
	    "  Advertisement timer: %lus, Master router: %s\n" \
	    "  Virtual MAC: %s\n", 
	    vrrp_data->vif.ifname, addr_buf,
	    internal_state_buf, vrrp_data->vrid, state_buf,
	    vrrp_data->priority, vrrp_data->adver_int / 1000000, auth_buf,
	    vrrp_data->preempt ? "yes" : "no", vrrp_data->naddr, vip_buf,
	    vrrp_data->adver_timer / 1000000, master_buf,
	    print_mac_address((unsigned char*)vrrp_hwaddr, sbuf));
	    }
}


/**
 * Walks through the interface looking for a match within the subnet.
 * Once a match is found then the mask length is returned.
 *
 **/
int find_masklength(const char *interface, unsigned long address)
{
  struct sockaddr_in addr;
  struct interface ife;
  int i;
  struct sockaddr sa;
  
  memcpy(ife.name, interface, IFNAMSIZ-1);

  int err = fetch_interface(&ife);

  if (err < 0) {
    return -1;
  }

  sa = ife.netmask;
  memcpy(&addr, (struct sockaddr_in *) &sa,
	 sizeof(struct sockaddr_in));
  //  printf("      Netmask: %s\n", inet_ntoa(addr.sin_addr));
  for (i = 0; i < 32; ++i) {
    if ((unsigned long)(addr.sin_addr.s_addr) == (unsigned long)(g_net_mask[i])) {
      return i;
    }
  }
  return -1;
  address = address;
}


/* Fetch interface configuration */
int fetch_interface(struct interface *ife)
{
  struct ifreq ifr;
  int fd;
  char *ifname = ife->name;
  
  fd = socket(AF_INET, SOCK_STREAM, 0);
  
  if (fd < 0) {
    fprintf(stderr, "Warning: no inet socket available: %s\n",
	    strerror(errno));
    return (-1);
  }
  
  strncpy(ifr.ifr_name, ifname, IFNAMSIZ);
  
  if (ioctl(fd, SIOCGIFHWADDR, &ifr) < 0)
    memset(ife->hwaddr, 0, 32);
  else
    memcpy(ife->hwaddr, ifr.ifr_hwaddr.sa_data, 8);
  strncpy(ifr.ifr_name, ifname, IFNAMSIZ);
  if (ioctl(fd, SIOCGIFADDR, &ifr) == 0) {
    ife->addr = ifr.ifr_addr;
    ife->has_ip = 1;
    strncpy(ifr.ifr_name, ifname, IFNAMSIZ);
    if (ioctl(fd, SIOCGIFNETMASK, &ifr) < 0)
      memset(&ife->netmask, 0, sizeof(struct sockaddr));
    else
      ife->netmask = ifr.ifr_netmask;
  } else
    memset(&ife->addr, 0, sizeof(struct sockaddr));

  return 0;
}


bool get_link_status( char *ifname )
{
  struct ifreq	ifr;
  int		fd	= socket(AF_INET, SOCK_DGRAM, 0);
  if (fd < 0) 	return (-1);
  strncpy(ifr.ifr_name, ifname, sizeof(ifr.ifr_name));
  struct ethtool_value edata;
  int err = 0;
  bool rtn = true;

  edata.cmd = ETHTOOL_GLINK;
  ifr.ifr_data = (caddr_t) & edata;
  err = ioctl(fd, SIOCETHTOOL, &ifr);
  if (err == 0)
    rtn = (edata.data) ? true : false; 
  close(fd);

  if (rtn == false) {
    //get monotonic time
    struct sysinfo si;
    sysinfo(&si);

    if (g_link_status == true) {
      //first time through
      g_link_status = false;
      //start holddown timer
      g_link_holddown_time = si.uptime;
      //return true still though
      return true;
    }
    else {
      //link still down on subsequent call, holddown not expired
      if (si.uptime - g_link_holddown_time < 2) {
	return true;
      }
      //expired holddown, now mark the link as down
      return false; 
    }
  }

  g_link_status = true;
  return true;
}
