// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-

// Copyright (c) 2001-2007 International Computer Science Institute
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software")
// to deal in the Software without restriction, subject to the conditions
// listed in the XORP LICENSE file. These conditions include: you must
// preserve this copyright notice, and you cannot mention the copyright
// holders in advertising related to the Software without their permission.
// The Software is provided WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED. This
// notice is a summary of the XORP LICENSE file; the license in that file is
// legally binding.

// $XORP$

#ifndef __LIBXIPC_FINDER_UNIX_MESSENGER_HH__
#define __LIBXIPC_FINDER_UNIX_MESSENGER_HH__

#include <list>

#include "libxorp/ref_ptr.hh"

#include "finder_unix.hh"
#include "finder_msgs.hh"
#include "finder_messenger.hh"

class FinderUnixMessenger
    : public FinderMessengerBase, protected FinderUnixBase
{
public:
    FinderUnixMessenger(EventLoop&		e,
		       FinderMessengerManager*	mm,
		       XorpFd			sock,
		       XrlCmdMap&		cmds);

    virtual ~FinderUnixMessenger();

    bool send(const Xrl& xrl, const SendCallback& scb);

    bool pending() const;

    inline void close() { FinderUnixBase::close(); }
    
protected:
    // FinderTcpBase methods
    void read_event(int		   errval,
		    const uint8_t* data,
		    uint32_t	   data_bytes);

    void write_event(int	    errval,
		     const uint8_t* data,
		     uint32_t	    data_bytes);

    void close_event();

    void error_event();
    
protected:
    void reply(uint32_t seqno, const XrlError& xe, const XrlArgs* reply_args);

protected:
    void push_queue();
    void drain_queue();
    
    /* High water-mark to disable reads, ie reading faster than writing */
    static const uint32_t OUTQUEUE_BLOCK_READ_HI_MARK = 6;

    /* Low water-mark to enable reads again */
    static const uint32_t OUTQUEUE_BLOCK_READ_LO_MARK = 4;

    typedef list<const FinderMessageBase*> OutputQueue;
    OutputQueue _out_queue;
};

/**
 * Class that creates FinderMessengers for incoming connections.
 */
class FinderUnixListener : public FinderUnixListenerBase {
public:
    FinderUnixListener(EventLoop&		e,
			FinderMessengerManager& mm,
			XrlCmdMap&		cmds,
			const string&		path,
			bool			enabled = true)
	throw (InvalidAddress);

    ~FinderUnixListener();

    /**
     * Instantiate a Messenger instance for sock.
     * @return true on success, false on failure.
     */
    bool connection_event(XorpFd sock);

protected:
    FinderMessengerManager& _mm;
    XrlCmdMap& _cmds;
};

class FinderUnixConnector {
public:
    FinderUnixConnector(EventLoop&		e,
		       FinderMessengerManager&	mm,
		       XrlCmdMap&		cmds,
		       const string&		path);

    virtual ~FinderUnixConnector();

    /**
     * Connect to socket path specified in constructor.
     *
     * @param created_messenger pointer to be assigned messenger created upon
     * successful connect.
     * @return 0 on success, errno on failure.
     */
    int connect(FinderUnixMessenger*& created_messenger);

    string finder_path() const;

protected:
    EventLoop&		    _e;
    FinderMessengerManager& _mm;
    XrlCmdMap&		    _cmds;
    string		    _path;
};

/**
 * Class to establish and manage a single connection to a FinderUnixListener.
 * Should the connection fail after being established a new connection is
 * started.
 */
class FinderUnixAutoConnector
    : public FinderUnixConnector, public FinderMessengerManager
{
public:
    FinderUnixAutoConnector(
			   EventLoop&		     	e,
			   FinderMessengerManager& 	mm,
			   XrlCmdMap&		     	cmds,
			   const string&		path,
			   bool		     		enabled		= true,
			   uint32_t			give_up_ms	= 0
			   );
    virtual ~FinderUnixAutoConnector();

    void set_enabled(bool en);
    bool enabled() const;
    bool connected() const;

protected:
    void do_auto_connect();
    void start_timer(uint32_t ms = 0);
    void stop_timer();

protected:
    /*
     * Implement FinderMessengerManager interface to catch death of
     * active messenger to trigger auto-reconnect.  All methods are
     * forwarded to _real_manager.
     */
    void messenger_birth_event(FinderMessengerBase*);
    void messenger_death_event(FinderMessengerBase*);
    void messenger_active_event(FinderMessengerBase*);
    void messenger_inactive_event(FinderMessengerBase*);
    void messenger_stopped_event(FinderMessengerBase*);
    bool manages(const FinderMessengerBase*) const;
    string active_target_name() const;

protected:
    FinderMessengerManager& _real_manager;
    bool		    _connected;
    bool		    _enabled;
    bool		    _once_active;
    XorpTimer		    _retry_timer;
    XorpTimer		    _giveup_timer;
    int			    _last_error;
    size_t		    _consec_error;

    static const uint32_t CONNECT_RETRY_PAUSE_MS = 100;
    static const uint32_t CONNECT_FAILS_BEFORE_LOGGING = 10;
};

#endif // __LIBXIPC_FINDER_UNIX_MESSENGER_HH__
