//  test_smart_ptr.cc

#include "smart_ptr.hh"
#include "libtpl/fmwk/task.hh"
#include "libtpl/fmwk/task_policy.hh"
#include "libtpl/fmwk/task_mgr.hh"
#include "libtpl/thread/atomic.hh"
#include "libtpl/thread/lock_guards.hh"
#include "libtpl/debug.hh"

#include <iostream>
#include <string>

using namespace tpl;
using namespace std;

struct TestData
{
    TestData(double x, double y) : _x(x), _y(y) {
    	atomic_inc(_obj_counter);
    }
    ~TestData() {
    	atomic_dec(_obj_counter);
    }
    double calc(int coeff) {
    	return _x * coeff + coeff / _y;
    }

    double _x;
    double _y;
    static volatile unsigned long _obj_counter;
};

volatile unsigned long TestData::_obj_counter = 0;

class TestTask : public Task<RUN_THREAD_ONCE>
{
public:
    TestTask(string name, const smart_ptr<TestData>& data) : _name(name), _data(data) {}
    static void broadcast() {
    	LockGuard<Mutex> guard(TestTask::_mutex);
    	TestTask::_cond.broadcast();
    }
protected:
    virtual void run();
    virtual void print_on(ostream& os) const { os << _name << " (" << id() << ")"; }
private:
    string _name;
    smart_ptr<TestData> _data;

    static Mutex     _mutex;
    static StatefullCondition _cond;
};

Mutex TestTask::_mutex;
StatefullCondition TestTask::_cond(_mutex);

void TestTask::run()
{
    cout << "Thread: " << _name << " (" << Thread::current_thread_id() << ")" << endl;

    do {
    	LockGuard<Mutex> guard(_mutex);
    	TestTask::_cond.wait();
    } while(0);

    for (int i = 0; i < 20; ++i) {
    	cout << "Thread: " << _name << ": ref_count=" << _data.ref_count() << endl;
    	{
    	    smart_ptr<TestData> d = _data;
    	    double res = d->calc(i);
    	    cout << "Thread: " << _name << ": ref_count=" << _data.ref_count() << ", res=" << res << endl;
    	    do {
    	    	LockGuard<Mutex> guard(TestTask::_mutex);
    	    	TestTask::_cond.wait(10);
    	    } while(0);
    	}
    }
    cout << "Thread: " << _name << " (" << Thread::current_thread_id() << ") exiting..."<< endl;
}

// Simple interface test
void test_smart_ptr();

int main(int argc, void *argv[])
{
    UNUSED(argc);
    UNUSED(argv);
    test_smart_ptr();

    return 0;
}

void test_smart_ptr()
{
    cout << "Main thread: before creation obj_counter=" << TestData::_obj_counter << endl;
    {
    	smart_ptr<TestData> ptr(new TestData(3.14, 2.97));
    	cout << "Main thread: after creation obj_counter=" << TestData::_obj_counter << endl;
    	cout << "Main thread: creation ref_count=" << ptr.ref_count() << endl;
    	{
    	    TaskManager* mgr = TaskManager::instance();
    	    TestTask t1("task1", ptr);
    	    TestTask t2("task2", ptr);
    	    cout << "Main thread: initial ref_count=" << ptr.ref_count() << endl;

    	    t1.start();
    	    t2.start();
    	    sleep(1);  // guard delay before starting concurrent tasks
    	    TestTask::broadcast();
            // threads are working here...
    	    mgr->join_all();
    	    cout << "Main thread: final ref_count=" << ptr.ref_count() << endl;
    	}
    	cout << "Main thread: before destruction ref_count=" << ptr.ref_count() << endl;
    	cout << "Main thread: before scope exit obj_counter=" << TestData::_obj_counter << endl;
    }
    cout << "Main thread: after scope exit obj_counter=" << TestData::_obj_counter << endl;
}

