// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_SMART_PTR_HH__
#define __TPL_SMART_PTR_HH__

#include "libtpl/thread/atomic.hh"

#include <iostream>

namespace tpl {

template<typename T>
class ref_count_rep;

template<typename T>
class ref_count_array_rep;

template <typename T, template <class> class REP>
class smart_ptr;

/*
template <typename T>
class smart_ptr<T, ref_count_rep>;

template <typename T>
class smart_ptr<T, ref_count_array_rep>;
*/

/**
 *  Reference counted pointer holder object
 */
template<typename T>
class ref_count_rep
{
    friend class smart_ptr<T, tpl::ref_count_rep>;

protected:  // Create/Copy/Destroy

    ref_count_rep(T* ptr);

    ~ref_count_rep();

protected:  // Usage

    inline T* get_ptr() { return _ptr; }

    inline const T* get_ptr() const { return _ptr; }

    inline void set_ptr(T* ptr) { _ptr = ptr; }

    inline void set_ptr(const T* ptr) { _ptr = ptr; }

    inline bool is_null() const { return !_ptr; }

    inline unsigned long inc_ref_count() const { return atomic_inc(_counter); }

    inline unsigned long dec_ref_count() const { return atomic_dec(_counter); }

    inline unsigned long ref_count() const { return _counter; }

protected:

    mutable T*  _ptr;
    volatile mutable unsigned long  _counter;
};

/**
 *  Reference counted array pointer holder object
 */
template<typename T>
class ref_count_array_rep : public ref_count_rep<T>
{
    friend class smart_ptr<T, tpl::ref_count_array_rep>;

protected:  // Create/Copy/Destroy

    ref_count_array_rep(T* ptr);

    ~ref_count_array_rep();
};

template<typename T, template <class> class REP>
inline bool operator== (const smart_ptr<T, REP>& lhs, const smart_ptr<T, REP>& rhs)
{
    return lhs._rep == rhs._rep;
}

template<typename T, template <class> class REP>
inline bool operator> (const smart_ptr<T, REP>& lhs, const smart_ptr<T, REP>& rhs)
{
    return (lhs._rep && rhs._rep && lhs._rep->get_ptr() > rhs._rep->get_ptr());
}

template<typename T, template <class> class REP>
inline bool operator< (const smart_ptr<T, REP>& lhs, const smart_ptr<T, REP>& rhs)
{
    return (lhs._rep && rhs._rep && lhs._rep->get_ptr() < rhs._rep->get_ptr());
}

template <typename T, template <class> class REP>
inline bool operator!= (const smart_ptr<T, REP>& lhs, const smart_ptr<T, REP>& rhs)
{
   return !(lhs == rhs);
}

template <typename T, template <class> class REP>
inline bool operator>= (const smart_ptr<T, REP>& lhs, const smart_ptr<T, REP>& rhs)
{
   return !(lhs < rhs);
}

template <typename T, template <class> class REP>
inline bool operator<= (const smart_ptr<T, REP>& lhs, const smart_ptr<T, REP>& rhs)
{
   return !(lhs > rhs);
}

template <typename T, template <class> class REP>
inline std::ostream& operator<< (std::ostream& os, const smart_ptr<T, REP>& obj)
{
    obj.print_on(os);
    return os;
}

/**
 *  Reference counted smart pointer implementation
 */
template <typename T, template <class> class REP = tpl::ref_count_rep>
class smart_ptr
{
public:  // Create/Copy/Destroy

    smart_ptr() {}

    smart_ptr(T* ptr);

    ~smart_ptr() { unref(); }

    smart_ptr(const smart_ptr& other);

    smart_ptr& operator= (const smart_ptr& other);

public:  // Usage

    inline T* operator->() { return _rep->get_ptr(); }

    inline const T* operator->() const { return _rep->get_ptr(); }

    inline T& operator*() const { return *(_rep->get_ptr()); }

    inline bool is_null() const { return (!_rep || _rep->is_null()); }

    inline unsigned long ref_count() const { return (_rep)? _rep->ref_count() : 0; }

    friend bool operator== <T, REP>(const smart_ptr<T, REP>& lhs, const smart_ptr<T, REP>& rhs);

    friend bool operator> <T, REP>(const smart_ptr<T, REP>& lhs, const smart_ptr<T, REP>& rhs);

    friend bool operator< <T, REP>(const smart_ptr<T, REP>& lhs, const smart_ptr<T, REP>& rhs);

    friend std::ostream& operator<< <T, REP>(std::ostream& os, const smart_ptr<T, REP>& obj);

private:

    void unref();
    void print_on(std::ostream& os) const;

    mutable REP<T>* _rep;
};

};  // namespace tpl

#ifndef NO_COMPILE_INSTANTIATE
#include "smart_ptr.cc"
#endif
#endif // __TPL_SMART_PTR_HH__
