// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_THREAD_HH__
#define __TPL_THREAD_HH__

#include "libtpl/tpl_decls.hh"
#include "runnable.hh"

#if defined(TPL_WIN32)
#include "bits/win32_thread.hh"
#include "bits/win32_errors.hh"
#else // !TPL_WIN32
#include "bits/posix_thread.hh"
#include "bits/posix_errors.hh"
#endif // !TPL_WIN32

namespace tpl {

template <class THREAD>
class ThreadDispatcher
{
public:  // Types

    typedef typename THREAD::thread_id_t thread_id_t;
    typedef typename THREAD::THREAD_FUNCTION THREAD_FUNCTION;

#if 0
#if defined(TPL_WIN32)
    typedef uint32_t (*THREAD_FUNCTION) (void*);
#else // !TPL_WIN32
    typedef void* (*THREAD_FUNCTION) (void*);
#endif // !TPL_WIN32
#endif

public:  // Create/Copy/Destroy

    ThreadDispatcher(THREAD_FUNCTION func = 0) : _thread(func) {}
    virtual ~ThreadDispatcher() {}

    TPL_NO_COPY_TEMPLATE(ThreadDispatcher, THREAD);

public:  // Usage

    /**
     *  Register thread function.
     */
    void set_thread_function(THREAD_FUNCTION func) { _thread.set_thread_function(func); }

    /**
     *  Start a thread with a context parameter.
     */
    int start(void *ctx) { return _thread.start(ctx); }

    /**
     *  Detaches underlying thread.
     */
    int detach() { return _thread.detach(); }

    /**
     *  Joins underlying thread.
     */
    int join(void **ret_val) { return _thread.join(ret_val); }

    /**
     *  Cancels underlying thread.
     */
    int cancel() { return _thread.cancel(); }

public:  // State

    /**
     *  @return thread id of underlying thread
     */
    thread_id_t id() const { return _thread.id(); }

    /**
     *  @return current thread id.
     */
    static thread_id_t current_thread_id() { return THREAD::current_thread_id(); }

    /**
     *  @return true if thread has been started.
     */
    bool is_started() const { return _thread.is_started(); }

    /**
     *  @return true if thread has been joined.
     */
    bool is_joined() const { return _thread.is_joined(); }

    /**
     *  @return true if thread has been started and joined.
     */
    bool is_done() const { return _thread.is_done(); }

    /**
     *	@return true if thread has been started and has not yet been joined.
     */
    bool is_active() const { return _thread.is_active(); }

    /**
     *	@return true if thread has been cancelled.
     */
    bool is_cancelled() const { return _thread.is_cancelled(); }

private:

    THREAD    _thread;
};

/**
 *  @short A thread object.
 *  This interface should be implemented by any class whose instances can be
 *  executed by a thread.
 */
template <class THREAD, template <class> class DISPATCHER>
class ThreadRunner : protected Runnable, public DISPATCHER<THREAD>
{
public:  // Constants

    static const uint32_t STILL_ACTIVE = 259;

public:  // Create/Copy/Destroy

    ThreadRunner();
    ThreadRunner(Runnable *target);
    virtual ~ThreadRunner();

    TPL_NO_COPY_TEMPLATE(ThreadRunner, THREAD, DISPATCHER);

public:  // Usage

    /**
     *  Starts separately executing thread.
     */
    virtual void start();

    /**
     *  Joins separately executing thread.
     */
    virtual void join();

    /**
     *  @return termination status of a thread function.
     */
    virtual int exit_code() const;

    // protecting ancestor's non-virtual methods from hiding
    using DISPATCHER<THREAD>::start;
    using DISPATCHER<THREAD>::join;

protected:  // Runnable interface

    virtual void run();

protected:  // Printing

    virtual void print_on(std::ostream& os) const;

private:

    static void* thread_proc(void *arg);

    Runnable *_target;
    void     *_retval;
};


#if defined(FAT_INTERFACE)
#  if defined(TPL_WIN32)
typedef ThreadRunner<WinThread, ThreadDispatcher> Thread;
#  else // !TPL_WIN32
typedef ThreadRunner<PosixThread, ThreadDispatcher> Thread;
#  endif // !TPL_WIN32
#else  // !FAT_INTERFACE
#  if defined(TPL_WIN32)
typedef WinThread Thread;
#  else // !TPL_WIN32
typedef PosixThread Thread;
typedef PosixThreadSchedParams SchedParams;
typedef PosixThreadAttributes ThreadAttributes;
#  endif // !TPL_WIN32
#endif  // !FAT_INTERFACE

};  // namespace tpl

#endif // __TPL_THREAD_HH__
