// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_SYNCHRONIZER_HH__
#define __TPL_SYNCHRONIZER_HH__

#include "libtpl/tpl_decls.hh"

#include <iostream>

namespace tpl {

/**
 * @short A multithreaded synchronizer facility interface
 */
class Synchronizer
{
public: // Create/Copy/Destroy

    Synchronizer() {}

    virtual ~Synchronizer() {}

    TPL_NO_COPY(Synchronizer);

public:  // Usage

    /**
     *  Blocks on a condition for an interval in milliseconds.
     *  With statefull policy if condition has been signalled before this call
     *  it will exit immediately.
     *  XXX: if specified interval is 0 this call will block until signalled.
     *  @return 0 if condition signalled, otherwise - error code
     */
    virtual int wait(unsigned long ms = 0) const = 0;

    /**
     *	Blocks on a condition until specified moment in time (absolute time).
     *  With statefull policy if condition has been signalled before this call
     *  it will exit immediately.
     *  XXX: if time specified in the past this call will block until signalled.
     *  @return 0 if condition signalled, otherwise - POSIX error code:
     *  ETIMEDOUT - specified time reached; EINTR - call was interrupted.
     */
    //int wait_until(const DateTime& deadline) const = 0;

    /**
     *  Signals the thread which is waiting on a condition.
     */
    virtual void signal() const = 0;

    /**
     *  Signals to all threads waiting on a condition (bradcast)
     */
    virtual void broadcast() const = 0;
	
public:  // State

    /**
     *  @return true if the condition is shared.
     */
    virtual bool is_shared() const = 0;

public:  // Printing

    friend inline std::ostream& operator<< (std::ostream& os, const Synchronizer& obj);

protected:

    virtual void print_on(std::ostream& os) const = 0;

};

inline std::ostream& operator<< (std::ostream& os, const Synchronizer& obj)
{
    obj.print_on ( os );
    return os;
}

};  // namespace tpl

#endif // __TPL_SYNCHRONIZER_HH__
