// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#include "debug.hh"

#include "rw_lock.hh"
#include "lock_guards.hh"

using namespace tpl;

////////////////////////////////////////////////////////////
//                    ReadWriteLock                       //
////////////////////////////////////////////////////////////

ReadWriteLock::ReadWriteLock() : _active_readers(0), _active_writers(0), 
    _waiting_readers(0), _waiting_writers(0), _read_cond(&_mutex), _write_cond(&_mutex) 
{}

ReadWriteLock::~ReadWriteLock()
{}

bool
ReadWriteLock::try_read_lock()
{
    LockGuard<Mutex> guard(_mutex);
    if (_active_writers != 0) {
    	return false;
    }

    ++_active_readers;
    return true;
}

void
ReadWriteLock::read_lock()
{
    LockGuard<Mutex> guard(_mutex);
    while (_active_writers != 0) {
    	++_waiting_readers;
    	_read_cond.wait();
    	--_waiting_readers;
    }

    ++_active_readers;
}

bool
ReadWriteLock::try_write_lock()
{
    LockGuard<Mutex> guard(_mutex);
    if (_active_writers + _active_readers > 0) {
    	return false;
    }
    ++_active_writers;
    return true;
}

void
ReadWriteLock::write_lock()
{
    LockGuard<Mutex> guard(_mutex);
    while (_active_writers + _active_readers > 0) {
    	++_waiting_writers;
    	_write_cond.wait();
    	--_waiting_writers;
    }
    ++_active_writers;
}

void
ReadWriteLock::unlock()
{ 
    LockGuard<Mutex> guard(_mutex);
    if (_active_readers > 0)
    	--_active_readers;
    else if (_active_writers == 1)
    	_active_writers = 0;
    else
    	TPL_ASSERT(_active_readers == 0 && _active_writers != 1);

    if (_waiting_writers > 0) {
    	if (_active_readers == 0)
    	_write_cond.signal();
    }
    else if (_waiting_readers > 0)
    	_read_cond.broadcast();
}

int
ReadWriteLock::waiting_readers() const
{
    return _waiting_readers;
}

int
ReadWriteLock::waiting_writers() const
{
    return _waiting_writers;
}

int
ReadWriteLock::active_readers() const
{
    return _active_readers;
}

int
ReadWriteLock::active_writers() const
{
    return _active_writers;
}
