// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_MUTEX_HH__
#define __TPL_MUTEX_HH__

#include "lock.hh"

#if defined(TPL_WIN32)
#include "bits/win32_lock_objects.hh"
#else // !TPL_WIN32
#include "bits/posix_lock_objects.hh"
#endif // !TPL_WIN32

namespace tpl {

/**
 *  Mutex implementation through OS mutex facility
 */
template <class MUTEX>
class MutexLock : public Lock
{
public:  // Create/Copy/Destroy

    MutexLock();
    virtual ~MutexLock();

    TPL_NO_COPY_TEMPLATE(MutexLock, MUTEX);

public:  // Usage

    /**
     *  Acquires the lock if it is free at the time of invocation.
     *  Acquires the lock if it is available and returns immediately with the value true.
     *  If the lock is not available then this method will return immediately with the value false.
     *
     *  @return false if lock was acquired and false otherwise 
     */
    virtual bool try_lock() const { return _mutex.try_lock(); }

    /**
     *  Acquires the lock
     *  If the lock is currently acquired by another thread
     *  the calling thread blocks until lock becomes available.
     *  XXX: lock implementation may be able to detect erroneous usage of the lock
     *  such as potential deadlocks, and may throw an exception.
     */
    virtual void lock() const { _mutex.lock(); }

    /**
     *  Releases the lock.
     *  If any other thread is blocked on this lock it becomes the owner.
     */
    virtual void unlock() const { _mutex.unlock(); }

    /**
     *  Returns a new Condition instance that is bound to this Lock instance.
     *  @return Condition instance for this Lock instance.
     */
    virtual Synchronizer* new_synchronizer() const;

public:  // Local

    /**
     *  @return Internal mutex handle
     */
    const MUTEX& handle() const { return _mutex; }

public:  // State

    /**
     *  @return true if the mutex is shared.
     */
    virtual bool is_shared() const { return _mutex.is_shared(); }

protected:

    virtual void print_on(std::ostream&) const {}

private:

    MUTEX _mutex;
};

#if defined(FAT_INTERFACE)
#  if defined(TPL_WIN32)
typedef MutexLock<WinMutex> Mutex;
typedef MutexLock<WinThreadMutex> RecursiveMutex;
typedef MutexLock<WinMutex> SafeMutex;
#  else // !TPL_WIN32
typedef MutexLock<PosixMutex<PosixMutexAttributes::NORMAL> > Mutex;
typedef MutexLock<PosixMutex<PosixMutexAttributes::RECURSIVE> > RecursiveMutex;
typedef MutexLock<PosixMutex<PosixMutexAttributes::ERRORCHECKING> > SafeMutex;
#  endif // !TPL_WIN32
#else  // !FAT_INTERFACE
#  if defined(TPL_WIN32)
typedef WinMutex Mutex;
typedef WinThreadMutex RecursiveMutex;
typedef WinMutex SafeMutex;
#  else // !TPL_WIN32
typedef PosixMutex<PosixMutexAttributes::NORMAL> Mutex;
typedef PosixMutex<PosixMutexAttributes::RECURSIVE> RecursiveMutex;
typedef PosixMutex<PosixMutexAttributes::ERRORCHECKING> SafeMutex;
#  endif // !TPL_WIN32
#endif  // !FAT_INTERFACE

};  // namespace tpl

#endif // __TPL_MUTEX_HH__
