// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_LOCK_HH__
#define __TPL_LOCK_HH__

#include "libtpl/tpl_decls.hh"

#include <iostream>

namespace tpl {

class Synchronizer;

/**
 *  @short A multithreaded lock facility interface
 */
class Lock
{
public:  // Create/Copy/Destroy

    Lock() {}
    virtual ~Lock() {}

    TPL_NO_COPY(Lock);

public:  // Usage

    /**
     *  Acquires the lock if it is free at the time of invocation.
     *  Acquires the lock if it is available and returns immediately with the value true.
     *  If the lock is not available then this method will return immediately with the value false.
     *
     *  @return false if lock was acquired and false otherwise 
     */
    virtual bool try_lock() const = 0;

    /**
     *  Acquires the lock
     *  If the lock is currently acquired by another thread
     *  the calling thread blocks until lock becomes available.
     *  XXX: lock implementation may be able to detect erroneous usage of the lock
     *  such as potential deadlocks, and may throw an exception.
     */
    virtual void lock() const = 0;

    /**
     *  Releases the lock.
     *  If any other thread is blocked on this lock it becomes the owner.
     */
    virtual void unlock() const = 0;

    /**
     *  Returns a new Synchronizer instance that is bound to this Lock instance.
     *  @return Synchronizer instance for this Lock instance.
     *  @throw UnsupportedOperation if this Lock implementation does not support synchronizers.
     */
    virtual Synchronizer* new_synchronizer() const = 0;

public:  // State

    /**
     *  @return true if the mutex is shared.
     */
    virtual bool is_shared() const = 0;

public:  // Printing

    friend inline std::ostream& operator<< (std::ostream& os, const Lock& obj);

protected:

    virtual void print_on(std::ostream& os) const = 0;

};

inline std::ostream& operator<< (std::ostream& os, const Lock& obj)
{
    obj.print_on ( os );
    return os;
}

};  // namespace tpl

#endif // __TPL_LOCK_HH__
