// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_CONDITION_HH__
#define __TPL_CONDITION_HH__

#include "synch.hh"
#include "mutex.hh"

#if defined(TPL_WIN32)
#include "bits/win32_condition.hh"
#else // !TPL_WIN32
#include "bits/posix_condition.hh"
#endif // !TPL_WIN32

namespace tpl {

/**
 * @short A multithreaded condition synchronizer implementation
 */
template <class CONDITION>
class ConditionSynchronizer : public Synchronizer
{
public: // Create/Copy/Destroy

    template <class LOCK>
    ConditionSynchronizer(const LOCK& lock);

    virtual ~ConditionSynchronizer();

    TPL_NO_COPY_TEMPLATE(ConditionSynchronizer, CONDITION);

public:  // Usage

    /**
     *  Blocks on a condition for an interval in milliseconds.
     *  With statefull policy if condition has been signalled before this call
     *  it will exit immediately.
     *  XXX: if specified interval is 0 this call will block until signalled.
     *  @return 0 if condition signalled, otherwise - error code
     */
    virtual int wait(unsigned long time_ms = 0) const;

    /**
     *	Blocks on a condition until specified moment in time (absolute time).
     *  With statefull policy if condition has been signalled before this call
     *  it will exit immediately.
     *  XXX: if time specified in the past this call will block until signalled.
     *  @return 0 if condition signalled, otherwise - error code
     */
    //int wait_until(const DateTime& deadline) const;

    /**
     *  Signals the thread which is waiting on a condition.
     */
    virtual void signal() const { _cond.signal(); }

    /**
     *  Signals to all threads waiting on a condition (bradcast)
     */
    virtual void broadcast() const { _cond.broadcast(); }

public:  // State

    /**
     *  @return true if the condition is shared.
     */
    virtual bool is_shared() const { return _cond.is_shared(); }

protected:

    virtual void print_on(std::ostream&) const {}

private:

    CONDITION _cond;
};

#if defined(FAT_INTERFACE)
#  if defined(TPL_WIN32)
typedef ConditionSynchronizer<WinCondition> Condition;
#  else // !TPL_WIN32
typedef ConditionSynchronizer<PosixCondition<CONDITION_STATELESS> > Condition;
typedef ConditionSynchronizer<PosixCondition<CONDITION_STATEFULL> > StatefullCondition;
#  endif // !TPL_WIN32
#else  // !FAT_INTERFACE
#  if defined(TPL_WIN32)
typedef WinCondition Condition;
#  else // !TPL_WIN32
typedef PosixCondition<CONDITION_STATELESS> Condition;
typedef PosixCondition<CONDITION_STATEFULL> StatefullCondition;
#  endif // !TPL_WIN32
#endif  // !FAT_INTERFACE

};  // namespace tpl

#endif // __TPL_CONDITION_HH__
