// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_WIN32_THREAD_HH__
#define __TPL_WIN32_THREAD_HH__

#include "win32_handle.hh"

namespace tpl {

class TPL_EXPORT WinThread : public WinHandle
{
public: // Types & Constants

    typedef uint32_t thread_id_t;
    typedef uint32_t (*THREAD_FUNCTION) (void*);
    static const uint32_t THREAD_CANCELED = 0x0000ffff;

public:  // Create/Copy/Destroy

    WinThread(THREAD_FUNCTION func = 0);
    ~WinThread();

    TPL_NO_COPY(WinThread);

public:  // Usage

    /**
     *  Register thread function.
     *  PRE_CONDITION: func != NULL;
     *  PRE_CONDITION: function has not been set yet;
     */
    void set_thread_function(THREAD_FUNCTION func);

    /**
     *  Start a thread with a context parameter.
     */
    int start(void *ctx);

    /**
     *  Detach thread.
     *  XXX: Since executable will fail on termination when not all threads
     *  are terminated custom notification for thread termination is required.
     *  PRE_CONDITION: cannot detach twice.
     */
    int detach();

    /**
     *  Wait for thread termination (blocking).
     *  PRE_CONDITION cannot join itself.
     */
    int join(void **ret_val);

    /**
     *  Cancel a thread.
     */
    int cancel();

    /**
     *  Send asynchronous signal to a thread.
     *  if signo == 0 signal is not sent.
     *
     *  1. Global signal handler for 'signo' is installed and
     *  signal is not masked in the target thread:
     *  Execution of the target thead will be interrupted
     *  and signal will be processed by the global signal handler
     *  wich will be executed in receiving thread. Thread will
     *  resume it's execution after signal handler is done.
     *
     *  2. Signal is masked in the target thread:
     *  Signal will never be delivered.
     *
     *  3. Global signal handler for 'signo' is not installed
     *  and signal is not masked in the target thread:
     *  Default behaviour will be applied for the whole process.
     *
     *  PRE_CONDITION thread has to be active (started not joined)
     */
    int kill(int signo);

    void resume();
    void suspend();
    bool suspended() const ;

public:  // Properties

    /**
     *  @return thread id.
     */
    thread_id_t id() const { return _id; }

    /**
     *  @return current thread id.
     */
    static thread_id_t current_thread_id();

    /**
     *  Thread priority property
     */
    int  priority() const;
    void set_priority(int) const;

public:  // State
    /**
     *  @return true if thread has been started.
     */
    bool is_started() const { return _started; }

    /**
     *  @return true if thread has been joined.
     */
    bool is_joined() const { return _joined; }

    /**
     *  @return true if thread has been started and joined.
     */
    bool is_done() const { return (_started && _joined); }

    /**
     *	@return true if thread has been started and has not yet been joined.
     */
    bool is_active() const { return (_started && !_joined); }

    /**
     *	@return true if thread has been started and has not yet been joined.
     */
    bool is_cancelled() const { return (_started && !_joined && _cancelled); }

private:

    uint32_t	    _id;		// Thread id
    uint32_t	    _attr;		// Thread attributes
    THREAD_FUNCTION _func;		// Thread function.
    bool	    _started;		// TRUE when thread has been started.
    bool	    _joined;		// TRUE when thread has been joined.
    bool	    _suspended;
    bool	    _detached;
    bool	    _cancelled;
    WinThreadMutex  _lock;
};

};  // namespace tpl

#endif // __TPL_WIN32_THREAD_HH__
