// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>


#include "win32_thread.hh"
#include "libtpl/debug.hh"

#if !defined(USE_OS_CREATE_THREAD_ROUTINE)
#  include <process.h>
    typedef unsigned (__stdcall* t_beginthread_proc)(void *);
#endif

WinThread::WinThread(THREAD_FUNCTION func) : _id(0), _attr(0), _func(func),
    _started(false), _joined(false), _suspended(false), _detached(false)
{
}

WinThread::~WinThread()
{
    TPL_ASSERT(!_started || _suspended || _joined || _detached);
}

void WinThread::set_thread_function(THREAD_FUNCTION func)
{
    LockGuard<WinThreadMutex> guard(_lock);
    TPL_ASSERT(func);
    TPL_ASSERT(!_func);
    _func = func;
}

int WinThread::start(void *ctx)
{
    LockGuard<WinThreadMutex> guard(_lock);
#if defined(USE_OS_CREATE_THREAD_ROUTINE)
    _handle = ::CreateThread(0, 0, thread_proc, ctx, _attr, &_id)),
#else // !USE_OS_CREATE_THREAD_ROUTINE
    _handle = (HANDLE)_beginthreadex(0, 0, (t_beginthread_proc) thread_proc, ctx,
                                   _attr, (unsigned int*)&_id)),
#  if defined(NEED_DUP_THREAD_HANDLE)
    HANDLE dup_handle;
    TPL_VERIFY(
    	::DuplicateHandle(::GetCurrentProcess(), _handle, ::GetCurrentProcess(),
    		&dup_handle, 0, FALSE, DUPLICATE_SAME_ACCESS)
    );
    _handle = dup_handle;
#  endif
#endif // !USE_OS_CREATE_THREAD_ROUTINE
    if (_handle == INVALID_HANDLE_VALUE) {
    	return ::GetLastError();
    }
    _started = true;
    return 0;
}

int WinThread::join(void **ret_val)
{
    LockGuard<WinThreadMutex> guard(_lock);
    TPL_ASSERT(is_active());
    TPL_ASSERT(!_detached);
    int err = 0;
    if (!_suspended && !_joined) {
        err = ::WaitForSingleObject(_handle, INFINITE);
    	TPL_VERIFY(err == WAIT_OBJECT_0);
        if (err == WAIT_FAILED) {
    	    err = ::GetLastError();
    	    return err;
    	}
    }
    _joined = true;
    uint32_t* rc = static_cast<uint32_t*>(ret_val);
    err = ::GetExitCodeThread(_handle, rc);
    _cancelled = (*ret_val == THREAD_CANCELED);
    if (err == 0)
    	return ::GetLastError();

    return 0;
}

int WinThread::detach()
{
    LockGuard<WinThreadMutex> guard(_lock);
    TPL_ASSERT(!_joined);
    _detached = true;
    return 0;
}

int WinThread::cancel()
{
    LockGuard<WinThreadMutex> guard(_lock);
    TPL_ASSERT(!_joined && !_cancelled);
    int err = ::TerminateThread(_handle, THREAD_CANCELED);
    if (err == 0)
    	err = ::GetLastError();
    return err;
}

int WinThread::kill(int signo)
{
    LockGuard<WinThreadMutex> guard(_lock);
    TPL_ASSERT(!_joined);
    int err = ::TerminateThread(_handle, signo);
    if (err == 0)
    	err = ::GetLastError();
}

void WinThread::suspend()
{
    LockGuard<WinThreadMutex> guard(_lock);
    TPL_ASSERT(!_joined);
    TPL_VERIFY( ::SuspendThread(_handle) != 0xffffffff );
    _suspended = true;
}

bool WinThread::suspended() const
{
    LockGuard<WinThreadMutex> guard(_lock);
    return _suspended;
}

void WinThread::resume()
{
    LockGuard<WinThreadMutex> guard(_lock);
    TPL_ASSERT(!_joined);
    _suspended = ::ResumeThread(_handle) > 1;
}

int WinThread::priority() const
{
    return ::GetThreadPriority(_handle);
}

void WinThread::set_priority(int priority) const
{
    ::SetThreadPriority(_handle, priority);
}

uint32_t WinThread::current_thread_id()
{
    return ::GetCurrentThreadId();
}

