// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_WIN32_LOCK_OBJECTS_HH_
#define __TPL_WIN32_LOCK_OBJECTS_HH_

#include "win32_handle.hh"

namespace tpl {

/**
 * @short WinSpinlock object - a lightweight locking mechanism for short locks.
 * A lightweight user-space locking facility
 */
class TPL_EXPORT WinSpinlock
{
public:  // Create/Copy/Destroy

    WinSpinlock()
      : _spinlock(0) {}

    TPL_NO_COPY(WinSpinlock);

public:	 // Usage

    bool try_lock() const {
    	return TPL_MD::try_acquire_spinlock(&_spinlock))
    }

    void lock() const {
      #if defined(TPL_DEBUG)
        long __ = _spinlock;
        ASSERT(!__ || __ == 1);
      #endif
      while (TPL_MD::try_acquire_spinlock(&_spinlock))
        TPL_OS::yield();
    }

    void unlock() const { TPL_MD::release_spinlock(&_spinlock); }

private:
    mutable uint32_t _spinlock;
};


/**
 *  @short WinThreadMutex - critical section wrapper class.
 *  A lightweight mutex implementation that can't be shared by processes.
 */
class TPL_EXPORT WinThreadMutex
{
public:  // Create/Copy/Destroy

    WinThreadMutex(){ ::InitializeCriticalSection(&_mutex); }
    ~WinThreadMutex(){ ::DeleteCriticalSection(&_mutex); }

    TPL_NO_COPY(WinThreadMutex);

public:  // Usage

    bool try_lock() const {
    	return (bool) ::TryEnterCriticalSection(&_mutex);
    }

    void lock() const { ::EnterCriticalSection(&_mutex); }

    void unlock() const { ::LeaveCriticalSection(&_mutex); }

private:
    mutable CRITICAL_SECTION _mutex;
};


/**
 *  @short WinMutex - windows mutex wrapper class.
 *  An errorchecking mutex that can be shared between processes.
 */
class TPL_EXPORT WinMutex : public WinHandle
{
public:  // Create/Copy/Destroy
    explicit WinMutex(const char* name = 0)
      : WinHandle(::CreateMutex(0, false, name)) {}

    explicit WinMutex(HANDLE handle)
      : WinHandle(handle) {}

    TPL_NO_COPY(WinMutex);

public:  // Usage

    bool try_lock() const {
    	DWORD rval = ::WaitForSingleObject(_handle, 0);
    	ASSERT( rval != WAIT_FAILED);
    	return (rval != WAIT_TIMEOUT);
    }

    void lock() const { ASSERT( ::WaitForSingleObject(_handle, INFINITE) != WAIT_FAILED ); }

    void unlock() const { ASSERT( ::ReleaseMutex(_handle) ); }

public:  // State

    /**
     *  @return true if the mutex is shared.
     */
    bool is_shared() const { return true; }

};

}; // namespace tpl

#endif // __WIN32_LOCK_OBJECTS_H_
