// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_WIN32_CONDITION_HH__
#define __TPL_WIN32_CONDITION_HH__

#include "win32_lock_objects.hh"

#include <windows.h>

namespace tpl {

/**
 * @short Mimic POSIX conditional variable.
 */
class TPL_EXPORT WinCondition
{
public:  // Create/Copy/Destroy

    /**
     *  @param mutex - external mutex to guard against race condition in wait.
     */
    WinCondition(const WinMutex& mutex);

    ~WinCondition();

    TPL_NO_COPY(WinCondition);

public:	 // Usage

    /**
     *  Waits until conditional variable become signaled.
     *  XXX: releases external mutex.
     *  @param time_ms how long to wait in msec. -1 means wait forever
     *  @return -1 if time out expired.
     */
    int wait(uint32_t time_ms = 0) const;

    /**
     *  Waits until conditional variable become signaled.
     *  XXX: releases external mutex.
     *  @param time_ms how long to wait in msec. -1 means wait forever
     *  @return -1 if time out expired.
     */
    int wait_until(uint32_t time_ms) const { return -1; }

    /**
     * Signals condition variable.
     */
    void signal() const;

    /**
     * Sends signal to all threads waiting on condition
     */
    void broadcast() const;

public:  // State

    /**
     *  @return true if the condition is shared.
     */
    bool is_shared() const { return false; }

private:

    /** External mutex */
    mutable const WinMutex& _mutex;
    /** Number of waiting threads */
    mutable int _wait_count;
    /** Serialize access to _waiting_count. */
    mutable WinThreadMutex _wait_count_lock;
    /**
     * Semaphore used to queue up threads waiting for the condition to
     * become signaled
     */
    mutable HANDLE _sema;
    /**
     * An auto-reset event used by the broadcast/signal thread to wait
     * for all the waiting thread(s) to wake up and be released from the
     * semaphore.
     */
    mutable HANDLE _wait_done;
    /**
     * Keeps track of whether we were broadcasting or signaling.
     * This allows us to optimize the code if we're just signaling.
     */
    mutable bool _broadcast_pending;
};

};  // namespace tpl

#endif // __TPL_WIN32_CONDITION_HH__
