// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#include "win32_condition.hh"
#include "libtpl/lock_guards.hh"

WinCondition::WinCondition(const WinMutex& mutex) :
    _mutex(mutex), _wait_count(0), _broadcast_pending(false)
{
    _sema = ::CreateSemaphore(NULL, 0, 0x7fffffff, NULL);
    _wait_done = ::CreateEvent (NULL,
                                   FALSE, // auto-reset
                                   FALSE, // non-signaled initially
                                   NULL);
}

WinCondition::~WinCondition()
{
    ::CloseHandle(_sema);
    ::CloseHandle(_wait_done);
}

int WinCondition::wait(uint32_t time_ms)
{
    {
    	LockGuard<WinThreadMutex> guard(_wait_count_lock);
    	_wait_count++;
    }

    // This call atomically releases the mutex and waits on the
    // semaphore until <pthread_cond_signal> or <pthread_cond_broadcast>
    // are called by another thread.
    DWORD res;
    if (time_ms == 0) {
    	res = ::SignalObjectAndWait (_mutex, _sema, INFINITE, FALSE);
    }
    else {
    	res = ::SignalObjectAndWait (_mutex, _sema, time_ms, FALSE);
    }

    bool last_waiter = false;
    {
    	LockGuard<WinThreadMutex> guard(_wait_count_lock);
    	_wait_count--;
    	// Check to see if we're the last waiter after <pthread_cond_broadcast>.
    	last_waiter = _broadcast_pending && _wait_count == 0;
    }

    // If we're the last waiter thread during this particular broadcast
    // then let all the other threads proceed.
    if (last_waiter) {
    	// This call atomically signals the <waiters_done_> event and waits until
    	// it can acquire the <external_mutex>.  This is required to ensure fairness. 
    	::SignalObjectAndWait (_wait_done, _mutex, INFINITE, FALSE);
    }
    else {
    	// Always regain the external mutex since that's the guarantee we
    	// give to our callers. 
    	::WaitForSingleObject (_mutex, INFINITE);
    }
    return 0;
}

void WinCondition::signal()
{
    bool have_waiters = false;
    {
    	LockGuard<WinThreadMutex> guard(_wait_count_lock);
    	have_waiters = waiters_count_ > 0;
    }

    // If there aren't any waiters, then this is a no-op.  
    if (have_waiters)
    	::ReleaseSemaphore(_sema, 1, 0);
}

void WinCondition::broadcast()
{
    // This is needed to ensure that <waiters_count_> and <was_broadcast_> are
    // consistent relative to each other.
    _wait_count_lock.lock();
    bool have_waiters = false;

    if (_wait_count > 0) {
    	// We are broadcasting, even if there is just one waiter...
    	// Record that we are broadcasting, which helps optimize
    	// <pthread_cond_wait> for the non-broadcast case.
    	_broadcast_pending = true;
    	have_waiters = true;
    }

    if (have_waiters) {
    	// Wake up all the waiters atomically.
    	::ReleaseSemaphore (_sema, _wait_count, 0);

    	_wait_count_lock.unlock();

    	// Wait for all the awakened threads to acquire the counting
    	// semaphore. 
    	::WaitForSingleObject(_wait_done, 2000);
    	// This assignment is okay, even without the <waiters_count_lock_> held 
    	// because no other waiter threads can wake up to access it.
    	_broadcast_pending = false;
    }
    else
    	_wait_count_lock.unlock();
}
