// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#ifndef __TPL_POSIX_THREAD_HH__
#define __TPL_POSIX_THREAD_HH__

#include "libtpl/tpl_decls.hh"

#include <pthread.h>

// #include <posix_signals.hh>

namespace tpl {

/**
 *
 */
class PosixThreadSchedParams
{
public:  // Create/Copy/Destroy

    PosixThreadSchedParams(int policy, int priority = 0) :
    	_policy(policy) {
    	_param.sched_priority = priority;
    }

    PosixThreadSchedParams(int policy, const struct sched_param& param) :
    	_policy(policy), _param(param) {}

public:  // Properties

    /**
     *  Set scheduling policy.
     */
    void set_policy(int policy) {
    	_policy = policy;
    }

    /**
     *  Get scheduling policy.
     */
    int policy() const {
    	return _policy;
    }

    /**
     *  Set scheduling priority.
     */
    void set_priority(int priority) {
    	_param.sched_priority = priority;
    }

    /**
     *  Get scheduling priority.
     */
    int priority() const {
    	return _param.sched_priority;
    }

    /**
     *  Get scheduling parameters structure.
     */
    struct sched_param param() const {
    	return _param;
    }

private:
    int _policy;		// Sched policy
    struct sched_param _param;	// Sched parameters (priority)
};


/**
 *
 */
class PosixThreadAttributes
{
public:  // Create/Copy/Destroy

    PosixThreadAttributes(int detached = PTHREAD_CREATE_JOINABLE);
    PosixThreadAttributes(const PosixThreadSchedParams& sched_params);

    ~PosixThreadAttributes();

    TPL_NO_COPY(PosixThreadAttributes);

public:  // Properties

    /**
     *  Set thread detached attribute.
     */
    void set_detached(int detached);

    /**
     *  Get thread detached attribute value.
     */
    int detached() const;

    /**
     *  Set scheduling parameters.
     */
    void set_sched_params(const PosixThreadSchedParams& sched_params);

    /**
     *  Get scheduling parameters.
     */
    PosixThreadSchedParams sched_params() const;

    /**
     *  Set scheduling parameters inheritance.
     */
    void set_inherit(int inherit);

    /**
     *  Get scheduling parameters inheritance.
     */
    int inherit() const;

    /**
     *  Set scheduling contention scope.
     */
    void set_scope(int scope);

    /**
     *  Get scheduling contention scope.
     */
    int scope() const;

    /**
     *  Get scheduling contention scope.
     */
    const pthread_attr_t& threadattr_handle() const {
    	return _attr;
    }

private:
    pthread_attr_t  _attr;	// Thread attributes
};


/**
 *
 */
template <class CLEANUP_HANDLER>
class PosixThreadCancellationScope
{
    typedef void (*CLEANUP_FUNCTION) (void*);

public:
    PosixThreadCancellationScope(CLEANUP_HANDLER & handler,
    	void (CLEANUP_HANDLER::* cleanup_func)()) :
    	_handler(handler), _cleanup_func(cleanup_func) {
    	::_pthread_cleanup_push(&_buffer, (CLEANUP_FUNCTION) cancellation_handler, this);
    }

    PosixThreadCancellationScope(CLEANUP_HANDLER & handler,
    	void (CLEANUP_HANDLER::* setup_func)(), void (CLEANUP_HANDLER::* cleanup_func)()) :
    	_handler(handler), _cleanup_func(cleanup_func) {
    	::_pthread_cleanup_push(&_buffer, (CLEANUP_FUNCTION) cancellation_handler, this);
	(_handler.*setup_func)();
    }

    ~PosixThreadCancellationScope() {
    	::_pthread_cleanup_pop(&_buffer, 1); // execute cleanup handler on exit from scope
    }

    TPL_NO_COPY(PosixThreadCancellationScope);

private:
    void call_cleanup() {
    	(_handler.*_cleanup_func)();
    }

    static void cancellation_handler(void* ctx) {
    	PosixThreadCancellationScope *p = static_cast<PosixThreadCancellationScope>(ctx);
    	p->call_cleanup();
    }

    struct _pthread_cleanup_buffer _buffer;
    CLEANUP_HANDLER & _handler;
    void (CLEANUP_HANDLER::* _cleanup_func)();
};

/**
 *
 */
struct PosixThreadCancellationPolicy
{

    /**
     *  Enable cancellation and set the option.
     *  @param type can be set to:
     *  PTHREAD_CANCEL_DEFERRED - defer cancelation to CANCEL_POINT.  
     *  PTHREAD_CANCEL_ASYNCHRONOUS - immidiate cancelation.
     */
    static void enable_cancellation(int type);

    /**
     *  Disable thread cancellation.
     */
    static void disable_cancellation();
};


/**
 *
 */
struct PosixThreadSignalPolicy
{
};


/**
 *  @short Implements wrapper on pthread_t handle.
 */
class PosixThread
{
public:  // Types

    typedef pthread_t thread_id_t;
    typedef void* (*THREAD_FUNCTION) (void*);

public:  // Create/Copy/Destroy

    PosixThread(THREAD_FUNCTION func = 0);
    PosixThread(PosixThreadAttributes* attr, THREAD_FUNCTION func = 0);
    ~PosixThread();

    TPL_NO_COPY(PosixThread);

public:  // Usage

    /**
     *  Set thread attributes.
     *  PRE_CONDITION: thread has not been started yet;
     *  PRE_CONDITION: attr has not been set yet;
     */
    void set_attributes(PosixThreadAttributes* attr);

    /**
     *  @return copy of thread attributes object.
     */
    PosixThreadAttributes* attributes();

    /**
     *  Set thread scheduling parameters.
     *  XXX: if thread has not been started yet sets thread attributes
     *  otherwise sets active thread scheduling parameters
     */
    void set_sched_params(const PosixThreadSchedParams& sched_params);

    /**
     *  Get thread scheduling parameters.
     *  XXX: if thread has not been started yet gets thread attributes
     *  otherwise returns active thread scheduling parameters
     */
    PosixThreadSchedParams sched_params() const;

    /**
     *  Register thread function.
     *  PRE_CONDITION: func != NULL;
     *  PRE_CONDITION: function has not been set yet;
     */
    void set_thread_function(THREAD_FUNCTION func);

    /**
     *  Start a thread with a context parameter.
     */
    int start(void *ctx);

    /**
     *  Detach thread.
     *  XXX: Since executable will fail on termination when not all threads
     *  are terminated custom notification for thread termination is required.
     *  PRE_CONDITION: cannot detach twice.
     */
    int detach();

    /**
     *  Wait for thread termination (blocking).
     *  PRE_CONDITION cannot join itself.
     */
    int join(void **ret_value);

    /**
     *  Cancel a thread.
     */
    int cancel();

    /**
     *  Send asynchronous signal to a thread.
     *  if signo == 0 signal is not sent.
     *
     *  1. Global signal handler for 'signo' is installed and
     *  signal is not masked in the target thread:
     *  Execution of the target thead will be interrupted
     *  and signal will be processed by the global signal handler
     *  wich will be executed in receiving thread. Thread will
     *  resume it's execution after signal handler is done.
     *
     *  2. Signal is masked in the target thread:
     *  Signal will never be delivered.
     *
     *  3. Global signal handler for 'signo' is not installed
     *  and signal is not masked in the target thread:
     *  Default behaviour will be applied for the whole process.
     *
     *  PRE_CONDITION thread has to be active (started not joined)
     */
    int kill(int signo);

/**
 *  Enable thread cancellation and set type.
 */
#define ENABLE_THREAD_CANCELLATION(type) \
    (PosixThreadCancellationPolicy::enable_cancellation(type))

/**
 *  Disable thread cancellation.
 */
#define DISABLE_THREAD_CANCELLATION \
    (PosixThreadCancellationPolicy::disable_cancellation())

/**
 *  Define cancellation point.
 */
#define THREAD_CANCEL_POINT \
    (::pthread_testcancel())

/**
 *  Terminate thread and return value
 */
#define THREAD_EXIT(retval) \
    (::pthread_exit(retval))

/**
 *  Yield processor to another thread.
 */
#define THREAD_YIELD \
    (::pthread_yield())

/**
 *  Returns thread id of current thread.
 */
#define THREAD_SELF \
    (::pthread_self())

/**
 *  Add signal set.
 */
#define THREAD_ADD_SIGMASK(sigset) \
    (PosixThreadSignalPolicy::add_signal_mask(sigset))

/**
 *  Remove signal set.
 */
#define THREAD_REMOVE_SIGMASK(sigset) \
    (PosixThreadSignalPolicy::remove_signal_mask(sigset))

/**
 *  Replace signal set.
 */
#define THREAD_REPLACE_SIGMASK(sigset) \
    (PosixThreadSignalPolicy::replace_signal_mask(sigset))

public:  // State

    /**
     *  @return thread id.
     */
    thread_id_t id() const { return _id; }

    /**
     *  @return current thread id.
     */
    static thread_id_t current_thread_id() { return ::pthread_self(); }

    /**
     *  @return true if thread has been started.
     */
    bool is_started() const { return _started; }

    /**
     *  @return true if thread has been joined.
     */
    bool is_joined() const { return _joined; }

    /**
     *  @return true if thread has been started and joined.
     */
    bool is_done() const { return (_started && _joined); }

    /**
     *	@return true if thread has been started and has not been joined yet.
     */
    bool is_active() const { return (_started && ! _joined); }

    /**
     *	@return true if thread has been started and has not been joined yet.
     */
    bool is_cancelled() const { return (_started && _joined && _cancelled); }

private:

    thread_id_t		   _id;		// Thread handle.
    PosixThreadAttributes* _attr;	// Thread attributes.
    THREAD_FUNCTION	   _func;	// Thread function.
    bool		   _started;	// TRUE when thread has been started.
    bool		   _joined;	// TRUE when thread has been joined.
    bool		   _cancelled;	// TRUE when thread has been cancelled.
};

};  // namespace tpl

#endif // __TPL_POSIX_THREAD_HH__
