// -*- c-basic-offset: 4; tab-width: 8; indent-tabs-mode: t -*-
//
// Copyright (c) 2007 Vyatta, Inc.
// All Rights Reserved.
//
// Author: Alex Allahverdiev <alex@vyatta.com>

#include <errno.h>
#include "posix_thread.hh"
#include "libtpl/debug.hh"

using namespace tpl;

////////////////////////////////////////////////////////////
//               PosixThreadAttributes                    //
////////////////////////////////////////////////////////////

PosixThreadAttributes::PosixThreadAttributes(int detached)
{
    TPL_VERIFY(::pthread_attr_init(&_attr) == 0);
    set_detached(detached);
}

PosixThreadAttributes::PosixThreadAttributes(const PosixThreadSchedParams& sched_params)
{
    TPL_VERIFY(::pthread_attr_init(&_attr) == 0);
    set_sched_params(sched_params);
}

PosixThreadAttributes::~PosixThreadAttributes()
{
    TPL_VERIFY(::pthread_attr_destroy(&_attr) == 0);
}

void
PosixThreadAttributes::set_detached(int detached)
{
    TPL_ASSERT(detached == PTHREAD_CREATE_DETACHED || detached == PTHREAD_CREATE_JOINABLE);
    TPL_VERIFY(::pthread_attr_setdetachstate(&_attr, detached) == 0);
}

int
PosixThreadAttributes::detached() const
{
    int detachstate;
    TPL_VERIFY(::pthread_attr_getdetachstate(&_attr, &detachstate) == 0);
    return detachstate;
}

void
PosixThreadAttributes::set_sched_params(const PosixThreadSchedParams& sched_params)
{
    TPL_ASSERT(sched_params.policy() == SHED_RR ||
    	    	sched_params.policy() == SHED_FIFO ||
    	    	sched_params.policy() == SHED_OTHER);
    TPL_VERIFY(::pthread_attr_setschedpolicy(&_attr, sched_params.policy()) == 0);
    struct sched_param sp = sched_params.param();
    TPL_VERIFY(::pthread_attr_setschedparam(&_attr, &sp) == 0);
}

PosixThreadSchedParams
PosixThreadAttributes::sched_params() const
{
    int policy;
    struct sched_param sp;
    TPL_VERIFY(::pthread_attr_getschedpolicy(&_attr, &policy) == 0);
    TPL_VERIFY(::pthread_attr_getschedparam(&_attr, &sp) == 0);
    return PosixThreadSchedParams(policy, sp);
}

void
PosixThreadAttributes::set_inherit(int inherit)
{
    TPL_ASSERT(inherit == PTHREAD_EXPLICIT_SCHED || inherit == PTHREAD_INHERIT_SCHED);
    TPL_VERIFY(::pthread_attr_setinheritsched(&_attr, inherit) == 0);
}

int
PosixThreadAttributes::inherit() const
{
    int inherit;
    TPL_VERIFY(::pthread_attr_getinheritsched(&_attr, &inherit) == 0);
    return inherit;
}

void
PosixThreadAttributes::set_scope(int scope)
{
    TPL_ASSERT(scope == PTHREAD_SCOPE_SYSTEM || scope == PTHREAD_SCOPE_PROCESS);
    TPL_VERIFY(::pthread_attr_setscope(&_attr, scope) == 0);
}

int
PosixThreadAttributes::scope() const
{
    int scope;
    TPL_VERIFY(::pthread_attr_getscope(&_attr, &scope) == 0);
    return scope;
}


////////////////////////////////////////////////////////////
//                PosixThreadCancellation                 //
////////////////////////////////////////////////////////////

/**
 *  Set cancellability options.
 *  @param type can be set to:
 *  PTHREAD_CANCEL_DEFERRED - defer cancelation to CANCEL_POINT.  
 *  PTHREAD_CANCEL_ASYNCHRONOUS - immidiate cancelation.
 *  @throw ErrorFatal - failed to set option.
 */
void
PosixThreadCancellationPolicy::enable_cancellation(int type)
{
    int prev_state;
    TPL_VERIFY(::pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, &prev_state) == 0);
    int prev_type;
    TPL_VERIFY(::pthread_setcanceltype(type, &prev_type) == 0);
}

/**
 *  Disable thread cancellation.
 */
void
PosixThreadCancellationPolicy::disable_cancellation()
{
    int prev_state;
    TPL_VERIFY(::pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, &prev_state) == 0);
}


////////////////////////////////////////////////////////////
//              PosixThreadSignalPolicy                   //
////////////////////////////////////////////////////////////

/*

PosixSignalSet
PosixThreadSignalPolicy::add_signal_mask(const PosixSignalSet& sigset)
{
    int err;
    sigset_t original_sigset;
    if ((err = ::pthread_sigmask(SIG_BLOCK, sigset, &original_sigset)) != 0)
	throw ErrorFatal(err);
    return PosixSignalSet(original_sigset);
}

PosixSignalSet
PosixThreadSignalPolicy::remove_signal_mask(const PosixSignalSet& sigset)
{
    int err;
    sigset_t original_sigset;
    if ((err = ::pthread_sigmask(SIG_UNBLOCK, sigset, &original_sigset)) != 0)
	throw ErrorFatal(err);
    return PosixSignalSet(original_sigset);
}

PosixSignalSet
PosixThreadSignalPolicy::replace_signal_mask(const PosixSignalSet& sigset)
{
    int err;
    sigset_t original_sigset;
    if ((err = ::pthread_sigmask(SIG_SETMASK, sigset, &original_sigset)) != 0)
	throw ErrorFatal(err);
    return PosixSignalSet(original_sigset);
}

*/

////////////////////////////////////////////////////////////
//                     PosixThread                        //
////////////////////////////////////////////////////////////

PosixThread::PosixThread(THREAD_FUNCTION func) :
    _id(0), _attr(0), _func(func), _started(false), _joined(false)
{
}

PosixThread::PosixThread(PosixThreadAttributes* attr, THREAD_FUNCTION func) :
    _id(0), _attr(attr), _func(func), _started(false), _joined(false)
{
}

PosixThread::~PosixThread()
{
}

void
PosixThread::set_attributes(PosixThreadAttributes* attr)
{
    TPL_ASSERT(!is_active());
    _attr = attr;
}

PosixThreadAttributes*
PosixThread::attributes()
{
    return _attr;
}

void
PosixThread::set_sched_params(const PosixThreadSchedParams& sched_params)
{
    if (is_active()) {
    	struct sched_param sp = sched_params.param();
    	TPL_VERIFY(::pthread_setschedparam(_id, sched_params.policy(), &sp) == 0);
    }
    else if (_attr) {
    	_attr->set_sched_params(sched_params);
    }
    TPL_ASSERT(_attr);
}

PosixThreadSchedParams
PosixThread::sched_params() const
{
    if (is_active()) {
        int policy;
        struct sched_param sp;
    	TPL_VERIFY(::pthread_getschedparam(_id, &policy, &sp) == 0);
    	return PosixThreadSchedParams(policy, sp);
    }
    else if (_attr)
    	return _attr->sched_params();

    TPL_ASSERT(_attr);
    return PosixThreadSchedParams(SCHED_OTHER);
}

void
PosixThread::set_thread_function(THREAD_FUNCTION func)
{
    TPL_ASSERT(func);
    TPL_ASSERT(!_func);
    _func = func;
}

int
PosixThread::start(void *ctx)
{
    TPL_ASSERT(!is_active());
    TPL_ASSERT(_func);

    int err = ::pthread_create(&_id, &(_attr->threadattr_handle()), _func, ctx);
    TPL_ASSERT(err == 0 && _id != 0);

    _started = true;
    _joined = false;
    return err;
}

int
PosixThread::detach()
{
    int err = ::pthread_detach(_id);
    TPL_ASSERT(err == 0);
    return err;
}

int
PosixThread::join(void **ret_value)
{
    TPL_ASSERT(pthread_equal(pthread_self(), _id) != 0);
    TPL_ASSERT(is_active());

    int err = ::pthread_join(_id, ret_value);
    TPL_ASSERT(err == 0 || err == ESRCH);

    _joined = true;
    _cancelled = (ret_value != 0 && *ret_value == PTHREAD_CANCELED);
    return err;
}

int
PosixThread::cancel()
{
    TPL_ASSERT(is_active());
    int err =::pthread_cancel(_id);
    TPL_ASSERT(err == 0);
    return err;
}

int
PosixThread::kill(int signo)
{
    TPL_ASSERT(is_active());
    int err = ::pthread_kill(_id, signo);
    TPL_ASSERT(err == 0);
    return err;
}

